% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/pls.R
\name{pls}
\alias{pls}
\title{Partial Least Squares regression}
\usage{
pls(x, y, ncomp = 15, center = T, scale = F, cv = NULL, x.test = NULL,
  y.test = NULL, method = "simpls", alpha = 0.05, coeffs.ci = NULL,
  coeffs.alpha = 0.1, info = "", light = F, ncomp.selcrit = "min")
}
\arguments{
\item{x}{matrix with predictors.}

\item{y}{matrix with responses.}

\item{ncomp}{maximum number of components to calculate.}

\item{center}{logical, center or not predictors and response values.}

\item{scale}{logical, scale (standardize) or not predictors and response values.}

\item{cv}{number of segments for cross-validation (if cv = 1, full cross-validation will be used).}

\item{x.test}{matrix with predictors for test set.}

\item{y.test}{matrix with responses for test set.}

\item{method}{method for calculating PLS model.}

\item{alpha}{significance level for calculating statistical limits for residuals.}

\item{coeffs.ci}{method to calculate p-values and confidence intervals for regression coefficients (so far only
jack-knifing is availavle: \code{='jk'}).}

\item{coeffs.alpha}{significance level for calculating confidence intervals for regression coefficients.}

\item{info}{short text with information about the model.}

\item{light}{run normal or light (faster) version of PLS without calculationg some performance statistics.}

\item{ncomp.selcrit}{criterion for selecting optimal number of components (\code{'min'} for
first local minimum of RMSECV and \code{'wold'} for Wold's rule.)}
}
\value{
Returns an object of \code{pls} class with following fields:
\item{ncomp }{number of components included to the model.}
\item{ncomp.selected }{selected (optimal) number of components.}
\item{xloadings }{matrix with loading values for x decomposition.}
\item{yloadings }{matrix with loading values for y decomposition.}
\item{weights }{matrix with PLS weights.}
\item{selratio }{array with selectivity ratio values.}
\item{vipscores }{matrix with VIP scores values.}
\item{coeffs }{object of class \code{\link{regcoeffs}} with regression coefficients calculated for each component.}
\item{info }{information about the model, provided by user when build the model.}
\item{calres }{an object of class \code{\link{plsres}} with PLS results for a calibration data.}
\item{testres }{an object of class \code{\link{plsres}} with PLS results for a test data, if it was provided.}
\item{cvres }{an object of class \code{\link{plsres}} with PLS results for cross-validation, if this option was chosen.}
}
\description{
\code{pls} is used to calibrate, validate and use of partial least squares (PLS)
regression model.
}
\details{
So far only SIMPLS method [1] is available, more coming soon. Implementation works both with one
and multiple response variables.

Like in \code{\link{pca}}, \code{pls} uses number of components (\code{ncomp}) as a minimum of
number of objects - 1, number of x variables and the default or provided value. Regression
coefficients, predictions and other results are calculated for each set of components from 1
to \code{ncomp}: 1, 1:2, 1:3, etc. The optimal number of components, (\code{ncomp.selected}),
is found using Wold's R criterion, but can be adjusted by user using function
(\code{\link{selectCompNum.pls}}). The selected optimal number of components is used for all
default operations - predictions, plots, etc.

Selectivity ratio [2] and VIP scores [3] are calculated for any PLS model authomatically, however
while selectivity ratio values are calculated for all computed components, the VIP scores are
computed only for selected components (to save calculation time) and recalculated every time when
\code{selectCompNum()} is called for the model.

Calculation of confidence intervals and p-values for regression coefficients are available
only by jack-knifing so far. See help for \code{\link{regcoeffs}} objects for details.
}
\examples{
### Examples of using PLS model class
library(mdatools)

## 1. Make a PLS model for concentration of first component
## using full-cross validation and automatic detection of
## optimal number of components and show an overview

data(simdata)
x = simdata$spectra.c
y = simdata$conc.c[, 1]

model = pls(x, y, ncomp = 8, cv = 1)
summary(model)
plot(model)

## 2. Make a PLS model for concentration of first component
## using test set and 10 segment cross-validation and show overview

data(simdata)
x = simdata$spectra.c
y = simdata$conc.c[, 1]
x.t = simdata$spectra.t
y.t = simdata$conc.t[, 1]

model = pls(x, y, ncomp = 8, cv = 10, x.test = x.t, y.test = y.t)
model = selectCompNum(model, 2)
summary(model)
plot(model)

## 3. Make a PLS model for concentration of first component
## using only test set validation and show overview

data(simdata)
x = simdata$spectra.c
y = simdata$conc.c[, 1]
x.t = simdata$spectra.t
y.t = simdata$conc.t[, 1]

model = pls(x, y, ncomp = 6, x.test = x.t, y.test = y.t)
model = selectCompNum(model, 2)
summary(model)
plot(model)

## 4. Show variance and error plots for a PLS model
par(mfrow = c(2, 2))
plotXCumVariance(model, type = 'h')
plotYCumVariance(model, type = 'b', show.labels = TRUE, legend.position = 'bottomright')
plotRMSE(model)
plotRMSE(model, type = 'h', show.labels = TRUE)
par(mfrow = c(1, 1))

## 5. Show scores plots for a PLS model
par(mfrow = c(2, 2))
plotXScores(model)
plotXScores(model, comp = c(1, 3), show.labels = TRUE)
plotXYScores(model)
plotXYScores(model, comp = 2, show.labels = TRUE)
par(mfrow = c(1, 1))

## 6. Show loadings and coefficients plots for a PLS model
par(mfrow = c(2, 2))
plotXLoadings(model)
plotXLoadings(model, comp = c(1, 2), type = 'l')
plotXYLoadings(model, comp = c(1, 2), legend.position = 'topleft')
plotRegcoeffs(model)
par(mfrow = c(1, 1))

## 7. Show predictions and residuals plots for a PLS model
par(mfrow = c(2, 2))
plotXResiduals(model, show.label = TRUE)
plotYResiduals(model, show.label = TRUE)
plotPredictions(model)
plotPredictions(model, ncomp = 4, xlab = 'C, reference', ylab = 'C, predictions')
par(mfrow = c(1, 1))

## 8. Selectivity ratio and VIP scores plots
par(mfrow = c(2, 2))
plotSelectivityRatio(model)
plotSelectivityRatio(model, ncomp = 1)
par(mfrow = c(1, 1))

## 9. Variable selection with selectivity ratio
selratio = getSelectivityRatio(model)
selvar = !(selratio < 8)

xsel = x[, selvar]
modelsel = pls(xsel, y, ncomp = 6, cv = 1)
modelsel = selectCompNum(modelsel, 3)

summary(model)
summary(modelsel)

## 10. Calculate average spectrum and show the selected variables
i = 1:ncol(x)
ms = apply(x, 2, mean)

par(mfrow = c(2, 2))

plot(i, ms, type = 'p', pch = 16, col = 'red', main = 'Original variables')
plotPredictions(model)

plot(i, ms, type = 'p', pch = 16, col = 'lightgray', main = 'Selected variables')
points(i[selvar], ms[selvar], col = 'red', pch = 16)
plotPredictions(modelsel)

par(mfrow = c(1, 1))
}
\author{
Sergey Kucheryavskiy (svkucheryavski@gmail.com)
}
\references{
1. S. de Jong, Chemometrics and Intelligent Laboratory Systems 18 (1993) 251-263.
2. Tarja Rajalahti et al. Chemometrics and Laboratory Systems, 95 (2009), 35-48.
3. Il-Gyo Chong, Chi-Hyuck Jun. Chemometrics and Laboratory Systems, 78 (2005), 103-112.
}
\seealso{
Methods for \code{pls} objects:
\tabular{ll}{
 \code{print} \tab prints information about a \code{pls} object.\cr
 \code{\link{summary.pls}} \tab shows performance statistics for the model.\cr
 \code{\link{plot.pls}} \tab shows plot overview of the model.\cr
 \code{\link{pls.simpls}} \tab implementation of SIMPLS algorithm.\cr
 \code{\link{predict.pls}} \tab applies PLS model to a new data.\cr
 \code{\link{selectCompNum.pls}} \tab set number of optimal components in the model.\cr
 \code{\link{plotPredictions.pls}} \tab shows predicted vs. measured plot.\cr
 \code{\link{plotRegcoeffs.pls}} \tab shows regression coefficients plot.\cr
 \code{\link{plotXScores.pls}} \tab shows scores plot for x decomposition.\cr
 \code{\link{plotXYScores.pls}} \tab shows scores plot for x and y decomposition.\cr
 \code{\link{plotXLoadings.pls}} \tab shows loadings plot for x decomposition.\cr
 \code{\link{plotXYLoadings.pls}} \tab shows loadings plot for x and y decomposition.\cr
 \code{\link{plotRMSE.pls}} \tab shows RMSE plot.\cr
 \code{\link{plotXVariance.pls}} \tab shows explained variance plot for x decomposition.\cr
 \code{\link{plotYVariance.pls}} \tab shows explained variance plot for y decomposition.\cr
 \code{\link{plotXCumVariance.pls}} \tab shows cumulative explained variance plot for y
 decomposition.\cr
 \code{\link{plotYCumVariance.pls}} \tab shows cumulative explained variance plot for y
 decomposition.\cr
 \code{\link{plotXResiduals.pls}} \tab shows T2 vs. Q plot for x decomposition.\cr
 \code{\link{plotYResiduals.pls}} \tab shows residuals plot for y values.\cr
 \code{\link{plotSelectivityRatio.pls}} \tab shows plot with selectivity ratio values.\cr
 \code{\link{plotVIPScores.pls}} \tab shows plot with VIP scores values.\cr
 \code{\link{getSelectivityRatio.pls}} \tab returns vector with selectivity ratio values.\cr
 \code{\link{getVIPScores.pls}} \tab returns vector with VIP scores values.\cr
 \code{\link{getRegcoeffs.pls}} \tab returns matrix with regression coefficients.\cr
}

Most of the methods for plotting data (except loadings and regression coefficients) are also
available for PLS results
(\code{\link{plsres}}) objects. There is also a randomization test for PLS-regression
(\code{\link{randtest}}).
}

