% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stabit2.R
\name{stabit2}
\alias{stabit2}
\alias{stabitCpp2}
\title{Structural Matching Model to correct for sample selection bias in two-sided matching markets}
\usage{
stabit2(OUT, SEL = NULL, colleges = NULL, students = NULL,
  m.id = "m.id", c.id = "c.id", s.id = "s.id", outcome,
  selection = NULL, selection.student = NULL, selection.college = NULL,
  binary = FALSE, niter, gPrior = FALSE, censored = 1, seed = 123)
}
\arguments{
\item{OUT}{data frame with characteristics of all observed matches, including
market identifier \code{m.id}, college identifier \code{c.id} and student identifier \code{s.id}.}

\item{SEL}{optional: data frame with characteristics of all observed and unobserved matches, including 
market identifier \code{m.id}, college identifier \code{c.id} and student identifier \code{s.id}.}

\item{colleges}{character vector of variable names for college characteristics. These variables carry the same value for any college.}

\item{students}{character vector of variable names for student characteristics. These variables carry the same value for any student.}

\item{m.id}{character name of the market identifier variable. Defaults to \code{"m.id"}.}

\item{c.id}{character name of the college identifier variable. Defaults to \code{"c.id"}.}

\item{s.id}{character name of the student identifier variable. Defaults to \code{"s.id"}.}

\item{outcome}{formula for match outcomes.}

\item{selection}{formula for match valuations.}

\item{selection.student}{formula for match valuations of students. This argument is ignored when \code{selection} is provided.}

\item{selection.college}{formula for match valuations of colleges. This argument is ignored when \code{selection} is provided.}

\item{binary}{logical: if \code{TRUE} outcome variable is taken to be binary; if \code{FALSE} outcome variable is taken to be continuous.}

\item{niter}{number of iterations to use for the Gibbs sampler.}

\item{gPrior}{logical: if \code{TRUE} the g-prior (Zellner, 1986) is used for the variance-covariance matrix. (Not yet implemented)}

\item{censored}{draws of the \code{delta} parameter that estimates the covariation between the error terms in selection and outcome equation are 0:not censored, 1:censored from below, 2:censored from above.}

\item{seed}{integer setting the state for MCMC draws.}
}
\value{
\code{stabit} returns a list with the following items.
\item{model.list}{}
\item{draws}{}
\item{coefs}{}
}
\description{
The function provides a Gibbs sampler for a structural matching model that corrects 
for sample selection bias when the selection process is a two-sided matching game; i.e., 
a matching of students to colleges.

The structural model consists of a selection and an outcome equation. The \emph{Selection Equation} 
determines which matches are observed (\eqn{D=1}) and which are not (\eqn{D=0}).
\deqn{ \begin{array}{lcl}
       D &= & 1[V \in \Gamma] \\
       V &= & W\alpha + \eta
       \end{array}
     }{ D = 1[V in \Gamma] with V = W\alpha + \eta
     }
Here, \eqn{V} is a vector of latent valuations of \emph{all feasible} matches, ie observed and 
unobserved, and \eqn{1[.]} is the Iverson bracket. 
A match is observed if its match valuation is in the set of valuations \eqn{\Gamma}
that satisfy the equilibrium condition (see Sorensen, 2007). 
The match valuation \eqn{V} is a linear function of \eqn{W}, a matrix of characteristics for 
\emph{all feasible} groups, and \eqn{\eta}, a vector of random errors. \eqn{\alpha} is a paramter 
vector to be estimated.

The \emph{Outcome Equation} determines the outcome for \emph{observed} matches. The dependent
variable can either be continuous or binary, dependent on the value of the \code{binary}
argument. In the binary case, the dependent variable \eqn{R} is determined by a threshold 
rule for the latent variable \eqn{Y}.
\deqn{ \begin{array}{lcl}
       R &= & 1[Y > c] \\
       Y &= & X\beta + \epsilon
       \end{array}
     }{ R = 1[Y > c] with Y = X\beta + \epsilon
     }
Here, \eqn{Y} is a linear function of \eqn{X}, a matrix of characteristics for \emph{observed} 
matches, and \eqn{\epsilon}, a vector of random errors. \eqn{\beta} is a paramter vector to 
be estimated.

The structural model imposes a linear relationship between the error terms of both equations 
as \eqn{\epsilon = \delta\eta + \xi}, where \eqn{\xi} is a vector of random errors and \eqn{\delta}
is the covariance paramter to be estimated. If \eqn{\delta} were zero, the marginal distributions
of \eqn{\epsilon} and \eqn{\eta} would be independent and the selection problem would vanish.
That is, the observed outcomes would be a random sample from the population of interest.
}
\examples{
## --- SIMULATED EXAMPLE ---
\dontrun{
## 1. Simulate two-sided matching data for 20 markets (m=20) with 100 students
##    (nStudents=100) per market and 20 colleges with quotas of 5 students, each
##    (nSlots=rep(5,20)).

xdata <- stabsim2(m=20, nStudents=100, nSlots=rep(5,20), 
  colleges = "c1",
  students = "s1",
  outcome = ~ c1:s1 + eta +nu,
  selection = ~ -1 + c1:s1 + eta
)
head(xdata$OUT)


## 2-a. Bias from sorting
 lm1 <- lm(y ~ c1:s1, data=xdata$OUT)
 summary(lm1)

## 2-b. Cause of the bias
 with(xdata$OUT, cor(c1*s1, eta))

## 2-c. Correction for sorting bias
 lm2a <- lm(V ~ -1 + c1:s1, data=xdata$SEL); summary(lm2a)
 etahat <- lm2a$residuals[xdata$SEL$D==1]
 
 lm2b <- lm(y ~ c1:s1 + etahat, data=xdata$OUT)
 summary(lm2b)


## 3. Correction for sorting bias when match valuation V is unobserved

## 3-a. Run Gibbs sampler (when SEL is given)
 fit2 <- stabit2(OUT = xdata$OUT, 
           SEL = xdata$SEL,
           outcome = y ~ c1:s1, 
           selection = ~ -1 + c1:s1,
           niter=1000
 )

## 3-b. Run Gibbs sampler (when SEL is not given)
 fit2 <- stabit2(OUT = xdata$OUT, 
           colleges = "c1",
           students = "s1",
           outcome = y ~ c1:s1, 
           selection = ~ -1 + c1:s1,
           niter=1000
 )

## 4-a. Get marginal effects (for linear model)
 fit2$coefs
 
## 4-b. Get marginal effects (for probit)
 mfx(fit2)
 
 
## 5. Plot MCMC draws for coefficients
 plot(fit2$draws$alphadraws[1,], type="l")
 plot(fit2$draws$betadraws[1,], type="l")
 plot(fit2$draws$deltadraws[1,], type="l")
 
 
## 6. Obtain the model list used in estimation
 head(fit2$model.list)
}
}
\author{
Thilo Klein
}
\references{
Sorensen, M. (2007). How Smart is Smart Money? A Two-Sided Matching Model of Venture Capital.
\emph{Journal of Finance}, 62 (6): 2725-2762.
}
\keyword{regression}

