% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/R6.R
\name{ImapCon}
\alias{ImapCon}
\title{An IMAP Connection Class}
\description{
Configure an IMAP connection using the \code{ImapCon} \code{R6}
  class.
}
\note{
\href{#method-new}{\code{ImapCon$new()}}: The \code{\link{configure_imap}}
  should be preferred instead of \code{ImapCon$new()}.

\href{#method-search}{\code{ImapCon$search()}}: IMAP queries follows
  Polish notation, i.e. operators such as \code{OR} come before arguments,
  e.g. "OR argument1 argument2". Therefore, the relational-operator-helper-functions
  in this package should be used like the following examples:
  \code{OR(before("17-Apr-2015"), string("FROM", "John"))}. Even though there
  is no "AND" operator in IMAP, this package adds a helper function
  \code{\link{AND}} to indicate multiples arguments that must be searched
  together, e.g. \code{AND(since("01-Jul-2018"), smaller_than(16000))}.

\href{#method-sent_before}{\code{ImapCon$sent_before()}}: Search
  operations that use the origination/RFC-2822 Header date
  tend to be "slower" than those that use the internal date. Although the
  overhead is minimum, the difference is due to the fact that the internal date
  is kept on a database, while the origination date has to be retrieved from
  inside the message. Therefore, the server needs to access each message when
  executing this type of search. Despite this fact, both dates tend to be the
  same.

\href{#method-search_sent_since}{\code{ImapCon$search_sent_since()}}: Search
  operations that use the origination/RFC-2822 Header date
  tend to be "slower" than those that use the internal date. Although the
  overhead is minimum, the difference is due to the fact that the internal date
  is kept on a database, while the origination date has to be retrieved from
  inside the message. Therefore, the server needs to access each message when
  executing this type of search. Despite this fact, both dates tend to be the
  same.

\href{#method-search_sent_on}{\code{ImapCon$search_sent_on()}}: Search
  operations that use the origination/RFC-2822 Header date
  tend to be "slower" than those that use the internal date. Although the
  overhead is minimum, the difference is due to the fact that the internal date
  is kept on a database, while the origination date has to be retrieved from
  inside the message. Therefore, the server needs to access each message when
  executing this type of search. Despite this fact, both dates tend to be the
  same.

\href{#method-search_sent_period}{\code{ImapCon$search_sent_period()}}: Search
  operations that use the origination/RFC-2822 Header date
  tend to be "slower" than those that use the internal date. Although the
  overhead is minimum, the difference is due to the fact that the internal date
  is kept on a database, while the origination date has to be retrieved from
  inside the message. Therefore, the server needs to access each message when
  executing this type of search. Despite this fact, both dates tend to be the
  same.

\href{#method-search_older_than}{\code{ImapCon$search_older_than()}}:
  To be able to use this functionality, the server must support the
  \code{WITHIN} capability. You can check it by running
  \href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.

\href{#method-search_older_than}{\code{ImapCon$search_older_than()}}:
  To be able to use this functionality, the server must support the
  \code{WITHIN} capability. You can check it by running
  \href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.

\href{#method-search_string}{\code{ImapCon$search_string()}}: Using
  \code{where = "TEXT"}, may produce unexpected results since it
  will perform the search on raw data, i.e. the searched expression may be
  truncated by special formating characters such as \code{\\r\\n} for example.
  It is recommended to perform this type of search using \code{where = "BODY"},
  instead of \code{"TEXT"} (\cite{Heinlein, P. and Hartleben, P. (2008)}).

\href{#method-esearch_count}{\code{ImapCon$esearch_count()}}: This
  operation depends on the \code{ESEARCH} extension.

\href{#method-esearch_min_id}{\code{ImapCon$esearch_min_id()}}: This
  operation depends on the \code{ESEARCH} extension.

\href{#method-esearch_max_id}{\code{ImapCon$esearch_max_id()}}: This
  operation depends on the \code{ESEARCH} extension.

\href{#method-add_flags}{\code{ImapCon$add_flags()}}: Unlike the
  search operations, the add/replace/delete flags operations
  demand system flag names to be preceded by two backslashes \code{"\\\\"}.

\href{#method-add_flags}{\code{ImapCon$add_flags()}}: \code{add_flags},
  \code{remove_flags}, and \code{replace_flags} accept not only flags but
  also keywords (any word not beginning with two backslashes) which are
  custom flags defined by the user.

\href{#method-replace_flags}{\code{ImapCon$replace_flags()}}: Unlike the
  search operations, the add/replace/delete flags operations
  demand system flag names to be preceded by two backslashes \code{"\\\\"}.

\href{#method-replace_flags}{\code{ImapCon$replace_flags()}}: \code{add_flags},
  \code{remove_flags}, and \code{replace_flags} accept not only flags but
  also keywords (any word not beginning with two backslashes) which are
  custom flags defined by the user.

\href{#method-remove_flags}{\code{ImapCon$remove_flags()}}: Unlike the
  search operations, the add/replace/delete flags operations
  demand system flag names to be preceded by two backslashes \code{"\\\\"}.

\href{#method-remove_flags}{\code{ImapCon$remove_flags()}}: \code{add_flags},
  \code{remove_flags}, and \code{replace_flags} accept not only flags but
  also keywords (any word not beginning with two backslashes) which are
  custom flags defined by the user.

\href{#method-get_attachments}{\code{ImapCon$get_attachments()}}:
  This method is to be used after the body or the
  text part of one or more messages were fetched. This makes sense if the
  user is interested in keeping the message content (body or text) besides
  downloading the message attachments. Nonetheless, this is not the
  recommended approach if the user is only interested in downloading the files
  as the previous fetching operation will probably be costly. In this last
  case, the recommendation is to use
  \href{#method-fetch_attachments}{\code{ImapCon$fetch_attachments()}} as
  it will only fetch the attachment part.

\href{#method-get_attachments}{\code{ImapCon$get_attachments()}}: All
  attachments will be stored in a folder labeled with the message id
  inside the \code{working directory > servername > foldername}.
  This function currently handles only attachments
  encoded as \code{base64} text. It tries to guess all file extensions while
  decoding the text, but it may not be possible to do so in some circumstances.
  If it happens, you can try to change the file extension directly by renaming
  the file.

\href{#method-get_attachments}{\code{ImapCon$get_attachments()}}: The
  "Content-Disposition" header specifies if the multipart electronic
  messages will be presented as a main document with a list of separate
  attachments ("Content-Disposition: attachment") or as a single document
  with the various parts displayed inline. The first requires positive action
  on the part of the recipient (downloading the file, for example) whereas inline
  components are displayed automatically when the message is viewed
  (\cite{Troost, R., Dorner, S., and K. Moore, Ed. (1997)}). You can choose
  to download \code{both}, or only one type of attachment, using the
  argument \code{content_disposition}.

\href{#method-fetch_attachments}{\code{ImapCon$fetch_attachments()}}: All
  attachments will be stored in a folder labeled with the message id
  inside the \code{working directory > servername > foldername}.
  This function currently handles only attachments
  encoded as \code{base64} text. It tries to guess all file extensions while
  decoding the text, but it may not be possible to do so in some circumstances.
  If it happens, you can try to change the file extension directly by renaming
  the file.

\href{#method-fetch_attachments}{\code{ImapCon$fetch_attachments()}}: The
  "Content-Disposition" header specifies if the multipart electronic
  messages will be presented as a main document with a list of separate
  attachments ("Content-Disposition: attachment") or as a single document
  with the various parts displayed inline. The first requires positive action
  on the part of the recipient (downloading the file, for example) whereas inline
  components are displayed automatically when the message is viewed
  (\cite{Troost, R., Dorner, S., and K. Moore, Ed. (1997)}). You can choose
  to download \code{both}, or only one type of attachment, using the
  argument \code{content_disposition}.
}
\examples{
\dontrun{
# w/ Plain authentication
con <- configure_imap(
  url="imaps://outlook.office365.com",
  username="user@agency.gov.br",
  password=rstudioapi::askForPassword(),
  verbose = TRUE)

# OR
con <- ImapCon$new(
  url="imaps://outlook.office365.com",
  username="user@agency.gov.br",
  password=rstudioapi::askForPassword(),
  verbose = TRUE)

# w/ OAuth2.0 authentication
con <- configure_imap(
  url="imaps://outlook.office365.com",
  username="user@agency.gov.br",
  verbose = TRUE,
  xoauth2_bearer = "XX.Ya9...")

# OR
con <- ImapCon$new(
  url="imaps://outlook.office365.com",
  username="user@agency.gov.br",
  verbose = TRUE,
  xoauth2_bearer = "XX.Ya9...")

}



## ------------------------------------------------
## Method `ImapCon$list_server_capabilities`
## ------------------------------------------------

\dontrun{
cap <- con$list_server_capabilities()
cap
}

## ------------------------------------------------
## Method `ImapCon$list_mail_folders`
## ------------------------------------------------

\dontrun{
folders <- con$list_mail_folders()
folders
}

## ------------------------------------------------
## Method `ImapCon$select_folder`
## ------------------------------------------------

\dontrun{
con$select_mail_folder(name = "INBOX")
}

## ------------------------------------------------
## Method `ImapCon$examine_folder`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
con$examine_folder()

# or directly:
con$examine_folder("Sent")
}

## ------------------------------------------------
## Method `ImapCon$create_folder`
## ------------------------------------------------

\dontrun{
con$create_folder(name = "New Folder Name")
}

## ------------------------------------------------
## Method `ImapCon$rename_folder`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "Folder A")
con$rename_folder(new_name = "Folder B")
# or directly:
con$rename_folder(name = "Folder A", new_name = "Folder B")
}

## ------------------------------------------------
## Method `ImapCon$list_flags`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
con$list_flags()
}

## ------------------------------------------------
## Method `ImapCon$search`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# ex1
con$search(OR(before(date_char = "17-Apr-2015"),
              string(expr = "John", where = "FROM")))

# ex2
con$search(AND(smaller_than(size = "512000"),
               string(expr = "John", where = "FROM"),
               string(expr = "@ksu.edu", where = "CC")))
}

## ------------------------------------------------
## Method `ImapCon$search_larger_than`
## ------------------------------------------------

\dontrun{
# search for messages with size larger than 512Kb
con$search_larger_than(size = 512000))
}

## ------------------------------------------------
## Method `ImapCon$search_smaller_than`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# search for messages with size smaller than 512Kb
con$search_smaller_than(size = 512000))
}

## ------------------------------------------------
## Method `ImapCon$search_before`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# search for messages with date before "02-Jan-2020", presenting the
# .. results as unique identifiers (UID)
con$search_before(date = "02-Jan-2020", use_uid = TRUE)
}

## ------------------------------------------------
## Method `ImapCon$search_since`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# search for messages with date since "02-Jan-2020", presenting the
# .. results as unique identifiers (UID)
con$search_since(date = "02-Jan-2020", use_uid = TRUE)
}

## ------------------------------------------------
## Method `ImapCon$search_on`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# search for messages received on date "02-Jan-2020", presenting the
#... results as unique identifiers (UID)
con$search_on(date = "02-Jan-2020", use_uid = TRUE)
}

## ------------------------------------------------
## Method `ImapCon$search_period`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# search for all messages in the mail folder, EXCEPT (negate = TRUE) by
#... those received between the dates "02-Jan-2020" and "22-Mar-2020"
con$search_period(since_date_char = "02-Jan-2020",
                  before_date_char = "22-Mar-2020",
                  negate = TRUE))
}

## ------------------------------------------------
## Method `ImapCon$search_sent_before`
## ------------------------------------------------

\dontrun{
# search for messages with date before "02-Jan-2020", presenting the
# .. results as unique identifiers (UID)
con$search_sent_before(date = "02-Jan-2020", use_uid = TRUE)
}

## ------------------------------------------------
## Method `ImapCon$search_sent_since`
## ------------------------------------------------

\dontrun{
# search for messages with date before "02-Jan-2020", presenting the
# .. results as unique identifiers (UID)
con$search_sent_since(date = "02-Jan-2020", use_uid = TRUE)
}

## ------------------------------------------------
## Method `ImapCon$search_sent_on`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# search for messages received on date "02-Jan-2020", presenting the
#... results as unique identifiers (UID)
con$search_sent_on(date = "02-Jan-2020", use_uid = TRUE)
}

## ------------------------------------------------
## Method `ImapCon$search_sent_period`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# search for all messages in the mail folder, EXCEPT (negate = TRUE) by
#... those received between the dates "02-Jan-2020" and "22-Mar-2020"
con$search_sent_period(since_date_char = "02-Jan-2020",
                  before_date_char = "22-Mar-2020",
                  negate = TRUE))
}

## ------------------------------------------------
## Method `ImapCon$search_flag`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# search for all messages in the mail folder that are marked as "SEEN" AND
#.. "ANSWERED"
con$search_flag(name = c("SEEN", "ANSWERED"))
}

## ------------------------------------------------
## Method `ImapCon$search_older_than`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# search for all messages received in the last hour (not older than 3600 seconds)
con$search_older_than(seconds = 3600, negate = TRUE)
}

## ------------------------------------------------
## Method `ImapCon$search_younger_than`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# search for all messages received in the last hour (younger than 3600 seconds)
con$search_younger_than(seconds = 3600)
}

## ------------------------------------------------
## Method `ImapCon$search_string`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# search for all messages received in the last hour (younger than 3600 seconds)
con$search_string(expr = "@k-state.edu", where = "FROM")
}

## ------------------------------------------------
## Method `ImapCon$fetch_body`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# do a search and fetch the results (saving to disk) using the pipe
con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$fetch_body(write_to_disk = TRUE, keep_in_mem = FALSE)

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")

con$fetch_body(msg = res, write_to_disk = TRUE, keep_in_mem = FALSE)

}

## ------------------------------------------------
## Method `ImapCon$fetch_header`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# do a search and fetch the results (also saving to disk) using the pipe
out <- con$search_string(expr = "@k-state.edu", where = "CC") \%>\%
  con$fetch_header()

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "CC")
out <- con$fetch_header()

}

## ------------------------------------------------
## Method `ImapCon$fetch_metadata`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# do a search and fetch the results using the pipe
out <- con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$fetch_metadata()

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")
out <- con$fetch_metadata(msg = res)

}

## ------------------------------------------------
## Method `ImapCon$fetch_text`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# do a search and partially fetch the results using the pipe
# first 200 characters, writing to disk, silence results in the console
con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$fetch_text(partial = "0.200",
                 write_to_disk = TRUE,
                 keep_in_mem = FALSE)

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")
con$fetch_text(msg = res,
               partial = "0.200",
               write_to_disk = TRUE,
               keep_in_mem = FALSE)

}

## ------------------------------------------------
## Method `ImapCon$copy_msg`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# do a search and copy the results to another folder
con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$copy(to_folder = "Sent")

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")
con$copy(msg = res, to_folder = "Sent")

}

## ------------------------------------------------
## Method `ImapCon$move_msg`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# do a search and copy the results to another folder
con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$move(to_folder = "Sent")

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")
con$move(msg = res, to_folder = "Sent")

}

## ------------------------------------------------
## Method `ImapCon$esearch_count`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# count the number of messages marked as "Flagged" and "Answered"
con$esearch_count(flag = c("Flagged", "Answered"))
}

## ------------------------------------------------
## Method `ImapCon$delete_msg`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# delete
con$delete_msg(flag = c("Flagged", "Answered"))
}

## ------------------------------------------------
## Method `ImapCon$expunge`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# count the number of messages marked as "Flagged" and "Answered"
con$esearch_count(flag = c("Flagged", "Answered"))
}

## ------------------------------------------------
## Method `ImapCon$esearch_min_id`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# Search the minimum id of messages marked as "Answered"
con$esearch_min_id(flag = "Answered")
}

## ------------------------------------------------
## Method `ImapCon$esearch_max_id`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# Search the minimum id of messages marked as "Seen"
con$esearch_max_id(flag = "Seen")
}

## ------------------------------------------------
## Method `ImapCon$add_flags`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# Add the "\\Seen" permanent flag to the messages received in the last hour
con$search_younger_than(seconds = 3600) \%>\% # depends on the WITHIN extension
  con$add_flags(flags_to_set = "\\\\Seen")
}

## ------------------------------------------------
## Method `ImapCon$replace_flags`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# Replace the current flags of the messages in the search results for the
#.. flags "\\UNSEEN" and "\\Flagged"
con$search_since(date_char = "20-Aug-2020") \%>\%
  con$replace_flags(flags_to_set = c("\\\\UNSEEN", "\\\\Flagged")
}

## ------------------------------------------------
## Method `ImapCon$remove_flags`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# Remove the the "\\SEEN" flag from the messages in the search result
con$search_since(date_char = "20-Aug-2020") \%>\%
  con$remove_flags(flags_to_unset = "\\\\UNSEEN")
}

## ------------------------------------------------
## Method `ImapCon$get_attachments`
## ------------------------------------------------

\dontrun{
# example 1
con$select_folder(name = "INBOX")
con$search_string(expr = "@gmail", where = "CC") \%>\%
  con$fetch_text(write_to_disk = TRUE) \%>\% # saving the message's content as txt files
  con$get_attachments()

# example 2
res <- con$search_string(expr = "@gmail", where = "CC") \%>\%
out <- con$fetch_body(msg = res)
con$get_attachments(msg_list = out)
}

## ------------------------------------------------
## Method `ImapCon$fetch_attachments_list`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# do a search and fetch the attachments' list of the messages
out < con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$fetch_attachments_list()
out

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")
out <- con$fetch_attachments_list(msg = res)
out

}

## ------------------------------------------------
## Method `ImapCon$fetch_attachments`
## ------------------------------------------------

\dontrun{
con$select_folder(name = "INBOX")
# do a search and fetch the attachments' list of the messages
con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$fetch_attachments() # the attachments will be downloaded to disk


# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")
con$fetch_attachments(msg = res)

}
}
\references{
\href{#method-search_string}{\code{ImapCon$search_string()}}:
  Heinlein, P. and Hartleben, P. (2008). The Book of IMAP: Building a
  Mail Server with Courier and Cyrus. No Starch Press. ISBN 978-1-59327-177-0.

\href{#method-get_attachments}{\code{ImapCon$get_attachments()}}:
  Troost, R., Dorner, S., and K. Moore (1997), Communicating
  Presentation Information in Internet Messages: The Content-Disposition
  Header Field, RFC 2183, August 1997, https://www.rfc-editor.org/rfc/rfc2183.

\href{#method-fetch_attachments}{\code{ImapCon$fetch_attachments()}}:
  Troost, R., Dorner, S., and K. Moore (1997), Communicating
  Presentation Information in Internet Messages: The Content-Disposition
  Header Field, RFC 2183, DOI 10.17487/RFC2183, August 1997,
  https://www.rfc-editor.org/rfc/rfc2183.
}
\seealso{
Other custom search: 
\code{\link{AND}()},
\code{\link{OR}()},
\code{\link{before}()},
\code{\link{flag}()},
\code{\link{larger_than}()},
\code{\link{older_than}()},
\code{\link{on}()},
\code{\link{sent_before}()},
\code{\link{sent_on}()},
\code{\link{sent_since}()},
\code{\link{since}()},
\code{\link{smaller_than}()},
\code{\link{string}()},
\code{\link{younger_than}()}

Other attachments: 
\code{\link{list_attachments}()}
}
\concept{attachments}
\concept{complementary operations}
\concept{custom search}
\concept{fetch}
\concept{search by date}
\concept{search by flag}
\concept{search by size}
\concept{search within}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-ImapCon-new}{\code{ImapCon$new()}}
\item \href{#method-ImapCon-reset_url}{\code{ImapCon$reset_url()}}
\item \href{#method-ImapCon-reset_username}{\code{ImapCon$reset_username()}}
\item \href{#method-ImapCon-reset_use_ssl}{\code{ImapCon$reset_use_ssl()}}
\item \href{#method-ImapCon-reset_verbose}{\code{ImapCon$reset_verbose()}}
\item \href{#method-ImapCon-reset_buffersize}{\code{ImapCon$reset_buffersize()}}
\item \href{#method-ImapCon-reset_timeout_ms}{\code{ImapCon$reset_timeout_ms()}}
\item \href{#method-ImapCon-reset_password}{\code{ImapCon$reset_password()}}
\item \href{#method-ImapCon-reset_xoauth2_bearer}{\code{ImapCon$reset_xoauth2_bearer()}}
\item \href{#method-ImapCon-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}
\item \href{#method-ImapCon-list_mail_folders}{\code{ImapCon$list_mail_folders()}}
\item \href{#method-ImapCon-select_folder}{\code{ImapCon$select_folder()}}
\item \href{#method-ImapCon-examine_folder}{\code{ImapCon$examine_folder()}}
\item \href{#method-ImapCon-create_folder}{\code{ImapCon$create_folder()}}
\item \href{#method-ImapCon-rename_folder}{\code{ImapCon$rename_folder()}}
\item \href{#method-ImapCon-list_flags}{\code{ImapCon$list_flags()}}
\item \href{#method-ImapCon-search}{\code{ImapCon$search()}}
\item \href{#method-ImapCon-search_larger_than}{\code{ImapCon$search_larger_than()}}
\item \href{#method-ImapCon-search_smaller_than}{\code{ImapCon$search_smaller_than()}}
\item \href{#method-ImapCon-search_before}{\code{ImapCon$search_before()}}
\item \href{#method-ImapCon-search_since}{\code{ImapCon$search_since()}}
\item \href{#method-ImapCon-search_on}{\code{ImapCon$search_on()}}
\item \href{#method-ImapCon-search_period}{\code{ImapCon$search_period()}}
\item \href{#method-ImapCon-search_sent_before}{\code{ImapCon$search_sent_before()}}
\item \href{#method-ImapCon-search_sent_since}{\code{ImapCon$search_sent_since()}}
\item \href{#method-ImapCon-search_sent_on}{\code{ImapCon$search_sent_on()}}
\item \href{#method-ImapCon-search_sent_period}{\code{ImapCon$search_sent_period()}}
\item \href{#method-ImapCon-search_flag}{\code{ImapCon$search_flag()}}
\item \href{#method-ImapCon-search_older_than}{\code{ImapCon$search_older_than()}}
\item \href{#method-ImapCon-search_younger_than}{\code{ImapCon$search_younger_than()}}
\item \href{#method-ImapCon-search_string}{\code{ImapCon$search_string()}}
\item \href{#method-ImapCon-fetch_body}{\code{ImapCon$fetch_body()}}
\item \href{#method-ImapCon-fetch_header}{\code{ImapCon$fetch_header()}}
\item \href{#method-ImapCon-fetch_metadata}{\code{ImapCon$fetch_metadata()}}
\item \href{#method-ImapCon-fetch_text}{\code{ImapCon$fetch_text()}}
\item \href{#method-ImapCon-copy_msg}{\code{ImapCon$copy_msg()}}
\item \href{#method-ImapCon-move_msg}{\code{ImapCon$move_msg()}}
\item \href{#method-ImapCon-esearch_count}{\code{ImapCon$esearch_count()}}
\item \href{#method-ImapCon-delete_msg}{\code{ImapCon$delete_msg()}}
\item \href{#method-ImapCon-expunge}{\code{ImapCon$expunge()}}
\item \href{#method-ImapCon-esearch_min_id}{\code{ImapCon$esearch_min_id()}}
\item \href{#method-ImapCon-esearch_max_id}{\code{ImapCon$esearch_max_id()}}
\item \href{#method-ImapCon-add_flags}{\code{ImapCon$add_flags()}}
\item \href{#method-ImapCon-replace_flags}{\code{ImapCon$replace_flags()}}
\item \href{#method-ImapCon-remove_flags}{\code{ImapCon$remove_flags()}}
\item \href{#method-ImapCon-get_attachments}{\code{ImapCon$get_attachments()}}
\item \href{#method-ImapCon-fetch_attachments_list}{\code{ImapCon$fetch_attachments_list()}}
\item \href{#method-ImapCon-fetch_attachments}{\code{ImapCon$fetch_attachments()}}
\item \href{#method-ImapCon-clone}{\code{ImapCon$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-new"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-new}{}}}
\subsection{Method \code{new()}}{
Configure and create a new IMAP connection.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$new(
  url,
  username,
  password = NULL,
  xoauth2_bearer = NULL,
  use_ssl = TRUE,
  verbose = FALSE,
  buffersize = 16000,
  timeout_ms = 0,
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{url}}{A character string containing the IMAP server address}

\item{\code{username}}{A character string containing the username.}

\item{\code{password}}{A character string containing the user's password.}

\item{\code{xoauth2_bearer}}{A character string containing the oauth2 bearer token.}

\item{\code{use_ssl}}{A logical indicating the use or not of Secure Sockets Layer
encryption when connecting to the IMAP server. Default is \code{TRUE}.}

\item{\code{verbose}}{If \code{FALSE}, mutes the flow of information between the
server and the client. Default is \code{FALSE}.}

\item{\code{buffersize}}{The size in bytes for the receive buffer. Default is
16000 bytes or 16kb, which means it will use the libcurl's default value.
According to the libcurl's documentation, the maximum buffersize is 512kb
(or 512000 bytes), but any number passed to \code{buffersize} is treated
as a request, not an order.}

\item{\code{timeout_ms}}{Time in milliseconds (ms) to wait for the execution or
re-execution of a command. Default is 0, which means that no timeout limit is
set.}

\item{\code{...}}{Further curl parameters (see \code{curl::curl_options}) that
can be used with the IMAP protocol. Only for advanced users.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new `ImapCon` object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-reset_url"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-reset_url}{}}}
\subsection{Method \code{reset_url()}}{
Reset the previously informed url
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$reset_url(x)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{x}}{A character string containing a new url to be set.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-reset_username"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-reset_username}{}}}
\subsection{Method \code{reset_username()}}{
Reset the previously informed username
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$reset_username(x)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{x}}{A character string containing a new username to be set.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-reset_use_ssl"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-reset_use_ssl}{}}}
\subsection{Method \code{reset_use_ssl()}}{
Reset the previously informed use_ssl parameter
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$reset_use_ssl(x)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{x}}{A logical indicating the use or not of Secure Sockets Layer
encryption when connecting to the IMAP server. Default is \code{TRUE}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-reset_verbose"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-reset_verbose}{}}}
\subsection{Method \code{reset_verbose()}}{
Reset the previously informed verbose parameter
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$reset_verbose(x)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{x}}{If \code{FALSE}, mutes the flow of information between the
server and the client.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-reset_buffersize"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-reset_buffersize}{}}}
\subsection{Method \code{reset_buffersize()}}{
Reset the previously informed buffersize parameter
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$reset_buffersize(x)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{x}}{The size in bytes for the receive buffer. Default is
16000 bytes or 16kb, which means it will use the libcurl's default value.
According to the libcurl's documentation, the maximum buffersize is 512kb
(or 512000 bytes), but any number passed to \code{buffersize} is treated
as a request, not an order.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-reset_timeout_ms"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-reset_timeout_ms}{}}}
\subsection{Method \code{reset_timeout_ms()}}{
Reset the previously informed buffersize parameter
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$reset_timeout_ms(x)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{x}}{Time in milliseconds (ms) to wait for the execution or
re-execution of a command. Default is 0, which means that no timeout limit is
set.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-reset_password"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-reset_password}{}}}
\subsection{Method \code{reset_password()}}{
Reset the previously informed password
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$reset_password(x)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{x}}{A character string containing the user's password.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-reset_xoauth2_bearer"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-reset_xoauth2_bearer}{}}}
\subsection{Method \code{reset_xoauth2_bearer()}}{
Reset the previously informed oauth2 bearer token
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$reset_xoauth2_bearer(x)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{x}}{A character string containing the oauth2 bearer token.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-list_server_capabilities"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-list_server_capabilities}{}}}
\subsection{Method \code{list_server_capabilities()}}{
List the server's IMAP capabilities.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$list_server_capabilities(retries = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A character \code{vector} containing the server's IMAP capabilities.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
cap <- con$list_server_capabilities()
cap
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-list_mail_folders"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-list_mail_folders}{}}}
\subsection{Method \code{list_mail_folders()}}{
List mail folders in a mailbox.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$list_mail_folders(retries = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{list} containing the mail folder names and their inherent
  structure.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
folders <- con$list_mail_folders()
folders
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-select_folder"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-select_folder}{}}}
\subsection{Method \code{select_folder()}}{
Select a mail folder.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$select_folder(name, mute = FALSE, retries = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{A string containing the name of an existing mail folder on the
user's mailbox.}

\item{\code{mute}}{A \code{logical}. If \code{TRUE}, mutes the confirmation message
when the command is successfully executed. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{list} containing the mail folder names and their inherent
  structure.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_mail_folder(name = "INBOX")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-examine_folder"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-examine_folder}{}}}
\subsection{Method \code{examine_folder()}}{
Examine the number of messages in a mail folder.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$examine_folder(name = NULL, retries = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{A \code{character} string containing the name of an existing
mail folder on the user's mailbox. If no name is passed, the command
will be executed using the previously selected mail folder name.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{vector} (with names \code{"EXISTS"} and \code{"RECENT"})
  containing the number of messages in each category.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
con$examine_folder()

# or directly:
con$examine_folder("Sent")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-create_folder"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-create_folder}{}}}
\subsection{Method \code{create_folder()}}{
Create a new mail folder.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$create_folder(name, mute = FALSE, retries = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{A string containing the name of the new mail folder to be
created.}

\item{\code{mute}}{A \code{logical}. If \code{TRUE}, mutes the confirmation message
when the command is successfully executed. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\code{TRUE} in case the operation is successful.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$create_folder(name = "New Folder Name")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-rename_folder"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-rename_folder}{}}}
\subsection{Method \code{rename_folder()}}{
Rename a mail folder.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$rename_folder(
  name = NULL,
  new_name,
  reselect = TRUE,
  mute = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{A string containing the name of the new mail folder to be
renamed. If no name is passed, the command will be executed using the
previously selected mail folder name.}

\item{\code{new_name}}{A string containing the new name to be assigned.}

\item{\code{reselect}}{A logical. If \code{TRUE}, calls
\code{select_folder(name = to_folder)} under the hood before returning
the output. Default is \code{TRUE}.}

\item{\code{mute}}{A \code{logical}. If \code{TRUE}, mutes the confirmation message
when the command is successfully executed. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\code{TRUE} in case the operation is successful.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "Folder A")
con$rename_folder(new_name = "Folder B")
# or directly:
con$rename_folder(name = "Folder A", new_name = "Folder B")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-list_flags"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-list_flags}{}}}
\subsection{Method \code{list_flags()}}{
List flags in a selected mail folder
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$list_flags(retries = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\code{TRUE} in case the operation is successful.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
con$list_flags()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search}{}}}
\subsection{Method \code{search()}}{
Execute a custom search
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search(
  request,
  negate = FALSE,
  use_uid = FALSE,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{request}}{A string directly specifying what to search or
constructed by a combination of relational-operator-helper-functions \code{\link{OR}}
and \code{\link{AND}}, and criteria helper functions such as
\code{\link{before}}, \code{\link{since}}, \code{\link{on}},
\code{\link{sent_before}}, \code{\link{sent_since}}, \code{\link{sent_on}},
\code{\link{flag}}, \code{\link{string}}, \code{\link{smaller_than}},
\code{\link{larger_than}}, \code{\link{younger_than}}, or
\code{\link{younger_than}}.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERIA". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{list} containing the flags (\code{character vector}),
  the permanent flags (\code{character vector}), and an indication if custom
  flags are allowed by the server (\code{logical vector}).
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# ex1
con$search(OR(before(date_char = "17-Apr-2015"),
              string(expr = "John", where = "FROM")))

# ex2
con$search(AND(smaller_than(size = "512000"),
               string(expr = "John", where = "FROM"),
               string(expr = "@ksu.edu", where = "CC")))
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_larger_than"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_larger_than}{}}}
\subsection{Method \code{search_larger_than()}}{
Search by size (LARGER)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_larger_than(
  size,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{size}}{An integer specifying the size in bytes to be used as the
search criterion.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# search for messages with size larger than 512Kb
con$search_larger_than(size = 512000))
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_smaller_than"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_smaller_than}{}}}
\subsection{Method \code{search_smaller_than()}}{
Search by size (SMALLER)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_smaller_than(
  size,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{size}}{An integer specifying the size in bytes to be used as the
search criterion.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# search for messages with size smaller than 512Kb
con$search_smaller_than(size = 512000))
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_before"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_before}{}}}
\subsection{Method \code{search_before()}}{
Search by internal date (BEFORE)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_before(
  date_char,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{date_char}}{A \code{character string} with format "DD-Mon-YYYY", e.g.
"01-Apr-2019". We opt not to use \code{Date} or \code{POSIX*} like
objects, since IMAP servers use this uncommon date format.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# search for messages with date before "02-Jan-2020", presenting the
# .. results as unique identifiers (UID)
con$search_before(date = "02-Jan-2020", use_uid = TRUE)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_since"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_since}{}}}
\subsection{Method \code{search_since()}}{
Search by internal date (SINCE)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_since(
  date_char,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{date_char}}{A \code{character string} with format "DD-Mon-YYYY", e.g.
"01-Apr-2019". We opt not to use \code{Date} or \code{POSIX*} like
objects, since IMAP servers use this uncommon date format.
\code{POSIX*} like objects, since IMAP servers use this uncommon date format.
\code{POSIX*} like, since IMAP servers like this not so common date format.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# search for messages with date since "02-Jan-2020", presenting the
# .. results as unique identifiers (UID)
con$search_since(date = "02-Jan-2020", use_uid = TRUE)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_on"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_on}{}}}
\subsection{Method \code{search_on()}}{
Search by internal date (ON)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_on(
  date_char,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{date_char}}{A \code{character string} with format "DD-Mon-YYYY", e.g.
"01-Apr-2019". We opt not to use \code{Date} or \code{POSIX*} like
objects, since IMAP servers use this uncommon date format.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# search for messages received on date "02-Jan-2020", presenting the
#... results as unique identifiers (UID)
con$search_on(date = "02-Jan-2020", use_uid = TRUE)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_period"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_period}{}}}
\subsection{Method \code{search_period()}}{
Search by internal date (Period)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_period(
  since_date_char,
  before_date_char,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{since_date_char}}{A \code{character string} with format "DD-Mon-YYYY", e.g.
"01-Apr-2019". We opt not to use \code{Date} or \code{POSIX*} like
objects, since IMAP servers use this uncommon date format.}

\item{\code{before_date_char}}{A \code{character string} with format "DD-Mon-YYYY", e.g.
"01-Apr-2019". We opt not to use \code{Date} or \code{POSIX*} like
objects, since IMAP servers use this uncommon date format.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# search for all messages in the mail folder, EXCEPT (negate = TRUE) by
#... those received between the dates "02-Jan-2020" and "22-Mar-2020"
con$search_period(since_date_char = "02-Jan-2020",
                  before_date_char = "22-Mar-2020",
                  negate = TRUE))
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_sent_before"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_sent_before}{}}}
\subsection{Method \code{search_sent_before()}}{
Search by origination date  (RFC 2822 Header - SENT BEFORE)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_sent_before(
  date_char,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{date_char}}{A \code{character string} with format "DD-Mon-YYYY", e.g.
"01-Apr-2019". We opt not to use \code{Date} or \code{POSIX*} like
objects, since IMAP servers use this uncommon date format.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# search for messages with date before "02-Jan-2020", presenting the
# .. results as unique identifiers (UID)
con$search_sent_before(date = "02-Jan-2020", use_uid = TRUE)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_sent_since"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_sent_since}{}}}
\subsection{Method \code{search_sent_since()}}{
Search by origination date (RFC 2822 Header - SENT SINCE)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_sent_since(
  date_char,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{date_char}}{A \code{character string} with format "DD-Mon-YYYY", e.g.
"01-Apr-2019". We opt not to use \code{Date} or \code{POSIX*} like
objects, since IMAP servers use this uncommon date format.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# search for messages with date before "02-Jan-2020", presenting the
# .. results as unique identifiers (UID)
con$search_sent_since(date = "02-Jan-2020", use_uid = TRUE)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_sent_on"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_sent_on}{}}}
\subsection{Method \code{search_sent_on()}}{
Search by origination date (RFC 2822 Header - SENT ON)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_sent_on(
  date_char,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{date_char}}{A \code{character string} with format "DD-Mon-YYYY", e.g.
"01-Apr-2019". We opt not to use \code{Date} or \code{POSIX*} like
objects, since IMAP servers use this uncommon date format.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# search for messages received on date "02-Jan-2020", presenting the
#... results as unique identifiers (UID)
con$search_sent_on(date = "02-Jan-2020", use_uid = TRUE)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_sent_period"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_sent_period}{}}}
\subsection{Method \code{search_sent_period()}}{
Search by origination date (RFC 2822 Header - SENT Period)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_sent_period(
  since_date_char,
  before_date_char,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{since_date_char}}{A \code{character string} with format "DD-Mon-YYYY", e.g.
"01-Apr-2019". We opt not to use \code{Date} or \code{POSIX*} like
objects, since IMAP servers use this uncommon date format.}

\item{\code{before_date_char}}{A \code{character string} with format "DD-Mon-YYYY", e.g.
"01-Apr-2019". We opt not to use \code{Date} or \code{POSIX*} like
objects, since IMAP servers use this uncommon date format.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# search for all messages in the mail folder, EXCEPT (negate = TRUE) by
#... those received between the dates "02-Jan-2020" and "22-Mar-2020"
con$search_sent_period(since_date_char = "02-Jan-2020",
                  before_date_char = "22-Mar-2020",
                  negate = TRUE))
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_flag"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_flag}{}}}
\subsection{Method \code{search_flag()}}{
Search by flag(s)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_flag(
  name,
  negate = FALSE,
  use_uid = FALSE,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{A string containing one or more flags to search for. Use
\href{#method-list_flags}{\code{ImapCon$list_flags()}} to list the flags
in a selected mail folder.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# search for all messages in the mail folder that are marked as "SEEN" AND
#.. "ANSWERED"
con$search_flag(name = c("SEEN", "ANSWERED"))
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_older_than"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_older_than}{}}}
\subsection{Method \code{search_older_than()}}{
Search WITHIN a specific time (OLDER)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_older_than(
  seconds,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{seconds}}{An integer specifying the number of seconds to be used as
the search criterion.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# search for all messages received in the last hour (not older than 3600 seconds)
con$search_older_than(seconds = 3600, negate = TRUE)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_younger_than"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_younger_than}{}}}
\subsection{Method \code{search_younger_than()}}{
Search WITHIN a specific time (YOUNGER)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_younger_than(
  seconds,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{seconds}}{An integer specifying the number of seconds to be used as
the search criterion.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# search for all messages received in the last hour (younger than 3600 seconds)
con$search_younger_than(seconds = 3600)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-search_string"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-search_string}{}}}
\subsection{Method \code{search_string()}}{
Search by string or expression
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$search_string(
  expr,
  where,
  negate = FALSE,
  use_uid = FALSE,
  flag = NULL,
  esearch = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{expr}}{A character string specifying the word or expression to search
for in messages.}

\item{\code{where}}{A mandatory character string specifying in which
message's Section or Header Field to search for the provided string.}

\item{\code{negate}}{If \code{TRUE}, negates the search and seeks for "NOT SEARCH
CRITERION". Default is \code{FALSE}.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{flag}}{An optional argument that sets one or more flags as an additional
filter to the search. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder. Default is \code{NULL}.}

\item{\code{esearch}}{A logical. Default is \code{FALSE}. If the IMAP server has
\code{ESEARCH} capability, it can be used to optimize search results. It
will condense the results: instead of writing down the whole sequences of messages'
ids, such as \code{\{1 2 3 4 5\}}, it will be presented as \code{\{1:5\}},
which decreases transmission costs. This argument can be used along with
\code{buffersize} to avoid results stripping. Check if your IMAP server
supports \code{ESEARCH} with
\href{#method-list_server_capabilities}{\code{ImapCon$list_server_capabilities()}}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# search for all messages received in the last hour (younger than 3600 seconds)
con$search_string(expr = "@k-state.edu", where = "FROM")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-fetch_body"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-fetch_body}{}}}
\subsection{Method \code{fetch_body()}}{
Fetch message body (message's full content)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$fetch_body(
  msg_id,
  use_uid = FALSE,
  mime_level = NULL,
  peek = TRUE,
  partial = NULL,
  write_to_disk = FALSE,
  keep_in_mem = TRUE,
  mute = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_id}}{A \code{numeric vector} containing one or more message ids.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, the operation will
be performed using message sequence numbers. A message sequence number
is a message's relative position to the oldest message in a mail folder.
It may change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier.
UIDs are always the same during the life cycle of a message in a mail folder.}

\item{\code{mime_level}}{An \code{integer} specifying MIME multipart to fetch from
the message's body. Default is \code{NULL}, which retrieves the full body content.}

\item{\code{peek}}{If \code{TRUE}, it does not mark messages as "read" after
fetching. Default is \code{TRUE}.}

\item{\code{partial}}{\code{NULL} or a character string with format
"startchar.endchar" indicating the size (in characters) of a message slice
to fetch. Default is \code{NULL}, which will fetch the full specified content.}

\item{\code{write_to_disk}}{If \code{TRUE}, writes the fetched content of each message
to a text file in a local folder inside the working directory, also
returning the results with \code{invisible()}. Default is \code{FALSE}.}

\item{\code{keep_in_mem}}{If \code{TRUE}, keeps a copy of each fetch result while
the operation is being performed with \code{write_to_disk = TRUE}. Default
is \code{FALSE}, and it can only be set \code{TRUE} when
\code{write_to_disk = TRUE}.}

\item{\code{mute}}{A \code{logical}. It provides a confirmation message if the
command is successfully executed. It is only effective when \code{write_to_disk = TRUE}
and \code{keep_in_mem = FALSE}. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command. Default
is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{list} with the fetch contents or a logical if
  \code{write_to_disk = TRUE} and \code{keep_in_mem = FALSE}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# do a search and fetch the results (saving to disk) using the pipe
con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$fetch_body(write_to_disk = TRUE, keep_in_mem = FALSE)

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")

con$fetch_body(msg = res, write_to_disk = TRUE, keep_in_mem = FALSE)

}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-fetch_header"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-fetch_header}{}}}
\subsection{Method \code{fetch_header()}}{
Fetch message header
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$fetch_header(
  msg_id,
  use_uid = FALSE,
  fields = NULL,
  negate_fields = FALSE,
  peek = TRUE,
  partial = NULL,
  write_to_disk = FALSE,
  keep_in_mem = TRUE,
  mute = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_id}}{A \code{numeric vector} containing one or more message ids.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, the operation will
be performed using message sequence numbers. A message sequence number
is a message's relative position to the oldest message in a mail folder.
It may change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier.
UIDs are always the same during the life cycle of a message in a mail folder.}

\item{\code{fields}}{An optional \code{character vector} specifying which field(s)
will be fetched from the message's header. If none is specified, it will
fetch the full header.}

\item{\code{negate_fields}}{If \code{TRUE}, negates the operation and seeks for
"NOT in the field". Default is \code{FALSE}.}

\item{\code{peek}}{If \code{TRUE}, it does not mark messages as "read" after
fetching. Default is \code{TRUE}.}

\item{\code{partial}}{\code{NULL} or a character string with format
"startchar.endchar" indicating the size (in characters) of a message slice
to fetch. Default is \code{NULL}, which will fetch the full specified content.}

\item{\code{write_to_disk}}{If \code{TRUE}, writes the fetched content of each message
to a text file in a local folder inside the working directory, also
returning the results with \code{invisible()}. Default is \code{FALSE}.}

\item{\code{keep_in_mem}}{If \code{TRUE}, keeps a copy of each fetch result while
the operation is being performed with \code{write_to_disk = TRUE}. Default
is \code{FALSE}, and it can only be set \code{TRUE} when
\code{write_to_disk = TRUE}.}

\item{\code{mute}}{A \code{logical}. It provides a confirmation message if the
command is successfully executed. It is only effective when \code{write_to_disk = TRUE}
and \code{keep_in_mem = FALSE}. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command. Default
is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{list} with the fetch contents or a logical if
  \code{write_to_disk = TRUE} and \code{keep_in_mem = FALSE}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# do a search and fetch the results (also saving to disk) using the pipe
out <- con$search_string(expr = "@k-state.edu", where = "CC") \%>\%
  con$fetch_header()

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "CC")
out <- con$fetch_header()

}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-fetch_metadata"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-fetch_metadata}{}}}
\subsection{Method \code{fetch_metadata()}}{
Fetch message metadata
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$fetch_metadata(
  msg_id,
  use_uid = FALSE,
  attribute = NULL,
  write_to_disk = FALSE,
  keep_in_mem = TRUE,
  mute = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_id}}{A \code{numeric vector} containing one or more message ids.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, the operation will
be performed using message sequence numbers. A message sequence number
is a message's relative position to the oldest message in a mail folder.
It may change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier.
UIDs are always the same during the life cycle of a message in a mail folder.}

\item{\code{attribute}}{An optional \code{character vector} specifying one or more
attributes of the metadata of a message to fetch. See \link{metadata_options}.}

\item{\code{write_to_disk}}{If \code{TRUE}, writes the fetched content of each message
to a text file in a local folder inside the working directory, also
returning the results with \code{invisible()}. Default is \code{FALSE}.}

\item{\code{keep_in_mem}}{If \code{TRUE}, keeps a copy of each fetch result while
the operation is being performed with \code{write_to_disk = TRUE}. Default
is \code{FALSE}, and it can only be set \code{TRUE} when
\code{write_to_disk = TRUE}.}

\item{\code{mute}}{A \code{logical}. It provides a confirmation message if the
command is successfully executed. It is only effective when \code{write_to_disk = TRUE}
and \code{keep_in_mem = FALSE}. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command. Default
is \code{1}.}

\item{\code{peek}}{If \code{TRUE}, it does not mark messages as "read" after
fetching. Default is \code{TRUE}.}

\item{\code{partial}}{\code{NULL} or a character string with format
"startchar.endchar" indicating the size (in characters) of a message slice
to fetch. Default is \code{NULL}, which will fetch the full specified content.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{list} with the fetch contents or a logical if
  \code{write_to_disk = TRUE} and \code{keep_in_mem = FALSE}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# do a search and fetch the results using the pipe
out <- con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$fetch_metadata()

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")
out <- con$fetch_metadata(msg = res)

}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-fetch_text"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-fetch_text}{}}}
\subsection{Method \code{fetch_text()}}{
Fetch message text
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$fetch_text(
  msg_id,
  use_uid = FALSE,
  peek = TRUE,
  partial = NULL,
  write_to_disk = FALSE,
  keep_in_mem = TRUE,
  mute = FALSE,
  base64_decode = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_id}}{A \code{numeric vector} containing one or more message ids.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, the operation will
be performed using message sequence numbers. A message sequence number
is a message's relative position to the oldest message in a mail folder.
It may change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier.
UIDs are always the same during the life cycle of a message in a mail folder.}

\item{\code{peek}}{If \code{TRUE}, it does not mark messages as "read" after
fetching. Default is \code{TRUE}.}

\item{\code{partial}}{\code{NULL} or a character string with format
"startchar.endchar" indicating the size (in characters) of a message slice
to fetch. Default is \code{NULL}, which will fetch the full specified content.}

\item{\code{write_to_disk}}{If \code{TRUE}, writes the fetched content of each message
to a text file in a local folder inside the working directory, also
returning the results with \code{invisible()}. Default is \code{FALSE}.}

\item{\code{keep_in_mem}}{If \code{TRUE}, keeps a copy of each fetch result while
the operation is being performed with \code{write_to_disk = TRUE}. Default
is \code{FALSE}, and it can only be set \code{TRUE} when
\code{write_to_disk = TRUE}.}

\item{\code{mute}}{A \code{logical}. It provides a confirmation message if the
command is successfully executed. It is only effective when \code{write_to_disk = TRUE}
and \code{keep_in_mem = FALSE}. Default is \code{FALSE}.}

\item{\code{base64_decode}}{If \code{TRUE}, tries to guess and decode the fetched
text from base64 format to \code{character}. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command. Default
is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{list} with the fetch contents or a logical if
  \code{write_to_disk = TRUE} and \code{keep_in_mem = FALSE}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# do a search and partially fetch the results using the pipe
# first 200 characters, writing to disk, silence results in the console
con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$fetch_text(partial = "0.200",
                 write_to_disk = TRUE,
                 keep_in_mem = FALSE)

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")
con$fetch_text(msg = res,
               partial = "0.200",
               write_to_disk = TRUE,
               keep_in_mem = FALSE)

}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-copy_msg"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-copy_msg}{}}}
\subsection{Method \code{copy_msg()}}{
Copy message(s) between the selected folder and another one
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$copy_msg(
  msg_id,
  use_uid = FALSE,
  to_folder,
  reselect = TRUE,
  mute = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_id}}{A \code{numeric vector} containing one or more message ids.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, the operation will
be performed using message sequence numbers. A message sequence number
is a message's relative position to the oldest message in a mail folder.
It may change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier.
UIDs are always the same during the life cycle of a message in a mail folder.}

\item{\code{to_folder}}{A \code{character} string specifying the folder to which
the messages will be copied.}

\item{\code{reselect}}{A logical. If \code{TRUE}, calls
\href{#method-select_folder}{\code{ImapCon$select_folder(name = to_folder)}}
under the hood before returning the output. Default is \code{TRUE}.}

\item{\code{mute}}{A \code{logical}. If \code{TRUE}, mutes the confirmation message
when the command is successfully executed. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An invisible \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# do a search and copy the results to another folder
con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$copy(to_folder = "Sent")

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")
con$copy(msg = res, to_folder = "Sent")

}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-move_msg"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-move_msg}{}}}
\subsection{Method \code{move_msg()}}{
Move message(s) between the selected folder and another one
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$move_msg(
  msg_id,
  use_uid = FALSE,
  to_folder,
  reselect = TRUE,
  mute = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_id}}{A \code{numeric vector} containing one or more message ids.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, the operation will
be performed using message sequence numbers. A message sequence number
is a message's relative position to the oldest message in a mail folder.
It may change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier.
UIDs are always the same during the life cycle of a message in a mail folder.}

\item{\code{to_folder}}{A \code{character} string specifying the folder to which
the messages will be copied.}

\item{\code{reselect}}{A logical. If \code{TRUE}, calls
\href{#method-select_folder}{\code{ImapCon$select_folder(name = to_folder)}}
under the hood before returning the output. Default is \code{TRUE}.}

\item{\code{mute}}{A \code{logical}. If \code{TRUE}, mutes the confirmation message
when the command is successfully executed. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An invisible \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# do a search and copy the results to another folder
con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$move(to_folder = "Sent")

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")
con$move(msg = res, to_folder = "Sent")

}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-esearch_count"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-esearch_count}{}}}
\subsection{Method \code{esearch_count()}}{
Count the number of messages with a specific flag(s) in a
  folder (depend on ESEARCH capability)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$esearch_count(flag, use_uid = FALSE, retries = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{flag}}{A mandatory parameter that specifies one or more flags as a
filter to the counting operation. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A numeric \code{vector} of length \code{1} containing the number
  of messages in the folder that meet the specified criteria.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# count the number of messages marked as "Flagged" and "Answered"
con$esearch_count(flag = c("Flagged", "Answered"))
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-delete_msg"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-delete_msg}{}}}
\subsection{Method \code{delete_msg()}}{
Delete message(s) in the selected mail folder
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$delete_msg(msg_id, use_uid = FALSE, mute = FALSE, retries = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_id}}{A \code{numeric vector} containing one or more message ids.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, the operation will
be performed using message sequence numbers. A message sequence number
is a message's relative position to the oldest message in a mail folder.
It may change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier.
UIDs are always the same during the life cycle of a message in a mail folder.}

\item{\code{mute}}{A \code{logical}. If \code{TRUE}, mutes the confirmation message
when the command is successfully executed. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An invisible \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# delete
con$delete_msg(flag = c("Flagged", "Answered"))
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-expunge"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-expunge}{}}}
\subsection{Method \code{expunge()}}{
Permanently removes all or specific messages marked as deleted from the selected folder
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$expunge(msg_uid = NULL, mute = FALSE, retries = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_uid}}{A \code{numeric vector} containing one or more messages UIDs.
Only UIDs are allowed in this operation (note the "u" in msg_\emph{u}id).}

\item{\code{mute}}{A \code{logical}. If \code{TRUE}, mutes the confirmation message
when the command is successfully executed. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\code{TRUE} if the operation is successful.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# count the number of messages marked as "Flagged" and "Answered"
con$esearch_count(flag = c("Flagged", "Answered"))
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-esearch_min_id"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-esearch_min_id}{}}}
\subsection{Method \code{esearch_min_id()}}{
Search the minimum message id in the selected mail folder
  (depend on ESEARCH capability)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$esearch_min_id(flag, use_uid = FALSE, retries = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{flag}}{A mandatory parameter that specifies one or more flags as a
filter to the searching operation. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A numeric \code{vector} of length \code{1} containing the minimum
  message id in the folder.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# Search the minimum id of messages marked as "Answered"
con$esearch_min_id(flag = "Answered")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-esearch_max_id"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-esearch_max_id}{}}}
\subsection{Method \code{esearch_max_id()}}{
Search the maximum message id in the selected mail folder
  (depend on ESEARCH capability)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$esearch_max_id(flag, use_uid = FALSE, retries = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{flag}}{A mandatory parameter that specifies one or more flags as a
filter to the searching operation. Use \href{#method-list_flags}{\code{ImapCon$list_flags()}}
to list the flags in a selected mail folder.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, results will be
presented as message sequence numbers. A message sequence number is a
message's relative position to the oldest message in a mail folder. It may
change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier,
and results are presented as such. UIDs are always the same during the
life cycle of a message in a mail folder.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A numeric \code{vector} of length \code{1} containing the maximum
  message id in the folder.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# Search the minimum id of messages marked as "Seen"
con$esearch_max_id(flag = "Seen")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-add_flags"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-add_flags}{}}}
\subsection{Method \code{add_flags()}}{
Add flags to one or more messages
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$add_flags(
  msg_id,
  use_uid = FALSE,
  flags_to_set,
  mute = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_id}}{A \code{numeric vector} containing one or more message ids.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, the operation will
be performed using message sequence numbers. A message sequence number
is a message's relative position to the oldest message in a mail folder.
It may change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier.
UIDs are always the same during the life cycle of a message in a mail folder.}

\item{\code{flags_to_set}}{A \code{character vector} containing one or more flag
names to add to the specified message ids. If the flag to be set is a
system flag, such as \code{\\SEEN}, \code{\\ANSWERED}, the name should be
preceded by two backslashes \code{\\}.}

\item{\code{mute}}{A \code{logical}. If \code{TRUE}, mutes the confirmation message
when the command is successfully executed. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An invisible \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# Add the "\\Seen" permanent flag to the messages received in the last hour
con$search_younger_than(seconds = 3600) \%>\% # depends on the WITHIN extension
  con$add_flags(flags_to_set = "\\\\Seen")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-replace_flags"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-replace_flags}{}}}
\subsection{Method \code{replace_flags()}}{
Replace the current flags of one or more messages
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$replace_flags(
  msg_id,
  use_uid = FALSE,
  flags_to_set,
  mute = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_id}}{A \code{numeric vector} containing one or more message ids.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, the operation will
be performed using message sequence numbers. A message sequence number
is a message's relative position to the oldest message in a mail folder.
It may change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier.
UIDs are always the same during the life cycle of a message in a mail folder.}

\item{\code{flags_to_set}}{A \code{character vector} containing one or more flag
names that will replace the current ones. If the flag to be set is a
system flag, such as \code{\\SEEN}, \code{\\ANSWERED}, the name should be
preceded by two backslashes \code{\\}.}

\item{\code{mute}}{A \code{logical}. If \code{TRUE}, mutes the confirmation message
when the command is successfully executed. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An invisible \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# Replace the current flags of the messages in the search results for the
#.. flags "\\UNSEEN" and "\\Flagged"
con$search_since(date_char = "20-Aug-2020") \%>\%
  con$replace_flags(flags_to_set = c("\\\\UNSEEN", "\\\\Flagged")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-remove_flags"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-remove_flags}{}}}
\subsection{Method \code{remove_flags()}}{
Remove flag(s) of one or more messages
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$remove_flags(
  msg_id,
  use_uid = FALSE,
  flags_to_unset,
  mute = FALSE,
  retries = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_id}}{A \code{numeric vector} containing one or more message ids.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, the operation will
be performed using message sequence numbers. A message sequence number
is a message's relative position to the oldest message in a mail folder.
It may change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier.
UIDs are always the same during the life cycle of a message in a mail folder.}

\item{\code{flags_to_unset}}{A \code{character vector} containing one or more
flag names that will be unset (removed). If the flag to be removed is a
system flag, such as \code{\\SEEN}, \code{\\ANSWERED}, the name should be
preceded by two backslashes \code{\\}.}

\item{\code{mute}}{A \code{logical}. If \code{TRUE}, mutes the confirmation message
when the command is successfully executed. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command.
Default is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An invisible \code{numeric vector} containing the message ids.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# Remove the the "\\SEEN" flag from the messages in the search result
con$search_since(date_char = "20-Aug-2020") \%>\%
  con$remove_flags(flags_to_unset = "\\\\UNSEEN")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-get_attachments"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-get_attachments}{}}}
\subsection{Method \code{get_attachments()}}{
Extract attached file(s) from fetched message(s)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$get_attachments(
  msg_list,
  content_disposition = "both",
  override = FALSE,
  mute = FALSE,
  as_is = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_list}}{A \code{list} with the body or text content of the messages
fetched with \href{#method-fetch_body}{\code{ImapCon$fetch_body()}} or
\href{#method-fetch_text}{\code{ImapCon$fetch_text()}}.}

\item{\code{content_disposition}}{A \code{string} indicating which type of
"Content-Disposition" attachments should be retrieved. Default is
\code{"both"}, which retrieves regular attachments ("Content-Disposition:
attachment") and  inline attachments ("Content-Disposition: inline").}

\item{\code{override}}{A \code{logical}. Provides a confirmation message if the
command is successfully executed. Default is \code{FALSE}.}

\item{\code{mute}}{A \code{logical}. If \code{TRUE}, mutes the confirmation message
when the command is successfully executed. Default is \code{FALSE}.}

\item{\code{as_is}}{If \code{TRUE} then write out attachments without base64
decoding. Default is \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\code{TRUE} if the operation is successful. The files are saved
locally.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# example 1
con$select_folder(name = "INBOX")
con$search_string(expr = "@gmail", where = "CC") \%>\%
  con$fetch_text(write_to_disk = TRUE) \%>\% # saving the message's content as txt files
  con$get_attachments()

# example 2
res <- con$search_string(expr = "@gmail", where = "CC") \%>\%
out <- con$fetch_body(msg = res)
con$get_attachments(msg_list = out)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-fetch_attachments_list"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-fetch_attachments_list}{}}}
\subsection{Method \code{fetch_attachments_list()}}{
Fetch attachments' list
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$fetch_attachments_list(msg_id, use_uid = FALSE, retries = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_id}}{A \code{numeric vector} containing one or more message ids.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, the operation will
be performed using message sequence numbers. A message sequence number
is a message's relative position to the oldest message in a mail folder.
It may change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier.
UIDs are always the same during the life cycle of a message in a mail folder.}

\item{\code{retries}}{Number of attempts to connect and execute the command. Default
is \code{1}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{list} with the fetch contents.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# do a search and fetch the attachments' list of the messages
out < con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$fetch_attachments_list()
out

# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")
out <- con$fetch_attachments_list(msg = res)
out

}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-fetch_attachments"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-fetch_attachments}{}}}
\subsection{Method \code{fetch_attachments()}}{
Fetch message attachments
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$fetch_attachments(
  msg_id,
  use_uid = FALSE,
  content_disposition = "both",
  override = FALSE,
  mute = FALSE,
  retries = 1,
  as_is = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{msg_id}}{A \code{numeric vector} containing one or more message ids.}

\item{\code{use_uid}}{Default is \code{FALSE}. In this case, the operation will
be performed using message sequence numbers. A message sequence number
is a message's relative position to the oldest message in a mail folder.
It may change after deleting or moving messages. If a message is deleted,
sequence numbers are reordered to fill the gap. If \code{TRUE}, the
command will be performed using the \code{"UID"} or unique identifier.
UIDs are always the same during the life cycle of a message in a mail folder.}

\item{\code{content_disposition}}{A \code{string} indicating which type of
"Content-Disposition" attachments should be retrieved. The options are
\code{both}, \code{attachment}, and \code{inline}. Default is
\code{"both"}, which retrieves regular attachments ("Content-Disposition:
attachment") and  inline attachments ("Content-Disposition: inline").}

\item{\code{override}}{A \code{logical}. Provides a confirmation message if the
command is successfully executed. Default is \code{FALSE}.}

\item{\code{mute}}{A \code{logical}. If \code{TRUE}, mutes the confirmation message
when the command is successfully executed. Default is \code{FALSE}.}

\item{\code{retries}}{Number of attempts to connect and execute the command. Default
is \code{1}.}

\item{\code{as_is}}{If \code{TRUE} then write out attachments without base64
decoding. Default is \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{list} with the fetch contents.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
con$select_folder(name = "INBOX")
# do a search and fetch the attachments' list of the messages
con$search_string(expr = "@k-state.edu", where = "FROM") \%>\%
  con$fetch_attachments() # the attachments will be downloaded to disk


# or using a traditional approach
res <- con$search_string(expr = "@k-state.edu", where = "FROM")
con$fetch_attachments(msg = res)

}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ImapCon-clone"></a>}}
\if{latex}{\out{\hypertarget{method-ImapCon-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ImapCon$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
