% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lp_lin_panel.R
\name{lp_lin_panel}
\alias{lp_lin_panel}
\title{Compute linear impulse responses with local projections for panel data}
\usage{
lp_lin_panel(data_set = NULL, data_sample = "Full",
  endog_data = NULL, cumul_mult = TRUE, shock = NULL,
  diff_shock = TRUE, iv_reg = FALSE, instrum = NULL,
  panel_model = "within", panel_effect = "individual",
  robust_cov = NULL, robust_method = NULL, robust_type = NULL,
  robust_cluster = NULL, robust_maxlag = NULL, use_gmm = FALSE,
  gmm_model = "onestep", gmm_effect = "twoways",
  gmm_transformation = "d", c_exog_data = NULL, l_exog_data = NULL,
  lags_exog_data = NaN, c_fd_exog_data = NULL, l_fd_exog_data = NULL,
  lags_fd_exog_data = NaN, confint = NULL, hor = NULL)
}
\arguments{
\item{data_set}{A \link{data.frame}, containing the panel data set. The first column has to be the
variable denoting the cross section. The second column has to be the
variable denoting the time section.}

\item{data_sample}{Character or numeric. To use the full sample set value to "Full" (default). To estimate a subset, you have to provide
a sequence of dates. This sequence has to be in the same format as the second column (time-section).}

\item{endog_data}{Character. The column name of the endogenous variable. You can only provide one endogenous variable at a time.}

\item{cumul_mult}{Boolean. Estimate cumulative multipliers? TRUE (default) or FALSE. If TRUE, cumulative responses
are estimated via: \deqn{y_(t+h) - y_(t-1),} where h = 0,..., H-1.}

\item{shock}{Character. The column name of the variable to shock with.}

\item{diff_shock}{Boolean. Take first differences of the shock variable? TRUE (default) or FALSE.}

\item{iv_reg}{Boolean. Use instrument variable approach? TRUE or FALSE.}

\item{instrum}{NULL or Character. The name(s) of the instrument variable(s) if iv_reg = TRUE.}

\item{panel_model}{Character. Type of panel model. The default is "within" (fixed effects). Other options are "random", "ht",
"between", "pooling" or "fd". See vignette of the plm package for details.}

\item{panel_effect}{Character. The effects introduced in the model. Options are "individual" (default), "time", "twoways",
or "nested". See the vignette of the plm-package for details.}

\item{robust_cov}{NULL or Character. The character specifies the method how to estimate robust standard errors: Options are "vcovBK", "vcovDC",
"vcovG", "vcovHC", "vcovNW", "vcovSCC". For these options see vignette of plm package. Another option is "Vcxt". For details see Miller (2017)
                   If "use_gmm = TRUE", this option has to be NULL.}

\item{robust_method}{NULL (default) or Character. The character is an option when robust_cov = "vcovHC". See vignette of the plm package for details.}

\item{robust_type}{NULL (default) or Character. The character is an option when robust_cov  = "vcovBK", "vcovDC", "vcovHC", "vcovNW" or "vcovSCC". See vignette of the plm package for details.}

\item{robust_cluster}{NULL (default) or Character. The character is an option when robust_cov = "vcovBK", "vcovG" or "vcovHC". See vignette of the plm package for details.}

\item{robust_maxlag}{NULL (default) or Character. The character is an option when robust_cov  = "vcovNW" or "vcovSCC". See vignette of the plm package for details.}

\item{use_gmm}{Boolean. Use GMM for estimation? TRUE or FALSE (default). See vignette of plm package for details.
If TRUE, the option "robust_cov" has to be set to NULL.}

\item{gmm_model}{Character. Either "onestep" (default) or "twosteps". See vignette of the plm package for details.}

\item{gmm_effect}{Character. The effects introduced in the model: "twoways" (default) or "individual". See vignette of the plm-package for details.}

\item{gmm_transformation}{Character. Either "d" (default) for the "difference GMM" model or "ld" for the "system GMM".
See vignette of the plm package for details.}

\item{c_exog_data}{NULL or Character. Name(s) of the exogenous variable(s) with contemporaneous impact.}

\item{l_exog_data}{NULL or Character. Name(s) of the exogenous variable(s) with lagged impact.}

\item{lags_exog_data}{Integer. Lag length for the exogenous variable(s) with lagged impact.}

\item{c_fd_exog_data}{NULL or Character. Name(s) of the exogenous variable(s) with contemporaneous impact of first differences.}

\item{l_fd_exog_data}{NULL or Character. Name(s) of exogenous variable(s) with lagged impact of first differences.}

\item{lags_fd_exog_data}{NaN or Integer. Number of lags for variable(s) with impact of first differences.}

\item{confint}{Double. Width of confidence bands. 68\% = 1; 90\% = 1.65; 95\% = 1.96.}

\item{hor}{Integer. Number of horizons for impulse responses.}
}
\value{
A list containing:

\item{irf_lin_mean}{A \link{matrix}, containing the impulse responses.
                  The columns are the horizons.}

\item{irf_lin_low}{A \link{matrix}, containing all lower confidence bands.
                  The columns are the horizons.}

\item{irf_lin_up}{A \link{matrix}, containing all upper confidence bands.
                                  The columns are the horizons.}

\item{reg_summaries}{Regression output for each horizon.}

\item{xy_data_sets}{Data sets with endogenous and exogenous variables for each horizon.}

\item{specs}{A list with data properties for e.g. the plot function.}
}
\description{
This function estimates impulse responses with local projections for panel data, either with an
             identified shock or by an instrument variable approach.
}
\examples{
\donttest{

# This example is based on the STATA code 'LPs_basic_doall.do', provided on
# Òscar Jordà's website (https://sites.google.com/site/oscarjorda/home/local-projections)
# It estimates impulse reponses of the ratio of (mortgage lending/GDP) to a
# +1\% change in the short term interest rate

# Load libraries to download and read excel file from the website
 library(lpirfs)
 library(httr)
 library(readxl)
 library(dplyr)

# Retrieve the JST Macrohistory Database
 url_jst <-"http://www.macrohistory.net/JST/JSTdatasetR3.xlsx"
 GET(url_jst, write_disk(jst_link <- tempfile(fileext = ".xlsx")))
 jst_data <- read_excel(jst_link, 2L)

# Swap the first two columns so that 'country' is the first (cross section) and 'year' the
# second (time section) column
 jst_data <- jst_data \%>\%
             dplyr::filter(year <= 2013) \%>\%
             dplyr::select(country, year, everything())

# Prepare variables. This is based on the 'data.do' file
  data_set <- jst_data \%>\%
               mutate(stir     = stir)                         \%>\%
               mutate(mortgdp  = 100*(tmort/gdp))              \%>\%
               mutate(hpreal   = hpnom/cpi)                    \%>\%
               group_by(country)                               \%>\%
               mutate(hpreal   = hpreal/hpreal[year==1990][1]) \%>\%
               mutate(lhpreal  = log(hpreal))                  \%>\%

               mutate(lhpy     = lhpreal - log(rgdppc))        \%>\%
               mutate(lhpy     = lhpy - lhpy[year == 1990][1]) \%>\%
               mutate(lhpreal  = 100*lhpreal)                  \%>\%
               mutate(lhpy     = 100*lhpy)                     \%>\%
               ungroup()                                       \%>\%

               mutate(lrgdp    = 100*log(rgdppc))              \%>\%
               mutate(lcpi     = 100*log(cpi)) 		            \%>\%
               mutate(lriy     = 100*log(iy*rgdppc))           \%>\%
               mutate(cay      = 100*(ca/gdp))                 \%>\%
               mutate(tnmort   = tloans - tmort)               \%>\%
               mutate(nmortgdp = 100*(tnmort/gdp))             \%>\%
               dplyr::select(country, year, mortgdp, stir, ltrate,
                             lhpy, lrgdp, lcpi, lriy, cay, nmortgdp)


# Use data_sample from 1870 to 2013 and exclude observations during WWI and WWII
  data_sample <-   seq(1870, 2013)[which(!(seq(1870, 2016) \%in\%
                              c(seq(1914, 1918),
                                seq(1939, 1947))))]

# Estimate panel model
results_panel <-  lp_lin_panel(data_set          = data_set,
                               data_sample       = data_sample,
                               endog_data        = "mortgdp",
                               cumul_mult        = TRUE,

                               shock             = "stir",
                               diff_shock        = TRUE,
                               panel_model       = "within",
                               panel_effect      = "individual",
                               robust_cov        = "vcovSCC",

                               c_exog_data       = "cay",
                               l_exog_data       = "cay",
                               lags_exog_data    = 2,
                               c_fd_exog_data    = colnames(data_set)[c(seq(4,9),11)],
                               l_fd_exog_data    = colnames(data_set)[c(seq(3,9),11)],
                               lags_fd_exog_data = 2,

                               confint           = 1.67,
                               hor               = 5)

# Plot irfs
 plot(results_panel)


# Simulate and add instrument to data_set
 set.seed(123)
 data_set   <- data_set \%>\%
               group_by(country) \%>\%
               mutate(instrument = 0.8*stir + rnorm(length(stir), 0, sd(na.omit(stir))/10)) \%>\%
               ungroup()


# Estimate panel model with iv approach
results_panel <-  lp_lin_panel(data_set          = data_set,
                               data_sample       = data_sample,
                               endog_data        = "mortgdp",
                               cumul_mult        = TRUE,

                               shock             = "stir",
                               diff_shock        = TRUE,
                               iv_reg            = TRUE,
                               instrum           = "instrument",
                               panel_model       = "within",
                               panel_effect      = "individual",
                               robust_cov        = "vcovSCC",

                               c_exog_data       = "cay",
                               l_exog_data       = "cay",
                               lags_exog_data    = 2,
                               c_fd_exog_data    = colnames(data_set)[c(seq(4,9),11)],
                               l_fd_exog_data    = colnames(data_set)[c(seq(3,9),11)],
                               lags_fd_exog_data = 2,

                               confint           = 1.67,
                               hor               = 5)

# Create and plot irfs
 plot(results_panel)


##############################################################################
###                           Use GMM                                      ###
##############################################################################


# Use a much smaller sample to have fewer T than N
data_sample <-   seq(2000, 2012)

# Estimate panel model with gmm
# This example gives a warning at each iteration. The data set is not well suited for
# GMM as GMM is based on N-asymptotics and the data set only contains 27 countries

results_panel <-  lp_lin_panel(data_set          = data_set,
                              data_sample        = data_sample,
                              endog_data         = "mortgdp",
                              cumul_mult         = TRUE,

                              shock              = "stir",
                              diff_shock         = TRUE,

                              use_gmm            = TRUE,
                              gmm_model          = "onestep",
                              gmm_effect         = "twoways",
                              gmm_transformation = "ld",

                              l_exog_data        = "mortgdp",
                              lags_exog_data     = 2,
                              l_fd_exog_data     = colnames(data_set)[c(4, 6)],
                              lags_fd_exog_data  = 1,

                              confint            = 1.67,
                              hor                = 5)

# Create and plot irfs
plot(results_panel)



}

}
\references{
Croissant, Y., Millo, G. (2008). "Panel Data Econometrics in R: The plm Package." \emph{Journal of Statistical Software}, 27(2), 1-43. doi:
10.18637/jss.v027.i02 (URL: \url{http://doi.org/10.18637/jss.v027.i02}).

Jordà, Ò. (2005). "Estimation and Inference of Impulse Responses by Local Projections."
\emph{American Economic Review}, 95 (1): 161-182.

Jordà, Ò., Schualrick, M., Taylor, A.M. (2018). "Large and State-Dependent Effects of Quasi-Random Monetary Experiments",
\emph{NBER} working paper 23074, \emph{FRBSF} working paper 2017-02.

Millo G (2017). “Robust Standard Error Estimators for Panel Models: A Unifying Approach.” \emph{Journal of Statistical Software}, 82(3), 1-27. doi:
10.18637/jss.v082.i03 (URL: \url{http://doi.org/10.18637/jss.v082.i03}).
}
\author{
Philipp Adämmer
}
