#' Print methods
#'
#' @export
#' @param x An object returned by \code{\link{loo}}, \code{\link{psis}}, or
#'   \code{\link{waic}}.
#' @param digits An integer passed to \code{\link[base]{round}}.
#' @param plot_k Logical. If \code{TRUE} the estimates of the Pareto shape
#'   parameter \eqn{k} are plotted. Ignored if \code{x} was generated by
#'   \code{\link{waic}}. To just plot \eqn{k} without printing use the
#'   \code{\link[=pareto-k-diagnostic]{plot method}}.
#' @param ... Arguments passed to \code{\link{plot.psis_loo}} if \code{plot_k} is
#'   \code{TRUE}.
#'
#' @return \code{x}, invisibly.
#'
#' @seealso \code{\link{pareto-k-diagnostic}}
#'
print.loo <- function(x, digits = 1, ...) {
  cat("\n")
  print_dims(x)
  if (!("estimates" %in% names(x))) {
    x <- convert_old_object(x)
  }
  cat("\n")
  print(.fr(as.data.frame(x$estimates), digits), quote = FALSE)
  return(invisible(x))
}

#' @export
#' @rdname print.loo
print.waic <- function(x, digits = 1, ...) {
  print.loo(x, digits = digits, ...)
  throw_pwaic_warnings(x$pointwise[, "p_waic"], digits = digits)
  invisible(x)
}

#' @export
#' @rdname print.loo
print.psis_loo <- function(x, digits = 1, plot_k = FALSE, ...) {
  print.loo(x, digits = digits, ...)
  cat("------\n")
  print_mcse_summary(x, digits = digits)
  if (length(pareto_k_ids(x, threshold = 0.5))) {
    cat("\n")
  }
  print(pareto_k_table(x), digits = digits)
  cat(.k_help())
  if (plot_k) {
    plot(x, ...)
  }
  invisible(x)
}


#' @export
#' @rdname print.loo
print.psis <- function(x, digits = 1, plot_k = FALSE, ...) {
  print_dims(x)
  print(pareto_k_table(x), digits = digits)
  cat(.k_help())
  if (plot_k) {
    plot(x, ...)
  }
  invisible(x)
}


# internal ----------------------------------------------------------------

#' Print dimensions of log-likelihood or log-weights matrix
#'
#' @export
#' @keywords internal
#'
#' @param x The object returned by \code{\link{psis}}, \code{\link{loo}}, or
#'   \code{\link{waic}}.
#' @param ... Ignored.
print_dims <- function(x, ...) UseMethod("print_dims")

#' @rdname print_dims
#' @export
print_dims.psis <- function(x, ...) {
  cat(
    "Computed from",
    paste(dim(x), collapse = " by "),
    "log-weights matrix\n"
  )
}

#' @rdname print_dims
#' @export
print_dims.psis_loo <- function(x, ...) {
  cat(
    "Computed from",
    paste(dim(x), collapse = " by "),
    "log-likelihood matrix\n"
  )
}

#' @rdname print_dims
#' @export
print_dims.waic <- function(x, ...) {
  cat(
    "Computed from",
    paste(dim(x), collapse = " by "),
    "log-likelihood matrix\n"
  )
}

#' @rdname print_dims
#' @export
print_dims.kfold <- function(x, ...) {
  K <- attr(x, "K", exact = TRUE)
  if (!is.null(K)) {
    cat("Based on", paste0(K, "-fold"), "cross-validation\n")
  }
}


print_mcse_summary <- function(x, digits) {
  mcse_val <- mcse_loo(x)
  cat(
    "Monte Carlo SE of elpd_loo is",
    paste0(.fr(mcse_val, digits), ".\n")
  )
}


# print and warning helpers
.fr <- function(x, digits) format(round(x, digits), nsmall = digits)
.warn <- function(..., call. = FALSE) warning(..., call. = call.)
.k_help <- function() "See help('pareto-k-diagnostic') for details.\n"

# compatibility with old loo objects
convert_old_object <- function(x, digits = 1, ...) {
  z <- x[-grep("pointwise|pareto_k|n_eff", names(x))]
  uz <- unlist(z)
  nms <- names(uz)
  ses <- grepl("se", nms)
  list(estimates = data.frame(Estimate = uz[!ses], SE = uz[ses]))
}
