#' Print methods
#' @export
#'
#' @param x a list of class \code{'loo'} (as returned by \code{\link{loo}} or
#'   \code{\link{waic}}) or of class \code{'compare.loo'} (as returned by
#'   \code{\link{compare}}).
#' @param ... ignored.
#' @param digits passed to \code{\link[base]{round}}.
#' @param warn logical. If \code{TRUE} (the default), a warning message will be
#'   printed if any estimates for the Pareto shape parameter \eqn{k} are
#'   problematic. See section PSIS-LOO in \code{\link{loo-package}}. Ignored
#'   if \code{x} was generated by \code{\link{waic}}.
#' @param plot_k logical. If \code{TRUE} the estimates of \eqn{k} are plotted.
#' Ignored if \code{x} was generated by \code{\link{waic}}.
#'
#' @return Returns \code{x} invisibly.
#'
print.loo <- function(x, ..., digits = 1, warn = TRUE, plot_k = FALSE) {
  lldims <- paste(attr(x, "log_lik_dim"), collapse = " by ")
  cat("Computed from", lldims, "log-likelihood matrix\n\n")
  z <- x[-grep("pointwise|pareto_k", names(x))]
  uz <- unlist(z)
  nms <- names(uz)
  ses <- grepl("se", nms)
  out <- data.frame(Estimate = uz[!ses], SE = uz[ses])
  out <- format(round(out, digits), nsmall = digits)
  print(out, quote = FALSE)
  if ("pareto_k" %in% names(x)) {
    if (warn)
      k_warnings(x$pareto_k, digits)
    if (plot_k)
      plot_k(x$pareto_k)
  }
  invisible(x)
}

#' @rdname print.loo
#' @export
print.compare.loo <- function(x, ..., digits = 1) {
  ux <- unlist(x)
  names(ux) <- c("elpd_diff", "SE")
  out <- format(round(ux, digits), nsmall = digits)
  print(out, quote = FALSE)
  invisible(x)
}
