// Copyright 2015, 2016, 2017 Ingo Steinwart
//
// This file is part of liquidSVM.
//
// liquidSVM is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as 
// published by the Free Software Foundation, either version 3 of the 
// License, or (at your option) any later version.
//
// liquidSVM is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU Affero General Public License for more details.

// You should have received a copy of the GNU Affero General Public License
// along with liquidSVM. If not, see <http://www.gnu.org/licenses/>.


#if !defined (COMMAND_LINE_PARSER_SVM_SELECT_CPP)
	#define COMMAND_LINE_PARSER_SVM_SELECT_CPP


#include "sources/svm/command_line/command_line_parser_svm_select.h"


#include "sources/shared/basic_functions/basic_file_functions.h"


#include <cstdlib>
#include <cstdio>


//**********************************************************************************************************************************


unsigned const ERROR_clp_svm_select_N = 60;
unsigned const ERROR_clp_svm_select_R = 61;
unsigned const ERROR_clp_svm_select_W = 62;


//**********************************************************************************************************************************


Tcommand_line_parser_svm_select::Tcommand_line_parser_svm_select()
{
	command_name = "svm-select";
};


//**********************************************************************************************************************************


void Tcommand_line_parser_svm_select::make_consistent()
{
	Tcommand_line_parser::make_consistent();
};


//**********************************************************************************************************************************


void Tcommand_line_parser_svm_select::display_help(unsigned error_code)
{
	Tcommand_line_parser::display_help(error_code);
	
	if (error_code == ERROR_clp_svm_select_R)
	{
		display_separator("-R <method>");
		flush_info(INFO_1, 
		"Selects the method that produces decision functions from the different folds.\n");
		display_specifics();
		flush_info(INFO_1, 
		"<method> = %d  =>   select for best average validation error\n"
		"<method> = %d  =>   on each fold select for best validation error\n", SELECT_ON_ENTIRE_TRAIN_SET, SELECT_ON_EACH_FOLD);
		display_ranges();
		flush_info(INFO_1, "<method>: integer between %d and %d\n", 0, SELECT_METHODS_MAX-1);
		display_defaults();
		flush_info(INFO_1, "<method> = %d\n", SELECT_ON_EACH_FOLD);
	}

	if (error_code == ERROR_clp_svm_select_N)
	{
		display_separator("-N <class> <constraint>");
		flush_info(INFO_1, 
		"Replaces the best validation error in the search for the best hyper-parameter\n"
		"combination by an NPL criterion, in which the best detection rate is searched\n"
		"for given the false alarm constraint <constraint> on class <class>.\n");
		display_ranges();
		flush_info(INFO_1, "<class>: -1 or 1\n");
		flush_info(INFO_1, "<constraint>: float between 0.0 and 1.0\n");
		display_defaults();
		flush_info(INFO_1, "Option is deactivated.\n");
	}
	
	if (error_code == ERROR_clp_svm_select_W)
	{
		display_separator("-W <number>");
		flush_info(INFO_1, 
		"Restrict the search for the best hyper-parameters to weights with the number\n"
		"<number>.\n");
		display_specifics();
		flush_info(INFO_1, "<number> = 0  =>   all weights are considered.\n");
		display_defaults();
		flush_info(INFO_1, "<number> = %d\n", 0);
	}
}


//**********************************************************************************************************************************


void Tcommand_line_parser_svm_select::exit_with_help()
{
	flush_info(INFO_SILENCE,
	"\n\nsvm-select [options] <trainfile> <logtrainfile> <logselectfile> <solfile> [<summary_log_file>]\n"
	"\nsvm-select reads the validation errors from <logtrainfile>, which was generated\n"
	"generated by svm-train. With the help of the samples in <trainfile> and the extra\n"
	"information stored in the .aux file generated by svm-train, svm-select then\n"
	"selects the SVMs that belong to the best hyper-parameters. The resulting\n"
	"decision functions are appended to <solfile> and some extra information is\n"
	"appended to <logselectfile>.\n"
	"\nAllowed extensions:\n"
	"<trainfile>:     .csv, .lsv, and .uci\n"
	"<logtrainfile>:  .log\n"
	"<logselectfile>: .log\n"
	"<solfile>:       .sol\n");

	if (full_help == false)
		flush_info(INFO_SILENCE, "\nOptions:");
	display_help(ERROR_clp_gen_d);
	display_help(ERROR_clp_gen_h);
	display_help(ERROR_clp_svm_select_N);
	display_help(ERROR_clp_svm_select_R);
	display_help(ERROR_clp_svm_select_W);

	flush_info(INFO_SILENCE,"\n\n");
	copyright();
	flush_exit(ERROR_SILENT, "");
}


//**********************************************************************************************************************************


void Tcommand_line_parser_svm_select::parse(bool read_filenames)
{
	check_parameter_list_size();
	for(current_position=1; current_position<parameter_list_size; current_position++)
		if (Tcommand_line_parser::parse("-d-h") == false)
		{
			if(parameter_list[current_position][0] != '-') 
				break;
			if (string(parameter_list[current_position]).size() > 2)
				Tcommand_line_parser::exit_with_help(ERROR_clp_gen_unknown_option);
			
			switch(parameter_list[current_position][1])
			{
				case 'N':
					select_control.npl = true;
					select_control.npl_class = get_next_class(ERROR_clp_svm_select_N);
					select_control.npl_constraint = get_next_number(ERROR_clp_svm_select_N, 0.0, 1.0);
					break;
				case 'R':
					select_control.select_method = get_next_enum(ERROR_clp_svm_select_R, 0, SELECT_METHODS_MAX-1);
					break;
				case 'W':
					select_control.weight_number = get_next_number(ERROR_clp_svm_select_W, 0);
					break;
				default:
					Tcommand_line_parser::exit_with_help(ERROR_clp_gen_unknown_option);
			}
		}
	
	// Read filenames

	if (read_filenames == true)
	{
		train_filename = get_next_labeled_data_filename(ERROR_clp_gen_missing_train_file_name);
		select_control.read_log_train_filename = get_next_log_filename(ERROR_clp_gen_missing_log_file_name);
		select_control.read_aux_train_filename = convert_log_to_aux_filename(select_control.read_log_train_filename);
		select_control.write_log_select_filename = get_next_log_filename(ERROR_clp_gen_missing_log_file_name);
		select_control.write_sol_select_filename = get_next_solution_filename();
		
		if (current_position < parameter_list_size)
			select_control.summary_log_filename = get_next_log_filename();
	}
		
	make_consistent();
};

#endif

