% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fcut.R
\name{fcut}
\alias{fcut}
\alias{fcut.default}
\alias{fcut.numeric}
\alias{fcut.matrix}
\alias{fcut.data.frame}
\alias{fcut.factor}
\alias{fcut.logical}
\title{Transform data into a \code{fsets} S3 class using shapes derived from
triangles or raised cosines}
\usage{
fcut(x, ...)

\method{fcut}{default}(x, ...)

\method{fcut}{factor}(x, name = deparse(substitute(x)), ...)

\method{fcut}{logical}(x, name = deparse(substitute(x)), ...)

\method{fcut}{numeric}(
  x,
  breaks,
  name = deparse(substitute(x)),
  type = c("triangle", "raisedcos"),
  merge = 1,
  parallel = FALSE,
  ...
)

\method{fcut}{data.frame}(
  x,
  breaks = NULL,
  name = NULL,
  type = c("triangle", "raisedcos"),
  merge = 1,
  parallel = FALSE,
  ...
)

\method{fcut}{matrix}(x, ...)
}
\arguments{
\item{x}{Data to be transformed: a vector, matrix, or data frame.
Non-numeric data are allowed.}

\item{...}{Other parameters to some methods.}

\item{name}{A name to be added as a suffix to the created fuzzy attribute
names. This parameter can be used only if \code{x} is a vector. If \code{x}
is a matrix or data frame, \code{name} should be NULL because the fuzzy
attribute names are taken from column names of the argument \code{x}.}

\item{breaks}{This argument determines the break-points of the positions of
the fuzzy sets. It should be an ordered vector of numbers such that the
\eqn{i}-th index specifies the beginning, \eqn{(i+1)}-th the center, and
\eqn{(i+2)}-th the ending of the \eqn{i}-th fuzzy set.

I.e. the minimum number of breaks-points is 3; \eqn{n-2} elementary fuzzy
sets would be created for \eqn{n} break-points.

If considering an i-th fuzzy set (of \code{type='triangle'}), \code{x}
values lower than \eqn{i}-th break (and greater than \eqn{(i+2)}-th break)
would result in zero memberhsip degree, values equal to \eqn{(i+1)}-th break
would have membership degree equal 1 and values between them the appropriate
membership degree between 0 and 1.

The resulting fuzzy sets would be named after the original data by adding
dot (".") and a number \eqn{i} of fuzzy set.

Unlike \code{\link[base:cut]{base::cut()}}, \code{x} values, that are lower or greater than
the given break-points, will have all memberhsip degrees equal to zero.

For non-numeric data, this argument is ignored. For \code{x} being a numeric
vector, it must be a vector of numeric values. For \code{x} being a numeric
matrix or data frame, it must be a named list containing a numeric vector
for each column - if not, the values are repeated for each column.}

\item{type}{The type of fuzzy sets to create. Currently, \code{'triangle'} or
\code{'raisedcos'} may be used. The \code{type} argument may be also a
function with 3 or 4 arguments:
\itemize{
\item if \code{type} is a 4-argument function, it is assumed that that it computes
membership degrees from values of the first argument while considering
the boundaries given by the next 3 arguments;
\item if \code{type} is a 3-argument function, it is assumed that it is a factory
function similar to \code{\link[=triangular]{triangular()}} or \code{\link[=raisedcosinal]{raisedcosinal()}}, which, from given
three boundaries, creates a function that computes membership degrees.
}}

\item{merge}{This argument determines whether to derive additional fuzzy
sets by merging the elementary fuzzy sets (whose position is determined with
the \code{breaks} argument) into super-sets.  The argument is ignored for
non-numeric data in \code{x}.

\code{merge} may contain any integer number from \code{1} to
\code{length(breaks) - 2}.  Value \code{1} means that the elementary fuzzy
sets should be present in the output.  Value \code{2} means that the two
consecutive elementary fuzzy sets should be combined by using the Lukasiewic
t-conorm, value \code{3} causes combining three consecutive elementary fuzzy
sets etc.

The names of the derived (merged) fuzzy sets is derived from the names of
the original elementary fuzzy sets by concatenating them with the "|" (pipe)
separator.}

\item{parallel}{Whether the processing should be run in parallel or not.
Parallelization is implemented using the \code{\link[foreach:foreach]{foreach::foreach()}} function.
The parallel environment must be set properly in advance, e.g. with
the \code{\link[doMC:registerDoMC]{doMC::registerDoMC()}} function.  Currently this argument is
applied only if \code{x} is a matrix or data frame.}
}
\value{
An object of class "fsets" is returned, which is a numeric matrix
with columns representing the fuzzy attributes. Each source column of the
\code{x} argument corresponds to multiple columns in the resulting matrix.
Columns have names that indicate the name of the source as well as a index
\eqn{i} of fuzzy set(s) -- see the description of arguments \code{breaks}
and \code{merge} above.

The resulting object would also have set the \code{\link[=vars]{vars()}} and
\code{\link[=specs]{specs()}} properties with the former being created from original
column names (if \code{x} is a matrix or data frame) or the \code{name}
argument (if \code{x} is a numeric vector). The \code{\link[=specs]{specs()}}
incidency matrix would be created to reflect the superset-hood of the merged
fuzzy sets.
}
\description{
This function creates a set of fuzzy attributes from crisp data. Factors,
numeric vectors, matrix or data frame columns are transformed into a set of
fuzzy attributes, i.e. columns with membership degrees. Unlike
\code{\link[=lcut]{lcut()}}, for transformation is not used the linguistic linguistic
approach, but partitioning using regular shapes of the fuzzy sets (such as
triangle, raised cosine).
}
\details{
The aim of this function is to transform numeric data into a set of fuzzy
attributes.  The result is in the form of the object of class "fsets", i.e.
a numeric matrix whose columns represent fuzzy sets (fuzzy attributes) with
values being the membership degrees.

The function behaves differently to the type of input \code{x}.

If \code{x} is a factor or a logical vector (or other non-numeric data) then
for each distinct value of an input, a fuzzy set is created, and data would
be transformed into crisp membership degrees 0 or 1 only.

If \code{x} is a numeric vector then fuzzy sets are created accordingly to
break-points specified in the \code{breaks} argument with 1st, 2nd and 3rd
break-point specifying the first fuzzy set, 2nd, 3rd and 4th break-point
specifying th second fuzzy set etc. The shape of the fuzzy set is determined
by the \code{type} argument that may be equal either to a string
\code{'triangle'} or \code{'raisedcos'} or it could be a function that
computes the membership degrees for itself (see \code{\link[=triangular]{triangular()}} or
\code{\link[=raisedcosinal]{raisedcosinal()}} functions for details). Additionally, super-sets of
these elementary sets may be created by specifying the \code{merge}
argument. Values of this argument specify how many consecutive fuzzy sets
should be combined (by using the Lukasiewic's t-conorm) to produce
super-sets - see the description of \code{merge} above.

If a matrix (resp. data frame) is provided to this function instead of
single vector, all columns are processed separately as described above and
the result is combined with the \code{\link[=cbind.fsets]{cbind.fsets()}} function.

The function sets up properly the \code{\link[=vars]{vars()}} and \code{\link[=specs]{specs()}}
properties of the result.
}
\examples{

# fcut on non-numeric data
ff <- factor(substring("statistics", 1:10, 1:10), levels = letters)
fcut(ff)

# transform a single vector into a single fuzzy set
x <- runif(10)
fcut(x, breaks=c(0, 0.5, 1), name='age')

# transform single vector into a partition of the interval 0-1
# (the boundary triangles are right-angled)
fcut(x, breaks=c(0, 0, 0.5, 1, 1), name='age')

# also create supersets
fcut(x, breaks=c(0, 0, 0.5, 1, 1), name='age', merge=c(1, 2))

# transform all columns of a data frame
# with different breakpoints
data <- CO2[, c('conc', 'uptake')]
fcut(data, breaks=list(conc=c(95, 95, 350, 1000, 1000),
                       uptake=c(7, 7, 28.3, 46, 46)))

# using a custom 3-argument function (a function factory):
f <- function(a, b, c) {
  return(function(x) ifelse(a <= x & x <= b, 1, 0))
}
fcut(x, breaks=c(0, 0.5, 1), name='age', type=f)

# using a custom 4-argument function:
f <- function(x, a, b, c) {
  return(ifelse(a <= x & x <= b, 1, 0))
}
fcut(x, breaks=c(0, 0.5, 1), name='age', type=f)

}
\seealso{
\code{\link[=lcut]{lcut()}}, \code{\link[=farules]{farules()}}, \code{\link[=pbld]{pbld()}}, \code{\link[=vars]{vars()}}, \code{\link[=specs]{specs()}}, \code{\link[=cbind.fsets]{cbind.fsets()}}
}
\author{
Michal Burda
}
\keyword{models}
\keyword{multivariate}
\keyword{robust}
