% Please edit documentation in R/leaflet.R
\name{leaflet}
\alias{leaflet}
\title{Create a Leaflet map widget}
\usage{
leaflet(data = NULL, width = NULL, height = NULL, padding = 0)
}
\arguments{
\item{data}{a data object (currently supported objects are matrices, data
frames, and spatial objects from the \pkg{sp} package of classes
\code{SpatialPoints}, \code{SpatialPointsDataFrame}, \code{Polygon},
\code{Polygons}, \code{SpatialPolygons}, \code{SpatialPolygonsDataFrame},
\code{Line}, \code{Lines}, \code{SpatialLines}, and
\code{SpatialLinesDataFrame})}

\item{width}{the width of the map}

\item{height}{the height of the map}

\item{padding}{the padding of the map}
}
\value{
A HTML widget object, on which we can add graphics layers using
  \code{\%>\%} (see examples).
}
\description{
This function creates a Leaflet map widget using \pkg{htmlwidgets}. The
widget can be rendered on HTML pages generated from R Markdown, Shiny, or
other applications.
}
\details{
The \code{data} argument is only needed if you are going to reference
variables in this object later in map layers. For example, \code{data} can be
a data frame containing columns \code{latitude} and \code{longtitude}, then
we may add a circle layer to the map by \code{leaflet(data) \%>\%
addCircles(lat = ~latitude, lng = ~longtitude)}, where the variables in the
formulae will be evaluated in the \code{data}.
}
\examples{
library(leaflet)
m = leaflet() \%>\% addTiles()
m  # a map with the default OSM tile layer

# set bounds
m \%>\% fitBounds(0, 40, 10, 50)

# move the center to Snedecor Hall
m = m \%>\% setView(-93.65, 42.0285, zoom = 17)
m

# popup
m \%>\% addPopups(-93.65, 42.0285, 'Here is the <b>Department of Statistics</b>, ISU')
rand_lng = function(n = 10) rnorm(n, -93.65, .01)
rand_lat = function(n = 10) rnorm(n, 42.0285, .01)

# use automatic bounds derived from lng/lat data
m = m \%>\% clearBounds()

# popup
m \%>\% addPopups(rand_lng(), rand_lat(), 'Random popups')

# marker
m \%>\% addMarkers(rand_lng(), rand_lat())
m \%>\% addMarkers(
  rand_lng(), rand_lat(), popup = paste('A random letter', sample(LETTERS, 10))
)

Rlogo = file.path(R.home('doc'), 'html', 'logo.jpg')
m \%>\% addMarkers(
  174.7690922, -36.8523071, icon = list(
    iconUrl = Rlogo, iconSize = c(100, 76)
  ), popup = 'R was born here!'
)

m \%>\% addMarkers(rnorm(30, 175), rnorm(30, -37), icon = list(
  iconUrl = Rlogo, iconSize = c(25, 19)
))

m \%>\% addMarkers(
  c(-71.0382679, -122.1217866), c(42.3489054, 47.6763144), icon = list(
    iconUrl = 'http://www.rstudio.com/wp-content/uploads/2014/03/blue-125.png'
  ), popup = c('RStudio @ Boston', 'RStudio @ Seattle')
)


# circle (units in metres)
m \%>\% addCircles(rand_lng(50), rand_lat(50), radius = runif(50, 50, 150))

# circle marker (units in pixels)
m \%>\% addCircleMarkers(rand_lng(50), rand_lat(50), color = '#ff0000')
m \%>\% addCircleMarkers(rand_lng(100), rand_lat(100), radius = runif(100, 5, 15))

# rectangle
m \%>\% addRectangles(
  rand_lng(), rand_lat(), rand_lng(), rand_lat(),
  color = 'red', fill = FALSE, dashArray = '5,5', weight = 3
)

# polyline
m \%>\% addPolylines(rand_lng(50), rand_lat(50))

# polygon
m \%>\% addPolygons(rand_lng(), rand_lat(), layerId = 'foo')

# geoJSON
seattle_geojson = list(
  type = "Feature",
  geometry = list(
    type = "MultiPolygon",
    coordinates = list(list(list(
      c(-122.36075812146,  47.6759920119894),
      c(-122.360781646764, 47.6668890126755),
      c(-122.360782108665,  47.6614990696722),
      c(-122.366199035722, 47.6614990696722),
      c(-122.366199035722,  47.6592874248973),
      c(-122.364582509469, 47.6576254522105),
      c(-122.363887331445,  47.6569107302038),
      c(-122.360865528129, 47.6538418253251),
      c(-122.360866157644,  47.6535254473167),
      c(-122.360866581103, 47.6533126275176),
      c(-122.362526540691,  47.6541872926348),
      c(-122.364442114483, 47.6551892850798),
      c(-122.366077719797,  47.6560733960606),
      c(-122.368818463838, 47.6579742346694),
      c(-122.370115159943,  47.6588730808334),
      c(-122.372295967029, 47.6604350102328),
      c(-122.37381369088,  47.660582362063),
      c(-122.375522972109, 47.6606413027949),
      c(-122.376079703095,  47.6608793094619),
      c(-122.376206315662, 47.6609242364243),
      c(-122.377610811371,  47.6606160735197),
      c(-122.379857378879, 47.6610306942278),
      c(-122.382454873022,  47.6627496239169),
      c(-122.385357955057, 47.6638573778241),
      c(-122.386007328104,  47.6640865692306),
      c(-122.387186331506, 47.6654326177161),
      c(-122.387802656231,  47.6661492860294),
      c(-122.388108244121, 47.6664548739202),
      c(-122.389177800763,  47.6663784774359),
      c(-122.390582858689, 47.6665072251861),
      c(-122.390793942299,  47.6659699214511),
      c(-122.391507906234, 47.6659200946229),
      c(-122.392883050767,  47.6664166747017),
      c(-122.392847210144, 47.6678696739431),
      c(-122.392904778401,  47.6709016021624),
      c(-122.39296705153, 47.6732047491624),
      c(-122.393000803496,  47.6759322346303),
      c(-122.37666945305, 47.6759896300663),
      c(-122.376486363943,  47.6759891899754),
      c(-122.366078869215, 47.6759641734893),
      c(-122.36075812146,  47.6759920119894)
    )))
  ),
  properties = list(
    name = "Ballard",
    population = 48000,
    # You can inline styles if you want
    style = list(
      fillColor = "yellow",
      weight = 2,
      color = "#000000"
    )
  ),
  id = "ballard"
)
m \%>\% setView(-122.36075812146, 47.6759920119894, zoom = 13) \%>\% addGeoJSON(seattle_geojson)


# use the Dark Matter layer from CartoDB
leaflet() \%>\% addTiles('http://{s}.basemaps.cartocdn.com/dark_all/{z}/{x}/{y}.png',
  attribution = paste(
    '&copy; <a href="http://openstreetmap.org">OpenStreetMap</a> contributors',
    '&copy; <a href="http://cartodb.com/attributions">CartoDB</a>'
  )
) \%>\% setView(-122.36, 47.67, zoom = 10)

# provide a data frame to leaflet()
categories = LETTERS[1:10]
df = data.frame(
  lat = rand_lat(100), lng = rand_lng(100), size = runif(100, 5, 20),
  category = factor(sample(categories, 100, replace = TRUE), levels = categories),
  value = rnorm(100)
)
m = leaflet(df) \%>\% addTiles()
m \%>\% addCircleMarkers(~lng, ~lat, radius = ~size)
m \%>\% addCircleMarkers(~lng, ~lat, radius = runif(100, 4, 10), color = c('red'))

# Discrete colors using the "RdYlBu" colorbrewer palette, mapped to categories
RdYlBu = colorFactor("RdYlBu", domain = categories)
m \%>\% addCircleMarkers(~lng, ~lat, radius = ~size,
  color = ~RdYlBu(category), fillOpacity = 0.5)

# Continuous colors using the "Greens" colorbrewer palette, mapped to value
greens = colorNumeric("Greens", domain = NULL)
m \%>\% addCircleMarkers(~lng, ~lat, radius = ~size,
  color = ~greens(value), fillOpacity = 0.5)
}
