% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model.R
\name{multi_gpu_model}
\alias{multi_gpu_model}
\title{Replicates a model on different GPUs.}
\usage{
multi_gpu_model(model, gpus = NULL, cpu_merge = TRUE, cpu_relocation = FALSE)
}
\arguments{
\item{model}{A Keras model instance. To avoid OOM errors,
this model could have been built on CPU, for instance
(see usage example below).}

\item{gpus}{\code{NULL} to use all available GPUs (default). Integer >= 2 or
list of integers, number of GPUs or list of GPU IDs on which to create
model replicas.}

\item{cpu_merge}{A boolean value to identify whether to force
merging model weights under the scope of the CPU or not.}

\item{cpu_relocation}{A boolean value to identify whether to
create the model's weights under the scope of the CPU.
If the model is not defined under any preceding device
scope, you can still rescue it by activating this option.}
}
\value{
A Keras model object which can be used just like the initial
\code{model} argument, but which distributes its workload on multiple GPUs.
}
\description{
Replicates a model on different GPUs.
}
\details{
Specifically, this function implements single-machine
multi-GPU data parallelism. It works in the following way:
\itemize{
\item Divide the model's input(s) into multiple sub-batches.
\item Apply a model copy on each sub-batch. Every model copy
is executed on a dedicated GPU.
\item Concatenate the results (on CPU) into one big batch.
}

E.g. if your \code{batch_size} is 64 and you use \code{gpus=2},
then we will divide the input into 2 sub-batches of 32 samples,
process each sub-batch on one GPU, then return the full
batch of 64 processed samples.

This induces quasi-linear speedup on up to 8 GPUs.

This function is only available with the TensorFlow backend
for the time being.
}
\note{
This function is deprecated and has been removed from tensorflow on
2020-04-01. To distribute your training across all available GPUS,
you can use \code{tensorflow::tf$distribute$MirroredStrategy()}
by creating your model like this:\if{html}{\out{<div class="r">}}\preformatted{strategy <- tensorflow::tf$distribute$MirroredStrategy()
with(strategy$scope(), \{
  model <- application_xception(
    weights = NULL,
    input_shape = c(height, width, 3),
    classes = num_classes
\})
}\if{html}{\out{</div>}}
}
\section{Model Saving}{


To save the multi-gpu model, use \code{\link[=save_model_hdf5]{save_model_hdf5()}} or
\code{\link[=save_model_weights_hdf5]{save_model_weights_hdf5()}} with the template model (the argument you
passed to \code{multi_gpu_model}), rather than the model returned
by \code{multi_gpu_model}.
}

\examples{
\dontrun{

library(keras)
library(tensorflow)

num_samples <- 1000
height <- 224
width <- 224
num_classes <- 1000

# Instantiate the base model (or "template" model).
# We recommend doing this with under a CPU device scope,
# so that the model's weights are hosted on CPU memory.
# Otherwise they may end up hosted on a GPU, which would
# complicate weight sharing.
with(tf$device("/cpu:0"), {
  model <- application_xception(
    weights = NULL,
    input_shape = c(height, width, 3),
    classes = num_classes
  )
})

# Replicates the model on 8 GPUs.
# This assumes that your machine has 8 available GPUs.
parallel_model <- multi_gpu_model(model, gpus = 8)
parallel_model \%>\% compile(
  loss = "categorical_crossentropy",
  optimizer = "rmsprop"
)

# Generate dummy data.
x <- array(runif(num_samples * height * width*3),
           dim = c(num_samples, height, width, 3))
y <- array(runif(num_samples * num_classes),
           dim = c(num_samples, num_classes))

# This `fit` call will be distributed on 8 GPUs.
# Since the batch size is 256, each GPU will process 32 samples.
parallel_model \%>\% fit(x, y, epochs = 20, batch_size = 256)

# Save model via the template model (which shares the same weights):
model \%>\% save_model_hdf5("my_model.h5")
}

}
\seealso{
Other model functions: 
\code{\link{compile.keras.engine.training.Model}()},
\code{\link{evaluate.keras.engine.training.Model}()},
\code{\link{evaluate_generator}()},
\code{\link{fit.keras.engine.training.Model}()},
\code{\link{fit_generator}()},
\code{\link{get_config}()},
\code{\link{get_layer}()},
\code{\link{keras_model_sequential}()},
\code{\link{keras_model}()},
\code{\link{pop_layer}()},
\code{\link{predict.keras.engine.training.Model}()},
\code{\link{predict_generator}()},
\code{\link{predict_on_batch}()},
\code{\link{predict_proba}()},
\code{\link{summary.keras.engine.training.Model}()},
\code{\link{train_on_batch}()}
}
\concept{model functions}
