#' Creates a colored surface plot.
#'
#' @export
#' @import stats
#' @import grDevices
#' @import graphics
#' @description This function is a wrapper around \code{\link[graphics]{image}}
#' and \code{\link[graphics]{contour}}. See \code{vignette("plotfunctions")} 
#' for an example of how you could use \code{\link[graphics]{image}} and 
#' \code{\link[graphics]{contour}}.
#'
#' @param data Data frame or list with plot data. A data frame needs to have a 
#' column with x values, a column with y values and a column with z values. A 
#' list contains a vector with unique x values, a vector with unique y values, 
#' and a matrix with z-values. The output of the function fvisgam is an 
#' example of a suitable list. 
#' @param view A vector with the names or numbers of the columns to plot on 
#' the x axis and y axis respectively.
#' @param predictor Optional: the name of the column in the data frame 
#' \code{data} that provides the z-values. If data contains more than one 
#' column besides the x- and y-values, the \code{predictor} should be provided.
#' @param valCI Optional: the name of the column in the data frame 
#' \code{data} that provides the CI-values. If not NULL, CI contour lines
#' will be plotted.
#' @param main Text string, an overall title for the plot.
#' @param xlab Label for x axis. Default is name of first \code{view} variable.
#' @param ylab Label for y axis. Default is name of second \code{view} 
#' variable.
#' @param xlim x-limits for the plot.
#' @param ylim y-limits for the plot.
#' @param zlim z-limits for the plot.
#' @param col Color for the  contour lines and labels.
#' @param color a list of colors such as that generated by 
#' \code{\link[grDevices]{rainbow}}, \code{\link[grDevices]{heat.colors}}
#' \code{\link[grDevices]{colors}}, \code{\link[grDevices]{topo.colors}}, 
#' \code{\link[grDevices]{terrain.colors}} or similar functions.
#' @param ci.col Two-value vector with colors for the lower CI contour lines 
#' and for the upper CI contour lines.
#' @param nCol The number of colors to use in color schemes.
#' @param add.color.legend Logical: whether or not to add a color legend. 
#' Default is TRUE. If FALSE (omitted), one could use the function
#' \code{\link{gradientLegend}} to add a legend manually at any position.
#' @param ... Optional parameters for \code{\link[graphics]{image}}
#' and \code{\link[graphics]{contour}}.
#' @author Jacolien van Rij
#' @seealso \code{\link[graphics]{image}}, \code{\link[graphics]{contour}}, 
#' \code{\link{color_contour}}
#' @examples
#'
#' data(simdat)
#'
#' \dontrun{
#' # Model with interaction:
#' m1 <- bam(Y ~ s(Time) + s(Trial)
#' +ti(Time, Trial),
#' data=simdat)
#' 
#' # get partial prediction of the ti-term:
#' pp <- get_modelterm(m1, select=3, as.data.frame=TRUE)
#' head(pp)
#'
#' # plot surface:
#' plotsurface(pp, view=c('Time', "Trial"), predictor='fit')
#' # ...is the same as:
#' pvisgam(m1,view=c('Time', "Trial"), select=3)
#'
#' # add main effects of Time and Trial:
#' pp1  <- get_modelterm(m1, select=1, as.data.frame=TRUE)
#' pp2  <- get_modelterm(m1, select=2, as.data.frame=TRUE)
#' pp$fit.sum <- pp$fit + rep(pp1$fit, 30) + rep(pp2$fit, each=30)
#'
#' plotsurface(pp, view=c('Time', "Trial"), predictor='fit.sum')
#' # ...is not same as fvisgam, because in previous plot the intercept 
#' # is not included:
#' fvisgam(m1, view=c('Time', "Trial"))
#'
#' # This is same as fvisgam:
#' pp <- get_predictions(m1, cond=list(Time=seq(0,2000, length=30),
#' 	Trial=seq(-10,10,length=30)))
#' plotsurface(pp, view=c('Time', "Trial"), predictor='fit', valCI='CI')
#'
#' # Just contour plot:
#' plotsurface(pp, view=c('Time', "Trial"), predictor='fit', valCI='CI',
#' 	main='contour',	color=NULL, col=1, add.color.legend=FALSE)
#'}
#'
#' @family functions for model predictions

plotsurface <- function(data, view, predictor=NULL, valCI=NULL,
	main=NULL, xlab=NULL, ylab=NULL, 
	xlim=NULL, ylim=NULL, zlim=NULL,
	col=NULL, color=topo.colors(50), ci.col =c('red','green'), nCol=50,
	add.color.legend=TRUE, ...){

	xval <- c()
	yval <- c()
	zval <- c()
	cival.l <- NULL
	cival.u <- NULL

	# check input:
	# 1. check data:
	if(is.null(data)){
		if(is.list(data)){
			# 2a. check view
			if(is.numeric(view)){
				if(view[1] <= length(data)){
					xval <- data[[view[1]]]
				}else{
					stop(sprintf("First view element incorrect: data has only %d elements.", length(data)))
				}
				if(view[2] <= length(data)){
					yval <- data[[view[2]]]
				}else{
					stop(sprintf("Second view element incorrect: data has only %d elements.", length(data)))
				}
			}else{
				cn <- names(data)
				if(view[1] %in% cn){
					xval <- data[[view[1]]]
				}else{
					stop(sprintf("%s not available in data.", view[1]))
				}
				if(view[2] %in% cn){
					yval <- data[[view[2]]]
				}else{
					stop(sprintf("%s not available in data.", view[2]))
				}
			}
			# 3a. check predictor
			if(is.null(predictor)){
				if(length(data)==3){
					cn <- 1:3
					if(!is.numeric(view)){
						cn <- names(data)
					}
					zval <- data[[ cn[!cn %in% view] ]]
				}else{
					stop(sprintf("Not sure which element of %s should be plotted. Provide predictor.", deparse(substitute(data))))
				}
			}else{
				if(is.numeric(predictor)){
					if(length(data) >= predictor){
						zval <- data[[predictor]]
					}else{
						stop(sprintf("Value of predictor incorrect: data has only %d elements.", length(data)))
					}
				}else{
					cn <- names(data)
					if(predictor %in% cn){
						zval <- data[[predictor]]
					}else{
						stop(sprintf("%s not available in data.", predictor))
					}
				}
			}

			if(!is.matrix(zval)){
				stop('z-values should be provided as matrix. Alternatively, provide data frame with x values, y values, and z values (and optionally CI values). See examples.')
			}


			# 4a. check CI
			if(!is.null(valCI)){
				if(is.numeric(valCI)){
					if(length(data) >= valCI[1]){
						cival.l <- cival.u <- data[[valCI[1]]]
					}else{
						stop(sprintf("Value of valCI incorrect: data has only %d elements.", length(data)))
					}
					if(length(valCI)>1){
						valCI <- valCI[1:2]
						if(length(data) >= valCI[2]){
							cival.u <- data[[valCI[2]]]
						}else{
							warning(sprintf("Value of second valCI incorrect: data has only %d elements. First valCI is also used for upper limit.", length(data)))
						}
					}
				}else{
					cn <- names(data)
					if(valCI[1] %in% cn){
						cival.l <- cival.u <- data[[valCI[1]]]
					}else{
						stop(sprintf("%s not available in data.", predictor))
					}
					if(length(valCI)>1){
						valCI <- valCI[1:2]
						if(valCI[2] %in% cn){
							cival.u <- data[[valCI[2]]]
						}else{
							warning(sprintf("Value of second valCI incorrect: %s not available in data. First valCI is also used for upper limit.", valCI[2]))
						}
					}

				}
			}
		}else{
			stop('Data is not a list or data frame.')
		}
	}else{
		# 2b. check view
		if(is.numeric(view)){
			if(view[1] <= ncol(data)){
				xval <- sort(unique( data[,view[1]] ))
			}else{
				stop(sprintf("First view element incorrect: data has only %d columns.", ncol(data)))
			}
			if(view[2] <= ncol(data)){
				yval <- sort(unique( data[,view[2]] ))
			}else{
				stop(sprintf("Second view element incorrect: data has only %d columns.", ncol(data)))
			}
		}else{
			cn <- colnames(data)
			if(view[1] %in% cn){
				xval <- sort(unique( data[,view[1]] ))
			}else{
				stop(sprintf("%s not available in data.", view[1]))
			}
			if(view[2] %in% cn){
				yval <- sort(unique( data[,view[2]] ))
			}else{
				stop(sprintf("%s not available in data.", view[2]))
			}
		}
		# 3b. check predictor
		if(is.null(predictor)){
			if(ncol(data)==3){
				cn <- 1:3
				if(!is.numeric(view)){
					cn <- names(data)
				}
				predictor <- cn[!cn %in% view]
			}else if("fit" %in% colnames(data)){
				predictor <- "fit"
			} else {
				stop("Not sure which element of data should be plotted. Provide predictor.")
			}
		}else{
			if(is.numeric(predictor)){
				if(ncol(data) < predictor){
					stop(sprintf("Value of predictor incorrect: data has only %d columns.", ncol(data)))
				}
			}else{
				cn <- colnames(data)
				if(!predictor %in% cn){
					stop(sprintf("%s not available in data.", predictor))
				}
			}
		}

		# sort data:
		data <- data[order(data[,view[1]], data[,view[2]]),]
		zval <- matrix(data[, predictor], byrow=TRUE, 
			nrow=length(xval),ncol=length(yval))

		# 4b. check valCI
		if(!is.null(valCI)){
			if(is.numeric(valCI)){
				if(ncol(data) < valCI[1]){
					stop(sprintf("Value of valCI incorrect: data has only %d columns.", ncol(data)))
				}
				if(length(valCI)>1){
					valCI <- valCI[1:2]
					if(ncol(data) < valCI[2]){
						valCI <- valCI[1]
						warning(sprintf("Value of second valCI incorrect: data has only %d columns. First valCI is also used for upper limit.", ncol(data)))
					}
				}
			}else{
				cn <- colnames(data)
				if(!valCI[1] %in% cn){
					stop(sprintf("%s not available in data.", predictor))
				}
				if(length(valCI)>1){
					valCI <- valCI[1:2]
					if(!valCI[2] %in% cn){
						warning(sprintf("Value of second valCI incorrect: %s not available in data. First valCI is also used for upper limit.", valCI[2]))
						valCI <- valCI[1]
					}
				}
			}
			cival.l <- cival.u <- matrix(data[, valCI[1]], byrow=TRUE, 
				nrow=length(xval),ncol=length(yval))
			if(length(valCI)>1){
				cival.u <- matrix(data[, valCI[2]], byrow=TRUE, 
					nrow=length(xval),ncol=length(yval))
			}
		}
	}



	## Check plot settings
	if(is.null(main)){
		if(is.null(predictor)){
			main=""
		}else{
			main=predictor
		}
	}
	if(is.null(xlab)){
		xlab=view[1]
	}
	if(is.null(ylab)){
		ylab=view[2]
	}
	if(is.null(xlim)){
		xlim=range(xval)
	}
	if(is.null(ylim)){
		ylim=range(yval)
	}	
	if(is.null(zlim)){
		zlim=range(zval)
	}	

	# colors:
    if(is.null(color)){
    	color <- alphaPalette('white', f.seq=c(0,0), n=nCol)
    }
    if (color[1] == "heat") {
        color <- heat.colors(nCol)
        if(is.null(col)){
        	col <- 3
        }
    } else if (color[1] == "topo") {
        color <- topo.colors(nCol)
        if(is.null(col)){
        	col <- 2
        }
    } else if (color[1] == "cm") {
        color <- cm.colors(nCol)
        if(is.null(col)){
        	col <- 1
        }
    } else if (color[1] == "terrain") {
        color <- terrain.colors(nCol)
        if(is.null(col)){
        	col <- 2
        }
    } else if (color[1] == "bpy") {
        if (requireNamespace("sp", quietly = TRUE)) {
            color <- sp::bpy.colors(nCol)
            if(is.null(col)){
        		col <- 3
        	}
        } else {
            warning("Package 'sp' needed for bpy color palette. Using topo.colors instead (default).")
            color <- topo.colors(nCol)
            col <- 2
        }
    } else if (color[1] == "gray" || color[1] == "bw") {
        color <- gray(seq(0.1, 0.9, length = nCol))
        col <- 1
    } 
    if (is.null(col)){
    	col <- 'red'
    } 


	dnm <- list(...)
	parlist <- names(dnm)

	type2string <- function(x){
		out <- ""
		if(length(x)>1){
			if(is.character(x)){
				out <- sprintf("c(%s)", paste(sprintf("'%s'", x), collapse=','))
			}else{
				out <- sprintf("c(%s)", paste(x, collapse=','))
			}
		}else{
			if(is.character(x)){
				out <- sprintf("'%s'", x)
			}else{
				out <- sprintf("%s", x)
			}
		}
		return(out)
	}

	# check contour input:
	cpar <- c()
	contourarg <- c('nlevels', 'levels', 'labels', 'labcex', 'drawlabels', 'method', 'lty', 'lwd')
	for(i in parlist[parlist %in% contourarg] ){
		cpar <- c(cpar, sprintf("%s=%s", i, type2string(dnm[[i]])))
	}
	cpar <- paste(",", paste(cpar, collapse=','))

	cpar2 <- c()
	for(i in parlist[parlist %in% c('nlevels', 'levels', 'method')] ){
		cpar2 <- c(cpar2, sprintf("%s=%s", i, type2string(dnm[[i]])))
	}
	cpar2 <- paste(",", paste(cpar2, collapse=','))

	# check image input:
	ipar <- c()
	contourarg <- c('nlevels', 'levels', 'labels', 'labcex', 'drawlabels', 'method', 'lty', 'lwd')

	for(i in parlist[!parlist %in% contourarg] ){
		ipar <- c(ipar, sprintf("%s=%s", i, type2string(dnm[[i]])))
	}

	ipar <- paste(",", paste(ipar, collapse=','))

	eval(parse(text=sprintf("image(xval, yval, zval, col=color, xlim=xlim, ylim=ylim, zlim=zlim, main=main, xlab=xlab, ylab=ylab, add=FALSE%s)", ipar)))
	eval(parse(text=sprintf("contour(xval, yval, zval, col=col, add=TRUE%s)",
		cpar)))

	if(!is.null(valCI)){
		eval(parse(text=sprintf("contour(xval, yval, zval-cival.l, col=ci.col[1], add=TRUE, lty=3, drawlabels=FALSE%s)",
			cpar2)))
		eval(parse(text=sprintf("contour(xval, yval, zval+cival.u, col=ci.col[2], add=TRUE, lty=3, drawlabels=FALSE%s)",
			cpar2)))		
	}

    if(add.color.legend){
        gradientLegend(round(zlim, 3), n.seg=3, pos=.875, 
            color=color)
    }

	invisible(list(x=xval, y=yval, z=zval, ci.l = cival.l, ci.u = cival.u))
}