#' Generate an ACF plot of model residuals. Works for lm, lmer, gam, bam, ....
#' 
#' @description Wrapper around \code{\link{acf_plot}} and 
#' \code{\link{acf_n_plots}} for regression models.
#' @export
#' @import mgcv
#' @import stats
#' @param model A regression model generated by \code{lm}, \code{glm},
#' \code{lmer}, \code{glmer}, \code{\link[mgcv]{gam}}, 
#' or \code{\link[mgcv]{bam}}.
#' (See examples for how to avoid errors due to missing values.)
#' @param split_pred Vector with names of model predictors that determine
#' the time series in the data, or should be used to split the ACF plot by.
#' Alternatively, \code{split_pred} can be a names list as being used by 
#' \code{\link{acf_plot}} and \code{\link{acf_n_plots}}.
#' @param n The number of plots to generate. If \code{n}=1 (default) then 
#' \code{\link{acf_plot}} is being called. If \code{n}>1 then 
#' \code{\link{acf_n_plots}} is being called.
#' @param plot Logical: whether or not to produce plot. Default is TRUE.
#' @param ... Other arguments as input for \code{\link{acf_plot}} 
#' or \code{\link{acf_n_plots}}.
#' @return An aggregated ACF plot and / or optionally a list with the aggregated ACF values.
#' @author Jacolien van Rij
#' @seealso Use \code{\link[stats]{acf}} for the original ACF function, 
#' and \code{\link{acf_plot}}, or \code{\link{acf_n_plots}}.
#' @examples
#' data(simdat)
#'
#' # add missing values to simdat:
#' simdat[sample(nrow(simdat), 15),]$Y <- NA
#' 
#' \dontrun{
#' # Run GAMM model:
#' m1 <- bam(Y ~ te(Time, Trial)+s(Subject, bs='re'), data=simdat)
#'
#' # Using a list to split the data:
#' acf_resid(m1, split_pred=list(simdat$Subject, simdat$Trial))
#' # ...or using model predictors:
#' acf_resid(m1, split_pred=c("Subject", "Trial"))
#' 
#' # Calling acf_n_plots:
#' acf_resid(m1, split_pred=c("Subject", "Trial"), n=4)
#' # add some arguments:
#' acf_resid(m1, split_pred=c("Subject", "Trial"), n=4, max_lag=10)
#' 
#' # This does not work...
#' m2 <- lm(Y ~ Time, data=simdat)
#' acf_resid(m2, split_pred=c("Subject", "Trial"))
#' # ... but this is ok:
#' acf_resid(m2, split_pred=list(simdat$Subject, simdat$Trial))
#' 
#' # See acf_plot for how to deal wit missing data
#' }
#' # see the vignette for examples:
#' vignette("acf", package="itsadug")
#' @family functions for model criticism


acf_resid <- function(model, split_pred=NULL, n=1, plot=TRUE, ...){

	split_by=NULL
	res <- NULL

	if(!is.null(split_pred)){
		
		split_by=list()

		if(!is.list(split_pred)){
			# extract time series data from model:
			dat <- NULL

			if("lm" %in% class(model)){
				dat <- model$model

			}else if( "lmerMod" %in% class(model)){
				dat <- model@frame
			}

			if(!all(split_pred %in% colnames(dat))){
				notindata <- paste(split_pred[!split_pred %in% colnames(dat)], collapse=", ")
				stop(sprintf("split-pred value(s) %s is / are not included as predictor in the model.", 
					notindata))
			}else{
				for(i in split_pred){
					split_by[[i]] <- as.vector(dat[,i])
				}
			}
		}else{
			split_by <- split_pred
		}
	}

	if(n > 1){
		if(!all(is.na(resid_gam(model, incl_na=TRUE)))){
			if(is.null(split_pred)){
				out <- acf_n_plots(resid_gam(model), split_by=split_by, n=n, plot=plot, ...)
				invisible(out)
			}else if( !is.list(split_pred)){
				out <- acf_n_plots(resid_gam(model), split_by=split_by, n=n, plot=plot, ...)
				invisible(out)
			}else{
				out <- acf_n_plots(resid_gam(model, incl_na=TRUE), split_by=split_by, n=n, plot=plot, ...)
				invisible(out)
			}
		}
	}else{
		if(!all(is.na(resid_gam(model, incl_na=TRUE)))){
			if(is.null(split_pred)){
				out <- acf_plot(resid_gam(model), split_by=split_by, plot=plot, ...)
				invisible(out)
			}else if( !is.list(split_pred)){
				out <- acf_plot(resid_gam(model), split_by=split_by, plot=plot, ...)
				invisible(out)
			}else{
				out <- acf_plot(resid_gam(model, incl_na=TRUE), split_by=split_by, plot=plot, ...)
				invisible(out)
			}
		}
	}

}