% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sx2_fit.R
\name{sx2_fit}
\alias{sx2_fit}
\alias{sx2_fit.default}
\alias{sx2_fit.est_item}
\alias{sx2_fit.est_irt}
\title{S-X2 fit statistic}
\usage{
sx2_fit(x, ...)

\method{sx2_fit}{default}(
  x,
  data,
  D = 1,
  alpha = 0.05,
  min.collapse = 1,
  norm.prior = c(0, 1),
  nquad = 30,
  weights,
  pcm.loc = NULL,
  ...
)

\method{sx2_fit}{est_item}(
  x,
  alpha = 0.05,
  min.collapse = 1,
  norm.prior = c(0, 1),
  nquad = 30,
  weights,
  pcm.loc = NULL,
  ...
)

\method{sx2_fit}{est_irt}(
  x,
  alpha = 0.05,
  min.collapse = 1,
  norm.prior = c(0, 1),
  nquad = 30,
  weights,
  pcm.loc = NULL,
  ...
)
}
\arguments{
\item{x}{A data frame containing the item metadata (e.g., item parameters, number of categories, models ...), an object
of class \code{\link{est_item}} obtained from the function \code{\link{est_item}}, or an object of class \code{\link{est_irt}}
obtained from the function \code{\link{est_irt}}. See \code{\link{irtfit}}, \code{\link{info}}, or \code{\link{simdat}}
for more details about the item metadata. The data frame of item metadata can be easily obtained using the function \code{\link{shape_df}}.}

\item{...}{Further arguments passed to or from other methods.}

\item{data}{A matrix containing examinees' response data for the items in the argument \code{x}. A row and column indicate
the examinees and items, respectively.}

\item{D}{A scaling factor in IRT models to make the logistic function as close as possible to the normal ogive function (if set to 1.7).
Default is 1.}

\item{alpha}{A numeric value to specify significance \eqn{\alpha}-level of the hypothesis test for \eqn{S-X^{2}} fit statistic. Default is .05.}

\item{min.collapse}{An integer value to indicate the minimum frequency of cells to be collapsed. Default is 1. See below for details.}

\item{norm.prior}{A numeric vector of two components specifying a mean and standard deviation of the normal prior distribution.
These two parameters are used to obtain the gaussian quadrature points and the corresponding weights from the normal distribution. Default is
c(0,1).}

\item{nquad}{An integer value specifying the number of gaussian quadrature points from the normal prior distribution. Default is 30.}

\item{weights}{A two-column matrix or data frame containing the quadrature points (in the first column) and the corresponding weights
(in the second column) of the latent variable prior distribution. The weights and quadrature points can be easily obtained
using the function \code{\link{gen.weight}}. If missing, default values are used (see the arguments of \code{norm.prior} and \code{nquad}).}

\item{pcm.loc}{A vector of integer values indicating the locations of partial credit model (PCM) items whose slope parameters are fixed
to certain values. Default is NULL.}
}
\value{
This function returns a list. Within a list, several internal objects are contained such as:
\item{fit_stat}{A data frame containing the results of \eqn{S-X^{2}} fit statistics for all items.}
\item{item_df}{The item metadata specified in the argument \code{x}.}
\item{exp_freq}{A list containing the collapsed expected frequency tables for all items.}
\item{obs_freq}{A list containing the collapsed observed frequency tables for all items.}
\item{exp_prob}{A list containing the collapsed expected probability tables for all items.}
\item{obs_prop}{A list containing the collapsed observed proportion tables for all items.}
}
\description{
This function computes \eqn{S-X^{2}} (Orlando & Thissen, 2000, 2003) item fit statistic.
}
\details{
Often, very small expected frequencies in the contingency tables used to compute \eqn{\chi^{2}} fit statistics could
compromise the accuracy of the \eqn{\chi^{2}} approximation for their distribution (Orlando & Thissen, 2000).
To avoid this problem, Orlando and Thissen (2000) used an algorithm of collapsing adjacent test score groups to maintain
a minimum expected category frequency of 1. However, if Orlando and Thissen's cell collapsing approach is applied to polytomous data,
too much information would be lost (Kang & Chen, 2008). Thus, Kang and Chen (2008) collapsed adjacent cells of item score categories
for a specific score group to ensure a minimum expected category frequency of 1. The same collapsing strategies were applied
in the function \code{\link{sx2_fit}}. If a minimum expected category frequency needs to be set to different number, you can specify
the minimum value in the argument \code{min.collapse}.

Note that if "DRM" is specified for an item in the item metadata set, the item is considered as "3PLM" to compute degree of freedom of
the \eqn{S-X^{2}} fit statistic.

Also, any missing responses in \code{data} are replaced with incorrect responses (i.e., 0s).
}
\section{Methods (by class)}{
\itemize{
\item \code{sx2_fit(default)}: Default method to compute \eqn{S-X^{2}} fit statistics for a data frame \code{x} containing the item metadata.

\item \code{sx2_fit(est_item)}: An object created by the function \code{\link{est_item}}.

\item \code{sx2_fit(est_irt)}: An object created by the function \code{\link{est_irt}}.

}}
\examples{

## example 1: all five polytomous IRT models are GRM
## import the "-prm.txt" output file from flexMIRT
flex_sam <- system.file("extdata", "flexmirt_sample-prm.txt", package = "irtQ")

# select the item metadata
x <- bring.flexmirt(file = flex_sam, "par")$Group1$full_df

# generate examinees' abilities from N(0, 1)
set.seed(23)
score <- rnorm(500, mean = 0, sd = 1)

# simulate the response data
data <- simdat(x = x, theta = score, D = 1)

\donttest{
# compute fit statistics
fit1 <- sx2_fit(x = x, data = data, nquad = 30)

# fit statistics
fit1$fit_stat
}

## example 2: first 39th and 40th items follows GRM and 53rd, 54th, and 55th items
##            follow PCM (thus, the slope parameters are fixed to 1)
# replace the model names with GPCM and
# assign 1 to the slope parameters for the 53rd, 54th, and 55th items
x[53:55, 3] <- "GPCM"
x[53:55, 4] <- 1

# generate examinees' abilities from N(0, 1)
set.seed(25)
score <- rnorm(1000, mean = 0, sd = 1)

# simulate the response data
data <- simdat(x = x, theta = score, D = 1)

\donttest{
# compute fit statistics
fit2 <- sx2_fit(x = x, data = data, nquad = 30, pcm.loc = 53:55)

# fit statistics
fit2$fit_stat
}

}
\references{
Kang, T., & Chen, T. T. (2008). Performance of the generalized S-X2 item fit index for polytomous IRT models.
\emph{Journal of Educational Measurement, 45}(4), 391-406.

Orlando, M., & Thissen, D. (2000). Likelihood-based item-fit indices for dichotomous item response theory models.
\emph{Applied Psychological Measurement, 24}(1), 50-64.

Orlando, M., & Thissen, D. (2003). Further investigation of the performance of S-X2: An item fit index for use with
dichotomous item response theory models. \emph{Applied Psychological Measurement, 27}(4), 289-298.
}
\seealso{
\code{\link{irtfit}}, \code{\link{info}}, \code{\link{simdat}}, \code{\link{shape_df}}, \code{\link{est_item}}
}
\author{
Hwanggyu Lim \email{hglim83@gmail.com}
}
