calculate_rpd <- function(x)
{
  x <- as.matrix(x)
  x[is.infinite(x)] <- NA # Remove infinities
  min_cols <- matrixStats::rowMins(x, na.rm = TRUE)
  100 * (x - min_cols) / min_cols
}

orca_pdf <- function(filename, plot)
{
  # The filename value is worked to separate it and assign it to new values.
  nameFile <- basename(filename)
  nameFile <- maybe_add_file_extension(nameFile, "pdf")
  directory <- paste0(dirname(filename), sep = "/")
  withr::with_dir(directory, plotly::orca(plot, nameFile))
}


iraceplot_warn <- function(...)
  cli_alert_warning(text = paste0("{.strong Warning:} ", ...))


orca_save_plot <- function(plot_list, filename)
{
  if (!is.null(filename)) {
    directory <- paste0(dirname(filename), sep = "/")
    if (length(plot_list) == 1) {
      plotly::orca(plot_list[[1]], irace::path_rel2abs(filename))
    } else {
      base_name <- strsplit(basename(filename),split = '[.]')[[1]][1]
      ext <- strsplit(basename(filename),split = '[.]')[[1]][2]
      for (i in seq_along(plot_list)) {
        part <- paste0("-", i)
        cfile <- irace::path_rel2abs(paste0(directory, "/", base_name, part,"." , ext))
        plotly::orca(plot_list[[i]], cfile)
      }
    }
  }
}

maybe_add_file_extension <- function(filename, ext)
{
  if (startsWith(ext, ".")) ext <- substring(ext, 2L)
  if (!has_file_extension(filename, ext)) filename <- paste0(filename, ".", ext)
  filename
}

has_file_extension <- function(filename, ext)
{
  if (startsWith(ext, ".")) ext <- substring(ext, 2L)
  grepl(paste0('[.]', ext, '$'), filename, ignore.case = TRUE)
}

#' Check if the results object generated by irace has data about the testing phase.
#'
#' @template arg_irace_results
#'
#' @return `logical(1)`
#' @export
# FIXME: Make it a reexport with irace >= 3.5.1
has_testing_data <- function(irace_results)
{
  ins <- irace_results$scenario$testInstances
  exp <- irace_results$testing$experiments
  !(length(ins) == 0 ||
    (length(ins) == 1 && (is.na(ins) || nchar(ins) == 0)) ||
    length(exp) == 0 || !(is.matrix(exp) || is.data.frame(exp)))
}

check_unknown_param_names <- function(x, parameters_names)
{
  x <- unlist(x)
  if (any(!(x %in% parameters_names)))
    stop("Unknown parameter names: ", paste0(setdiff(x, parameters_names), collapse=", "))
  x
}

subset_param_names <- function(x, parameters_names, is_fixed)
{
  if (is.null(x)) return(parameters_names[!is_fixed])
  check_unknown_param_names(x, parameters_names)
}


