% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils-surf2surf.R
\name{surface_to_surface}
\alias{surface_to_surface}
\title{Transform surface between coordinate spaces}
\usage{
surface_to_surface(surface, space_from = "", space_to = "", transform = NULL)
}
\arguments{
\item{surface}{an \code{ieegio_surface} object or file path; see
\code{\link{as_ieegio_surface}} for valid inputs}

\item{space_from}{source coordinate space; either an \code{ieegio_space}
object (from \code{\link{new_space}}) or a character string; default
is empty string}

\item{space_to}{target coordinate space; either an \code{ieegio_space}
object or a character string; default is empty string}

\item{transform}{optional 4x4 affine transformation matrix or
\code{ieegio_transforms} object to apply; see
\code{\link{as_ieegio_transform}}}
}
\value{
A transformed \code{ieegio_surface} object with updated vertex
positions and transform metadata
}
\description{
Transforms surface vertex positions from one coordinate space or orientation
to another, optionally applying an additional custom transform.
}
\details{
The function handles orientation changes (e.g., \code{"RAS"} to \code{"LPS"})
and optional custom transforms. It creates a transform chain consisting of:
an affine (orientation alignment from source), the custom transform, and
a post-affine (final orientation alignment to target).

If the provided transform has a \code{"passive"} interpretation, it is
automatically converted to an \code{"active"} interpretation before
being applied to the vertex coordinates.
}
\examples{

library(ieegio)

# geometry
geom_file <- "gifti/GzipBase64/sujet01_Lwhite.surf.gii"

if(ieegio_sample_data(geom_file, test = TRUE)) {

surf_ras <- read_surface(ieegio_sample_data(geom_file))
plot(surf_ras)

# ---- Change axis orientation ------------------
# convert from RAS orientation to LPS
surf_lps <- surface_to_surface(
  surf_ras,
  space_from = new_space("", orientation = "RAS"),
  space_to = new_space("", orientation = "LPS")
)
plot(surf_lps)

# validate
lps_verts <- diag(c(-1, -1, 1, 1)) \%*\% surf_ras$geometry$vertices
range(surf_lps$geometry$vertices - lps_verts)

# ---- Apply transforms ------------------
transform <- matrix(
  byrow = TRUE, nrow = 4,
  c(
    0.5, 0, 0.3, 1,
    0, -1, 0.2, 2,
    0, 0.7, -0.5, 4,
    0, 0, 0, 1
  )
)
surf_stretch <- surface_to_surface(surf_ras, transform = transform)
plot(surf_stretch)

# validate
stretch_verts <- transform \%*\% surf_ras$geometry$vertices
range(surf_stretch$geometry$vertices - stretch_verts)


}
}
\seealso{
\code{\link{as_ieegio_surface}} for creating surface objects,
\code{\link{new_space}} for defining coordinate spaces,
\code{\link{transform_orientation}} for orientation transforms,
\code{\link{volume_to_surface}} for creating surfaces from volumes
}
