% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ibdsim.R
\name{ibdsim}
\alias{ibdsim}
\title{IBD simulation}
\usage{
ibdsim(
  x,
  N = 1,
  ids = labels(x),
  map = "decode",
  model = c("chi", "haldane"),
  skipRecomb = NULL,
  seed = NULL,
  verbose = TRUE
)
}
\arguments{
\item{x}{A \code{\link[pedtools:ped]{pedtools::ped()}} object.}

\item{N}{A positive integer indicating the number of simulations.}

\item{ids}{A subset of pedigree members whose IBD sharing should be analysed.
If NULL, all members are included.}

\item{map}{The genetic map to be used in the simulations: Allowed values are:
\itemize{
\item a \code{genomeMap} object, typically produced by \code{\link[=loadMap]{loadMap()}}
\item a single \code{chromMap} object, for instance as produced by \code{\link[=uniformMap]{uniformMap()}}
\item a character, which is passed on to \code{\link[=loadMap]{loadMap()}} with default parameters.
Currently the only valid option is "decode19" (or abbreviations of this).
}

Default: "decode19".}

\item{model}{Either "chi" or "haldane", indicating the statistical model for
recombination (see details). Default: "chi".}

\item{skipRecomb}{A vector of ID labels indicating individuals whose meioses
should be simulated without recombination. (Each child will then receive a
random strand of each chromosome.) The default action is to skip
recombination in founders who are uninformative for IBD sharing in the
\code{ids} individuals.}

\item{seed}{An integer to be passed on to \code{\link[=set.seed]{set.seed()}}).}

\item{verbose}{A logical.}
}
\value{
A list of \code{N} objects of class \code{genomeSim}.

A \code{genomeSim} object is essentially a numerical matrix describing the
allele flow through the pedigree in a single simulated. Each row
corresponds to a chromosomal segment. The first 4 columns describe the
segment (chromosome, start, end, length), and are followed by two columns
(paternal allele, maternal allele) for each of the \code{ids} individuals.

If \code{ids} has length 1, a column named "Aut" is added, whose entries are 1
for autozygous segments and 0 otherwise.

If \code{ids} has length 2, two columns are added:
\itemize{
\item \code{IBD} : The IBD status of each segment (= number of alleles shared
identical by descent). For a given segment, the IBD status is either 0, 1,
2 or NA. If either individual is autozygous in a segment, the IBD status is
reported as NA. With inbred individuals the \code{Sigma} column (see below) is
more informative than the \code{IBD} column.
\item \code{Sigma} : The condensed identity ("Jacquard") state of each segment,
given as an integer in the range 1-9. The numbers correspond to the
standard ordering of the condensed states. In particular, for non-inbred
individuals the states 9, 8, 7 correspond to IBD status 0, 1, 2
respectively.
}
}
\description{
This is the main function of the package, simulating the recombination
process in each meioses of a pedigree. The output summarises the IBD segments
between all or a subset of individuals.
}
\details{
Each simulation starts by unique alleles (labelled 1, 2, ...) being
distributed to the pedigree founders. In each meiosis, homologue chromosomes
are made to recombine according to the value of \code{model}:
\itemize{
\item \code{model = "haldane"}: In this model, crossover events are modelled as a
Poisson process along each chromosome.
\item \code{model = "chi"} (default): This uses a renewal process along the
four-strand bundle, with waiting times following a chi square distribution.
}

Recombination rates along each chromosome are determined by the \code{map}
parameter. The default value ("decode19") loads a thinned version of the
recombination map of the human genome published by Halldorsson et al (2019).

In many applications, the fine-scale default map is not necessary, and should
be replaced by simpler maps with constant recombination rates. See
\code{\link[=uniformMap]{uniformMap()}} and \code{\link[=loadMap]{loadMap()}} for ways to produce such maps.
}
\examples{

hs = halfSibPed()
sims = ibdsim(hs, N = 2, map = uniformMap(M = 1), seed = 10)
sims

# Inspect the first simulation
sims[[1]]
haploDraw(hs, sims[[1]], pos = 2)

# Full sib mating: all 9 states are possible
x = fullSibMating(1)
sim = ibdsim(x, N = 1, ids = 5:6, map = uniformMap(M = 10), seed = 1)
s = sim[[1]]
stopifnot(setequal(s[, 'Sigma'], 1:9))

}
\references{
Halldorsson et al. \emph{Characterizing mutagenic effects of
recombination through a sequence-level genetic map.} Science 363, no. 6425
(2019).
}
