#' Full Information Maximum Likelihood Estimates in Linear M-model and Linear
#' Y-model
#' 
#' Parameter estimates in system of correlated linear M-model and linear
#' Y-model with treatment-mediator interaction using the full information
#' maximum likelihood method.
#' 
#' %% ~~ If necessary, more details than the description above ~~ 
#' P-values are computed from normal distribution.
#' 
#' @param fit.M a fitted model object for mediator. It must be an object
#' generated by function ``\code{lm}''
#' @param fit.Y a fitted model object for outcome. It must be an object
#' generated by function ``\code{lm}''. It can contain treatment-mediator
#' interaction
#' @param X a character string of the name of the treatment variable
#' @param rho a numerical variable specifying the correlation coefficient
#' between the residual of the M-model and the residual of the Y-model. Its
#' range is between \code{-1} and \code{1}
#' @return A list containing the following components: \item{M.model}{a data
#' frame containing the results for the M-model } \item{Y.model}{a data frame
#' containing the results for the Y-model }
#' @author Kai Wang \code{<kai-wang@@uiowa.edu>}
#' @references Wang, K. (2017) Identification and estimation in mediation
#' models with interaction in the presence of unobserved confounders.
#' Submitted.
#' @keywords estimates
#' @examples
#' 
#' data("jobs", package = "mediation")
#' 
#' fit.M <- lm(job_seek ~ treat + econ_hard + sex + age, data=jobs)
#' fit.Y <- lm(depress2 ~ treat + job_seek + econ_hard + sex + age, data=jobs)
#' fimle.lnl(fit.M, fit.Y, "treat", rho=0.2)
#' 
#' 
#' fit.M <- lm(job_seek ~ treat + econ_hard + sex + age , data=jobs)
#' fit.Y <- lm(depress2 ~ treat*job_seek+ econ_hard + sex + age , data=jobs)
#' fimle.lnl(fit.M, fit.Y, "treat", rho=0.5)
#'
#' @export fimle.lnl
#' @import stats
fimle.lnl = function(fit.M, fit.Y, X, rho=0) {
     M = all.vars(formula(fit.M))[1]
     Y.factors = attr(terms(formula(fit.Y)), "term.labels")
     n = length(residuals(fit.M))
 
     s1 = sigma(fit.M)
     s2 = sigma(fit.Y)/sqrt(1-rho^2)              
     tau = rho*s2/s1

     M.c = coef(summary(fit.M))
     Y.c = coef(summary(fit.Y))
     for (var in row.names(M.c)){
        	Y.c[var,"Estimate"] = Y.c[var,"Estimate"] + M.c[var,"Estimate"]*tau
     }
     Y.c[M,"Estimate"] = Y.c[M,"Estimate"] - tau
     L1 = fit.M$model
     L1[,1] = 1
     names(L1)[1] = "(Intercept)"
     L2 = fit.Y$model
     L2[,1] = 1
     names(L2)[1] = "(Intercept)"
     MMM = with(L2, get(M))
     L2 = L2[, !(names(L2) %in% M)]
     L2$Mvar = MMM
     names(L2)[names(L2) == "Mvar"] = M
     if (paste(X, M, sep=":") %in% Y.factors){
 	    L2$interaction = with(L2, get(X) * get(M))
 	    names(L2)[names(L2) == "interaction"] = paste(X, M, sep=":")
     }
     if (paste(M, X, sep=":") %in% Y.factors){ 
 	    L2$interaction = with(L2, get(X) * get(M))
 	    names(L2)[names(L2) == "interaction"] = paste(M, X, sep=":")
     }
 
     L1 = as.matrix(L1)
     L2 = as.matrix(L2)

     A = rbind(cbind(1/s1^2*crossprod(L1), -rho/s1/s2*crossprod(L1, L2)), 
               cbind(-rho/s1/s2*crossprod(L2, L1), 1/s2^2*crossprod(L2)))/(1-rho^2)
     B = matrix(0, nrow = ncol(L1), ncol=2)
     B2 = kronecker(matrix(c(-1/s1^2, 1/s2^2), ncol=2)*rho/2/(1-rho^2)/s1/s2, crossprod(L2, residuals(fit.M))) 
     B = rbind(B, B2)
     C = n/4/(1-rho^2)*matrix(c((2-rho^2)/s1^4, -rho^2/s1^2/s2^2, -rho^2/s1^2/s2^2, (2-rho^2)/s2^4), ncol=2)
 
     VC = solve(A - B %*% solve(C, t(B)))
     SEs = sqrt(diag(VC))
     M.SEs = SEs[1:nrow(M.c)]
     Y.SEs = SEs[-(1:nrow(M.c))]
     for (var in row.names(M.c)){
      	M.c[var,"Std. Error"] = M.SEs[names(M.SEs) == var]
     }
     for (var in row.names(Y.c)){
      	Y.c[var,"Std. Error"] = Y.SEs[names(Y.SEs) == var]
     }
 
     M.c[, "t value"] = M.c[,"Estimate"]/M.c[,"Std. Error"]
     M.c[, "Pr(>|t|)"] = signif(2*pnorm(abs(M.c[,"t value"]), lower.tail=FALSE), 4)
     Y.c[, "t value"] = Y.c[,"Estimate"]/Y.c[,"Std. Error"]
     Y.c[, "Pr(>|t|)"] = signif(2*pnorm(abs(Y.c[,"t value"]), lower.tail=FALSE), 4)
 
    list(M.model = as.data.frame(M.c), Y.model = as.data.frame(Y.c))
 }
