% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcmc_functions.R
\name{hmc}
\alias{hmc}
\title{Fit a generic model using Hamiltonian Monte Carlo (HMC)}
\usage{
hmc(
  N = 10000,
  theta.init,
  epsilon = 0.01,
  L = 10,
  logPOSTERIOR,
  glogPOSTERIOR,
  randlength = FALSE,
  Mdiag = NULL,
  constrain = NULL,
  verbose = FALSE,
  varnames = NULL,
  param = list(),
  chains = 1,
  parallel = FALSE,
  ...
)
}
\arguments{
\item{N}{Number of MCMC samples}

\item{theta.init}{Vector of initial values for the parameters}

\item{epsilon}{Step-size parameter for \code{leapfrog}}

\item{L}{Number of \code{leapfrog} steps parameter}

\item{logPOSTERIOR}{Function to calculate and return the log posterior given a vector of values of \code{theta}}

\item{glogPOSTERIOR}{Function to calculate and return the gradient of the log posterior given a vector of values of  \code{theta}}

\item{randlength}{Logical to determine whether to apply some randomness to the number of leapfrog steps tuning parameter \code{L}}

\item{Mdiag}{Optional vector of the diagonal of the mass matrix \code{M}.  Defaults to unit diagonal.}

\item{constrain}{Optional vector of which parameters in \code{theta} accept positive values only.  Default is that all parameters accept all real numbers}

\item{verbose}{Logical to determine whether to display the progress of the HMC algorithm}

\item{varnames}{Optional vector of theta parameter names}

\item{param}{List of additional parameters for \code{logPOSTERIOR} and \code{glogPOSTERIOR}}

\item{chains}{Number of MCMC chains to run}

\item{parallel}{Logical to set whether multiple MCMC chains should be run in parallel}

\item{...}{Additional parameters for \code{logPOSTERIOR}}
}
\value{
Object of class \code{hmclearn}
}
\description{
This function runs the HMC algorithm on a generic model provided
the \code{logPOSTERIOR} and gradient \code{glogPOSTERIOR} functions.
All parameters specified within the list \code{param}are passed to these two functions.
The tuning parameters \code{epsilon} and \code{L} are passed to the
Leapfrog algorithm.
}
\section{Elements for \code{hmclearn} objects}{

\describe{
  \item{\code{N}}{
  Number of MCMC samples
  }
  \item{\code{theta}}{
  Nested list of length \code{N} of the sampled values of \code{theta} for each chain
  }
  \item{\code{thetaCombined}}{
  List of dataframes containing sampled values, one for each chain
  }
  \item{\code{r}}{
  List of length \code{N} of the sampled momenta
  }
  \item{\code{theta.all}}{
  Nested list of all parameter values of \code{theta} sampled prior to accept/reject step for each
  }
  \item{\code{r.all}}{
  List of all values of the momenta \code{r} sampled prior to accept/reject
  }
  \item{\code{accept}}{
  Number of accepted proposals.  The ratio \code{accept} / \code{N} is the acceptance rate
  }
  \item{\code{accept_v}}{
  Vector of length \code{N} indicating which samples were accepted
  }
  \item{\code{M}}{
  Mass matrix used in the HMC algorithm
  }
  \item{\code{algorithm}}{
  \code{HMC} for Hamiltonian Monte Carlo
  }
  \item{\code{varnames}}{
  Optional vector of parameter names
  }
  \item{\code{chains}}{
  Number of MCMC chains
  }
}
}

\section{Available \code{logPOSTERIOR} and \code{glogPOSTERIOR} functions}{

\describe{
  \item{\code{linear_posterior}}{
  Linear regression:  log posterior
  }
  \item{\code{g_linear_posterior}}{
  Linear regression:  gradient of the log posterior
  }
  \item{\code{logistic_posterior}}{
  Logistic regression:  log posterior
  }
  \item{\code{g_logistic_posterior}}{
  Logistic regression:  gradient of the log posterior
  }
  \item{\code{poisson_posterior}}{
  Poisson (count) regression:  log posterior
  }
  \item{\code{g_poisson_posterior}}{
  Poisson (count) regression: gradient of the log posterior
  }
  \item{\code{lmm_posterior}}{
  Linear mixed effects model:  log posterior
  }
  \item{\code{g_lmm_posterior}}{
  Linear mixed effects model:  gradient of the log posterior
  }
  \item{\code{glmm_bin_posterior}}{
  Logistic mixed effects model:  log posterior
  }
  \item{\code{g_glmm_bin_posterior}}{
  Logistic mixed effects model:  gradient of the log posterior
  }
  \item{\code{glmm_poisson_posterior}}{
  Poisson mixed effects model:  log posterior
  }
  \item{\code{g_glmm_poisson_posterior}}{
  Poisson mixed effects model:  gradient of the log posterior
  }
}
}

\examples{
# Linear regression example
set.seed(521)
X <- cbind(1, matrix(rnorm(300), ncol=3))
betavals <- c(0.5, -1, 2, -3)
y <- X\%*\%betavals + rnorm(100, sd=.2)

fm1_hmc <- hmc(N = 500,
          theta.init = c(rep(0, 4), 1),
          epsilon = 0.01,
          L = 10,
          logPOSTERIOR = linear_posterior,
          glogPOSTERIOR = g_linear_posterior,
          varnames = c(paste0("beta", 0:3), "log_sigma_sq"),
          param=list(y=y, X=X), parallel=FALSE, chains=1)

summary(fm1_hmc, burnin=100)


# poisson regression example
set.seed(7363)
X <- cbind(1, matrix(rnorm(40), ncol=2))
betavals <- c(0.8, -0.5, 1.1)
lmu <- X \%*\% betavals
y <- sapply(exp(lmu), FUN = rpois, n=1)

fm2_hmc <- hmc(N = 500,
          theta.init = rep(0, 3),
          epsilon = 0.01,
          L = 10,
          logPOSTERIOR = poisson_posterior,
          glogPOSTERIOR = g_poisson_posterior,
          varnames = paste0("beta", 0:2),
          param = list(y=y, X=X),
          parallel=FALSE, chains=1)

summary(fm2_hmc, burnin=100)


}
\references{
Neal, Radford. 2011. \emph{MCMC Using Hamiltonian Dynamics.} In Handbook of Markov Chain Monte Carlo, edited by Steve Brooks, Andrew Gelman, Galin L. Jones, and Xiao-Li Meng, 116–62. Chapman; Hall/CRC.

Betancourt, Michael. 2017.  \emph{A Conceptual Introduction to Hamiltonian Monte Carlo}.

Thomas, S., Tu, W. 2020. \emph{Learning Hamiltonian Monte Carlo in R}.
}
\author{
Samuel Thomas \email{samthoma@iu.edu}, Wanzhu Tu \email{wtu@iu.edu}
}
\keyword{carlo}
\keyword{hamiltonian}
\keyword{monte}
