% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evalmethod.R
\name{eval_method}
\alias{eval_method}
\title{Evaluation of the procedures determining the high-risk zone.}
\usage{
eval_method(
  ppdata,
  type,
  criterion,
  cutoff,
  numit = 100,
  nxprob = 0.1,
  distancemap = NULL,
  intens = NULL,
  covmatrix = NULL,
  simulate,
  radiusClust = NULL,
  clustering = 5,
  pbar = TRUE
)
}
\arguments{
\item{ppdata}{Observed spatial point process of class ppp.}

\item{type}{Method to use, can be one of \code{"dist"} (method of fixed radius or quantile-based method), or
\code{"intens"} (intensity-based method)}

\item{criterion}{criterion to limit the high-risk zone, can be one of 
\code{"area"} (giving size of hrz), \code{"indirect"} (giving quantile/alpha depending on type),
or \code{"direct"} (giving radius/threshold c depending on type)}

\item{cutoff}{Value of criterion (area, radius, quantile, alpha or threshold). 
Depending on criterion and type: 
If criterion = "direct" and type = "intens", cutoff is the maximum intensity of unexploded bombs
outside the risk zone. If type = "dist" instead, cutoff is the radius of the circle around each exploded bomb.
"If criterion = "indirect", cutoff is the quantile for the quantile-based method and the failure 
probability alpha for the intensity-base method. If criterion = "area", cutoff is the area the high-risk zone should
have.}

\item{numit}{Number of iterations}

\item{nxprob}{Probability of having unobserved events.
Default value is 0.1.}

\item{distancemap}{(optional) distance map: distance of every pixel to the nearest observation 
of the point pattern; only needed for \code{type="dist"}. If not given, 
it will be computed by \code{\link[spatstat.geom]{distmap}}.}

\item{intens}{(optional) estimated intensity of the observed process (object of class "im"), 
only needed for type="intens". If not given,
it will be estimated using \code{\link[spatstat.core]{density.ppp}}.}

\item{covmatrix}{(optional) Covariance matrix of the kernel of a normal distribution, only needed for 
\code{type="intens"} if no intensity is given. If not given, it will be estimated
using \code{\link[ks]{Hscv}}.}

\item{simulate}{The type of simulation, can be one of \code{"thinning", "intens"} or \code{"clintens"}}

\item{radiusClust}{(Optional) radius of the circles around the parent points in which the cluster
points are located. Only used for \code{simulate = "clintens"}.}

\item{clustering}{a value >= 1 which describes the amount of clustering; the
adjusted estimated intensity of the observed pattern is divided by
this value; it is also the parameter of the Poisson distribution
for the number of points per cluster. Only used for \code{simulate = "clintens"}.}

\item{pbar}{logical. Should progress bar be printed?}
}
\value{
A \code{data.frame} with variables 
   \item{ Iteration }{ Iterationstep of the result }
   \item{ Type, Criterion, Cutoff, nxprob }{ see arguments }
   \item{ threshold }{ determined threshold. If criterion="area", it is either the distance (if type="dist")
or the threshold c (for type="intens"). If criterion="indirect", it is either the quantile of the
nearest-neighbour distance which is used as radius (if type="dist") or the threshold c (for type="intens"). If criterion="direct",
it equals the cutoff for both types.}
   \item{ calccutoff }{ determined cutoff-value. For type="dist" and criterion="area", this is the
quantile of the nearest-neighbour distance. For type="intens" and criterion="area", it is the failure
probability alpha. For all other criterions it is NA.}
   \item{ covmatrix11, covmatrix12, covmatrix21, covmatrix22 }{ values in the covariance matrix. 
covmatrix11 and covmatrix22 are the diagonal elements (variances). }
   \item{ numbermiss }{ number of unobserved points outside the high-risk zone }
   \item{ numberunobserved }{ number of observations in the unobserved point pattern \ifelse{latex}{\eqn{\tilde Z}}{Z'} }
   \item{ missingfrac }{ fraction of unobserved events outside the high-risk zone (numbermiss/numberunobserved) }
   \item{ arearegion }{ area of the high-risk zone }
   \item{ numberobs }{ number of observations in the observed point pattern \ifelse{latex}{\eqn{\tilde Y}}{Y'} }
}
\description{
Evaluates the performance of the three methods:
\itemize{
   \item Method of fixed radius
   \item Quantile-based method
   \item Intensity-based method
}
For further details on the methods, see \code{\link{det_hrz}} or the paper of Mahling et al. (2013)(References). \cr
There are three ways to simulate data for the evaluation.
}
\details{
The three simulation types are:
\describe{
\item{ Data-based simulation }{
       Here a given data set is used. The data set is thinned as explained below.
       Note that this method is very different from the others, since it is using the real data.
       }
\item{ Simulation of an inhomogeneous Poisson process }{
       Here, an inhomogeneous Poisson process is simulated and then that data is thinned.
       }
\item{ Simulation of a Neyman-Scott process }{
       Here a Neyman-Scott process is simulated (see \code{\link{sim_nsppp}}, \code{\link[spatstat.core]{rNeymanScott}})
       and this data is then also thinned.
       }
}
Thinning:\cr
Let \eqn{X} be the spatial point process, which is the location of all events and let \eqn{Y}
be a subset of \eqn{X} describing the observed process. The process of unobserved events
then is \ifelse{latex}{\eqn{Z = X \setminus Y}}{Z = X \ Y} , meaning that \eqn{Z} and \eqn{Y} are disjoint and together 
forming \eqn{X}.\cr
Since \eqn{Z} is not known, in this function an observed or simulated spatial point pattern 
\code{ppdata} is taken as the full pattern (which we denote by \ifelse{latex}{\eqn{\tilde X}}{X'}) comprising the 
observed events \ifelse{latex}{\eqn{\tilde Y}}{Y'} as well as the unobserved \ifelse{latex}{\eqn{\tilde Z}}{Z'}.
Each event in \ifelse{latex}{\eqn{\tilde X}}{X'} is assigned to one of the two processes \ifelse{latex}{\eqn{\tilde Y}}{Y'} or 
\ifelse{latex}{\eqn{\tilde Z}}{Z'} by drawing independent Bernoulli random numbers. \cr
The resulting process of observed events \ifelse{latex}{\eqn{\tilde Y}}{Y'} is used to determine the high-risk zone. 
Knowing now the unobserved process, it can be seen how many events are outside and inside the
high-risk zone.\cr

\code{type} and \code{criterion} may be vectors in this function.
}
\examples{
\dontrun{    
 data(craterB)
 
 # the input values are mainly the same as in det_hrz, so for more example ideas, 
 # see the documentation of det_hrz.
 evalm <- eval_method(craterB, type = c("dist", "intens"), criterion = c("area", "area"), 
                      cutoff = c(1500000, 1500000), nxprob = 0.1, numit = 10, 
                      simulate = "clintens", radiusClust = 300, 
                      clustering = 15, pbar = FALSE)
 evalm_d <- subset(evalm, evalm$Type == "dist")
 evalm_i <- subset(evalm, evalm$Type == "intens")
 
 # pout:  fraction of high-risk zones that leave at least one unobserved event uncovered
 # pmiss:  Mean fraction of unobserved events outside the high-risk zone
 data.frame(pmiss_d = mean(evalm_d$missingfrac),
            pmiss_i = mean(evalm_i$missingfrac),
            pout_d = ( sum(evalm_d$numbermiss > 0) / nrow(evalm_d) ), 
            pout_i = ( sum(evalm_i$numbermiss > 0) / nrow(evalm_i) ))
}
}
\seealso{
\code{\link{det_hrz}}, \code{\link[spatstat.core]{rNeymanScott}}, \code{\link{thin}}, \code{\link{sim_nsppp}}, \code{\link{sim_intens}}
}
