/* $OpenXM: OpenXM/src/hgm/so3/src/so3_nc.c,v 1.9 2019/11/16 10:57:21 takayama Exp $ */
#include <stdio.h>
#include <math.h>
#include <string.h>
#ifdef USE_GSL_LIB
#include <gsl/gsl_errno.h>
#include <gsl/gsl_odeiv.h>
#else
#include "t-gsl_errno.h"
#include "t-gsl_odeiv.h"
#endif

#include "oxprint.h"

#ifndef STANDALONE
void mh_check_intr(int n);
#endif

/* gcc evalnc.c -lgsl -lblas -lm */
/* gcc evalnc.c `pkg-config --cflags gsl` `pkg-config --libs gsl` */

int so3_func();
void so3_nc(double a[3],double t0,double y[4]);
void so3_evalByS(int deg,double a,double b,double c,double t,double f[4]);
int so3_usage(void);

#define MDEG 20

#define SO3_QUIET_DEFAULT 0
#define SO3_Deg_DEFAULT 10
#define SO3_Log_DEFAULT 0
int SO3_Quiet = SO3_QUIET_DEFAULT;
int SO3_Deg = SO3_Deg_DEFAULT;
int SO3_Log = SO3_Log_DEFAULT;
#ifdef STANDALONE
int main(int argc,char *argv[]) {
  double a[3];
  double y[4];
  double t0;
  int i,j;

  t0 = 0.0001;  /* Small enough number seems to be good (Hueristics) */ 
  j = 0;
  for (i=1; i<argc; i++) {
	if (strcmp(argv[i],"--quiet")==0) SO3_Quiet = 1;
	else if (strcmp(argv[i],"--t0")==0) {
	  i++; sscanf(argv[i],"%lg",&t0);
	} else if (strcmp(argv[i],"--log")==0) {
	  SO3_Log = 1;
	}else if (strcmp(argv[i],"--deg")==0) {
	  i++; sscanf(argv[i],"%d",&SO3_Deg);
	  if ((SO3_Deg > MDEG-2) || (SO3_Deg < 0)) {
	    oxprintfe("Error: deg should be less than %d\n",MDEG-2); return(-1);
	  }
    } else if (j<3) sscanf(argv[i],"%lg",&(a[j++]));
  }
  if (j != 3) {
    so3_usage();
    return(-1);
  }
  so3_nc(a,t0,y);
// oxprintf("%lg, %lg, %lg, %lg\n",y[0],y[1],y[2],y[3]);
  if (SO3_Log) oxprintf("log of nc : ");
  oxprintf("%.15e, %.15e, %.15e, %.15e\n",y[0],y[1],y[2],y[3]);
}
#else
void so3_main(double *in1,double *in2,double *in3,double *t0p,int *quiet,int *deg,int *log,double *out) {
  double a[3];
  double y[4];
  double t0;
  int i;

  SO3_Quiet = SO3_QUIET_DEFAULT;
  SO3_Deg = SO3_Deg_DEFAULT;
  SO3_Log = SO3_Log_DEFAULT;
  if (*quiet) SO3_Quiet = 1;
  if (*deg) {
    SO3_Deg = *deg;
    if (!SO3_Quiet) oxprintfe("deg is set to %d\n",SO3_Deg);
  }
  if (*log) SO3_Log=1;
  t0 = 0.0001;  /* Small enough number seems to be good (Hueristics) */ 
  if (*t0p > 0.0) {
    t0 = *t0p;
    if (!SO3_Quiet) oxprintfe("t0 is set to %lf\n",t0);
  }
  //  j = 0;
  if ((SO3_Deg > MDEG-2) || (SO3_Deg < 0)) {
    oxprintfe("Error: deg should be less than %d\n",MDEG-2);
    *out = 0.0; return;
  }
  a[0] = *in1; a[1] = *in2; a[2] = *in3;
  //  oxprintfe("DEBUG: %lf,%lf,%lf,%lf\n",t0,a[0],a[1],a[2]);
  so3_nc(a,t0,y);
// oxprintf("%lg, %lg, %lg, %lg\n",y[0],y[1],y[2],y[3]);
  if ((!SO3_Quiet) && SO3_Log) oxprintf("log of nc : ");
  if (!SO3_Quiet) oxprintf("%.15e, %.15e, %.15e, %.15e\n",y[0],y[1],y[2],y[3]);
  for (i=0; i<4; i++) out[i] = y[i];
  return;
}
#endif

int so3_usage(void) {
  oxprintfe("Usage: so3_nc a b c returns nc(a,b,c) and its gradients\n");
  oxprintfe("   where nc is the normalization constant\n" );
  oxprintfe("   of the Fisher distribution on SO(3) for the diagonal matrix diag(a,b,c).\n");
  oxprintfe("   See http://arxiv.org/abs/1110.0721\n");
  oxprintfe("Options:  --quiet  --t0 T0 --deg DEG --log\n");
  oxprintfe("   Series is evaluated at T0*(a,b,c) and the value is extended to (a,b,c) by diff. eq. With --log, log(nc(a,b,c)) is returned.\n");
  return(0);
}

/* Evaluate normalization constant */
double SO3_A[3];
double SO3_R; /* rho */
int so3_func();
void so3_nc(double a[3],double t0,double y[4]) {
  int i;
  int deg;
  double r;
  double y0[4];
  double myerr,myerr2;
  double aa;
  deg = SO3_Deg;
  for (i=0; i<3; i++) SO3_A[i]=a[i];

  /* When the argument is small, eval it only by series */
  aa = 0.0; for (i=0; i<3; i++) aa += a[i]*a[i];
  if (aa < 0.01) {
    so3_evalByS(deg,a[0],a[1],a[2], 1.0, y);
    return;
  }

  SO3_R = 0.0;
  r = a[0]-a[1]-a[2]; if (r > SO3_R) SO3_R = r;
  r = -a[0]+a[1]-a[2]; if (r > SO3_R) SO3_R = r;
  r = -a[0]-a[1]+a[2]; if (r > SO3_R) SO3_R = r;
  r = a[0]+a[1]+a[2]; if (r > SO3_R) SO3_R = r;
  if (!SO3_Quiet) oxprintfe("SO3_R=%lg, exp(SO3_R t) is the asymptotics of nc.\n",SO3_R);

  const gsl_odeiv_step_type *T = gsl_odeiv_step_rkf45;
  gsl_odeiv_step *s = gsl_odeiv_step_alloc(T, 4); /* rank4 */

  /* 
   * Absolute error 1e-6
   * Relative error 0.0.
   */
  gsl_odeiv_control *c = gsl_odeiv_control_y_new(1e-6, 0.0);

  /* rank 4 system */
  gsl_odeiv_evolve *e = gsl_odeiv_evolve_alloc(4);
  gsl_odeiv_system sys = {so3_func, NULL, 4, NULL};
  /* t : start, t1 : goal */
  double t = t0, t1 = 1.0; 
  double h = 1e-6;
  if (!SO3_Quiet) {
	oxprintfe("Set initial values at t0*(a,b,c) by evaluating series and find relevant t0.\n");
	oxprintfe("t0=%lf, a=a[0]=%lf, b=a[1]=%lf, c=a[2]=%lf\n",t0,a[0],a[1],a[2]);
  }
  do {
    if (!SO3_Quiet) oxprintfe("t0=%lf\n",t0);
    so3_evalByS(deg,a[0],a[1],a[2], t0, y0);
    if (!SO3_Quiet) oxprintfe("[%2d]: %lg,%lg,%lg,%lg\n",deg,y0[0],y0[1],y0[2],y0[3]);
    so3_evalByS(deg+1,a[0],a[1],a[2], t0, y);
    if (!SO3_Quiet) oxprintfe("[%2d]:  %lg,%lg,%lg,%lg\n",deg+1,y[0],y[1],y[2],y[3]);
    myerr=0.0;
    for (i=0; i<4; i++) {
      myerr2 = (y0[i]-y[i])/y0[i]; if (myerr2 <0) myerr2 = -myerr2;
      if (myerr2 > myerr) myerr = myerr2;  /* sup norm */
    }
    if (myerr < 1e-6) break;  /* should take smaller value? */
    t0 = t0/2.0;
  } while (1);
  t=t0;
  for (i=0; i<4; i++) y[i]=y[i]*exp(-SO3_R*t0);
  if (!SO3_Quiet) oxprintfe("[%2d]*exp(-SO3_R*t0): %lg,%lg,%lg,%lg\n",deg+1,y[0],y[1],y[2],y[3]);
  if (!SO3_Quiet) oxprintfe("Result by HGM (solving ODE) ------ \n");

  while (t < t1) {
    int status = gsl_odeiv_evolve_apply(e, c, s, &sys, &t, t1, &h, y);
    if (status != GSL_SUCCESS) break;
  }
  if (!SO3_Quiet) oxprintfe("t and V   : t=%.5e %.5e %.5e, %.5e %.5e\n", t, y[0], y[1],y[2],y[3]);

  if (!SO3_Log) {
    for (i=0; i<4; i++) y[i]=y[i]*exp(SO3_R*t);
    if (!SO3_Quiet) oxprintfe("V*exp(SO3_R*1): t= %.5e %.5e %.5e, %.5e %.5e\n", t, y[0], y[1],y[2],y[3]);
    if (!SO3_Quiet) oxprintfe("Returned value is V=[so3_nc(a,b,c),  c_a, c_b, c_c]\n");
  }else{
    for (i=0; i<4; i++) y[i]=log(y[i]) + SO3_R*t;
    if (!SO3_Quiet) oxprintfe("log(V*exp(SO3_R*1)): t= %.5e %.5e %.5e, %.5e %.5e\n", t, y[0], y[1],y[2],y[3]);
    if (!SO3_Quiet) oxprintfe("Returned value is V=[log(so3_nc(a,b,c)),  log(c_a), log(c_b), log(c_c)]\n");
    /* test input: ./hgm_so3_nc --t0 0.001 --deg 18 --log 1000 500 50 
       2019.11.16  when we change the argument number, change also mh-r.c (interface module.
    */
  }

  gsl_odeiv_evolve_free(e);
  gsl_odeiv_control_free(c);
  gsl_odeiv_step_free(s);
}

/* From FB2/Prog/gls_ode_test2b.c */

/*
d/dt y_i = f_i(t, y_1, ..., y_n),  f_i=f[i]
 y[0] = y, y[1]=y'.

Ref: Note: See the Corollary 1 of the paper rotation.c on this function func.

*/
int so3_func(double t, const double y[], double f[], void *params)
{
  extern double SO3_A[3];
#ifndef STANDALONE
  mh_check_intr(100);
#endif
  f[0] = SO3_A[0]*y[1]+SO3_A[1]*y[2]+SO3_A[2]*y[3]               -SO3_R*y[0];
  f[1] = SO3_A[0]*y[0]+SO3_A[2]*y[2]+SO3_A[1]*y[3] - (2/t)*y[1]  -SO3_R*y[1];
  f[2] = SO3_A[1]*y[0]+SO3_A[2]*y[1]+SO3_A[0]*y[3] - (2/t)*y[2]  -SO3_R*y[2];
  f[3] = SO3_A[2]*y[0]+SO3_A[1]*y[1]+SO3_A[0]*y[2] - (2/t)*y[3]  -SO3_R*y[3];
  return GSL_SUCCESS;
}

/* Evaluation of nc by a series */
void so3_evalByS(int deg,double a,double b,double c,double t,double f[4]) {
  double Tnc[MDEG][MDEG][MDEG];
  double ex[MDEG][MDEG][MDEG];
  int i,j,k;
  if (deg >= MDEG) {
    oxprintfe("Error: degree is too high\n");
  }
  for (i=0; i<MDEG; i++)for (j=0; j<MDEG; j++)for (k=0; k<MDEG; k++)Tnc[i][j][k]=0.0;
  /* load("sord.rr"); forC(20); */
  /* begin table */
Tnc[19][19][19]=1.8373e-54;
Tnc[19][19][17]=6.62551e-52;
Tnc[19][17][19]=6.62551e-52;
Tnc[17][19][19]=6.62551e-52;
Tnc[18][18][18]=1.36444e-50;
Tnc[19][19][15]=1.9059e-49;
Tnc[19][17][17]=2.39276e-49;
Tnc[17][19][17]=2.39276e-49;
Tnc[19][15][19]=1.9059e-49;
Tnc[17][17][19]=2.39276e-49;
Tnc[15][19][19]=1.9059e-49;
Tnc[18][18][16]=4.41547e-48;
Tnc[18][16][18]=4.41547e-48;
Tnc[16][18][18]=4.41547e-48;
Tnc[19][19][13]=4.24702e-47;
Tnc[19][17][15]=6.89438e-47;
Tnc[17][19][15]=6.89438e-47;
Tnc[19][15][17]=6.89438e-47;
Tnc[17][17][17]=8.65399e-47;
Tnc[15][19][17]=6.89438e-47;
Tnc[19][13][19]=4.24702e-47;
Tnc[17][15][19]=6.89438e-47;
Tnc[15][17][19]=6.89438e-47;
Tnc[13][19][19]=4.24702e-47;
Tnc[18][18][14]=1.12446e-45;
Tnc[18][16][16]=1.43125e-45;
Tnc[16][18][16]=1.43125e-45;
Tnc[18][14][18]=1.12446e-45;
Tnc[16][16][18]=1.43125e-45;
Tnc[14][18][18]=1.12446e-45;
Tnc[19][19][11]=7.05695e-45;
Tnc[19][17][13]=1.53917e-44;
Tnc[17][19][13]=1.53917e-44;
Tnc[19][15][15]=1.99021e-44;
Tnc[17][17][15]=2.49763e-44;
Tnc[15][19][15]=1.99021e-44;
Tnc[19][13][17]=1.53917e-44;
Tnc[17][15][17]=2.49763e-44;
Tnc[15][17][17]=2.49763e-44;
Tnc[13][19][17]=1.53917e-44;
Tnc[19][11][19]=7.05695e-45;
Tnc[17][13][19]=1.53917e-44;
Tnc[15][15][19]=1.99021e-44;
Tnc[13][17][19]=1.53917e-44;
Tnc[11][19][19]=7.05695e-45;
Tnc[18][18][12]=2.17975e-43;
Tnc[18][16][14]=3.65163e-43;
Tnc[16][18][14]=3.65163e-43;
Tnc[18][14][16]=3.65163e-43;
Tnc[16][16][16]=4.64694e-43;
Tnc[14][18][16]=3.65163e-43;
Tnc[18][12][18]=2.17975e-43;
Tnc[16][14][18]=3.65163e-43;
Tnc[14][16][18]=3.65163e-43;
Tnc[12][18][18]=2.17975e-43;
Tnc[19][19][9]=8.30395e-43;
Tnc[19][17][11]=2.56293e-42;
Tnc[17][19][11]=2.56293e-42;
Tnc[19][15][13]=4.45255e-42;
Tnc[17][17][13]=5.58632e-42;
Tnc[15][19][13]=4.45255e-42;
Tnc[19][13][15]=4.45255e-42;
Tnc[17][15][15]=7.22178e-42;
Tnc[15][17][15]=7.22178e-42;
Tnc[13][19][15]=4.45255e-42;
Tnc[19][11][17]=2.56293e-42;
Tnc[17][13][17]=5.58632e-42;
Tnc[15][15][17]=7.22178e-42;
Tnc[13][17][17]=5.58632e-42;
Tnc[11][19][17]=2.56293e-42;
Tnc[19][9][19]=8.30395e-43;
Tnc[17][11][19]=2.56293e-42;
Tnc[15][13][19]=4.45255e-42;
Tnc[13][15][19]=4.45255e-42;
Tnc[11][17][19]=2.56293e-42;
Tnc[9][19][19]=8.30395e-43;
Tnc[18][18][10]=3.07779e-41;
Tnc[18][16][12]=7.09362e-41;
Tnc[16][18][12]=7.09362e-41;
Tnc[18][14][14]=9.3363e-41;
Tnc[16][16][14]=1.1878e-40;
Tnc[14][18][14]=9.3363e-41;
Tnc[18][12][16]=7.09362e-41;
Tnc[16][14][16]=1.1878e-40;
Tnc[14][16][16]=1.1878e-40;
Tnc[12][18][16]=7.09362e-41;
Tnc[18][10][18]=3.07779e-41;
Tnc[16][12][18]=7.09362e-41;
Tnc[14][14][18]=9.3363e-41;
Tnc[12][16][18]=7.09362e-41;
Tnc[10][18][18]=3.07779e-41;
Tnc[19][19][7]=6.42744e-41;
Tnc[19][17][9]=3.02313e-40;
Tnc[17][19][9]=3.02313e-40;
Tnc[19][15][11]=7.43206e-40;
Tnc[17][17][11]=9.3216e-40;
Tnc[15][19][11]=7.43206e-40;
Tnc[19][13][13]=9.98548e-40;
Tnc[17][15][13]=1.61866e-39;
Tnc[15][17][13]=1.61866e-39;
Tnc[13][19][13]=9.98548e-40;
Tnc[19][11][15]=7.43206e-40;
Tnc[17][13][15]=1.61866e-39;
Tnc[15][15][15]=2.09201e-39;
Tnc[13][17][15]=1.61866e-39;
Tnc[11][19][15]=7.43206e-40;
Tnc[19][9][17]=3.02313e-40;
Tnc[17][11][17]=9.3216e-40;
Tnc[15][13][17]=1.61866e-39;
Tnc[13][15][17]=1.61866e-39;
Tnc[11][17][17]=9.3216e-40;
Tnc[9][19][17]=3.02313e-40;
Tnc[19][7][19]=6.42744e-41;
Tnc[17][9][19]=3.02313e-40;
Tnc[15][11][19]=7.43206e-40;
Tnc[13][13][19]=9.98548e-40;
Tnc[11][15][19]=7.43206e-40;
Tnc[9][17][19]=3.02313e-40;
Tnc[7][19][19]=6.42744e-41;
Tnc[18][18][8]=2.97769e-39;
Tnc[18][16][10]=1.00404e-38;
Tnc[16][18][10]=1.00404e-38;
Tnc[18][14][12]=1.81805e-38;
Tnc[16][16][12]=2.31226e-38;
Tnc[14][18][12]=1.81805e-38;
Tnc[18][12][14]=1.81805e-38;
Tnc[16][14][14]=3.04251e-38;
Tnc[14][16][14]=3.04251e-38;
Tnc[12][18][14]=1.81805e-38;
Tnc[18][10][16]=1.00404e-38;
Tnc[16][12][16]=2.31226e-38;
Tnc[14][14][16]=3.04251e-38;
Tnc[12][16][16]=2.31226e-38;
Tnc[10][18][16]=1.00404e-38;
Tnc[18][8][18]=2.97769e-39;
Tnc[16][10][18]=1.00404e-38;
Tnc[14][12][18]=1.81805e-38;
Tnc[12][14][18]=1.81805e-38;
Tnc[10][16][18]=1.00404e-38;
Tnc[8][18][18]=2.97769e-39;
Tnc[19][19][5]=2.91875e-39;
Tnc[19][17][7]=2.34655e-38;
Tnc[17][19][7]=2.34655e-38;
Tnc[19][15][9]=8.79116e-38;
Tnc[17][17][9]=1.1022e-37;
Tnc[15][19][9]=8.79116e-38;
Tnc[19][13][11]=1.67142e-37;
Tnc[17][15][11]=2.70751e-37;
Tnc[15][17][11]=2.70751e-37;
Tnc[13][19][11]=1.67142e-37;
Tnc[19][11][13]=1.67142e-37;
Tnc[17][13][13]=3.63678e-37;
Tnc[15][15][13]=4.69881e-37;
Tnc[13][17][13]=3.63678e-37;
Tnc[11][19][13]=1.67142e-37;
Tnc[19][9][15]=8.79116e-38;
Tnc[17][11][15]=2.70751e-37;
Tnc[15][13][15]=4.69881e-37;
Tnc[13][15][15]=4.69881e-37;
Tnc[11][17][15]=2.70751e-37;
Tnc[9][19][15]=8.79116e-38;
Tnc[19][7][17]=2.34655e-38;
Tnc[17][9][17]=1.1022e-37;
Tnc[15][11][17]=2.70751e-37;
Tnc[13][13][17]=3.63678e-37;
Tnc[11][15][17]=2.70751e-37;
Tnc[9][17][17]=1.1022e-37;
Tnc[7][19][17]=2.34655e-38;
Tnc[19][5][19]=2.91875e-39;
Tnc[17][7][19]=2.34655e-38;
Tnc[15][9][19]=8.79116e-38;
Tnc[13][11][19]=1.67142e-37;
Tnc[11][13][19]=1.67142e-37;
Tnc[9][15][19]=8.79116e-38;
Tnc[7][17][19]=2.34655e-38;
Tnc[5][19][19]=2.91875e-39;
Tnc[18][18][6]=1.80281e-37;
Tnc[18][16][8]=9.74107e-37;
Tnc[16][18][8]=9.74107e-37;
Tnc[18][14][10]=2.58048e-36;
Tnc[16][16][10]=3.2807e-36;
Tnc[14][18][10]=2.58048e-36;
Tnc[18][12][12]=3.55017e-36;
Tnc[16][14][12]=5.93708e-36;
Tnc[14][16][12]=5.93708e-36;
Tnc[12][18][12]=3.55017e-36;
Tnc[18][10][14]=2.58048e-36;
Tnc[16][12][14]=5.93708e-36;
Tnc[14][14][14]=7.80966e-36;
Tnc[12][16][14]=5.93708e-36;
Tnc[10][18][14]=2.58048e-36;
Tnc[18][8][16]=9.74107e-37;
Tnc[16][10][16]=3.2807e-36;
Tnc[14][12][16]=5.93708e-36;
Tnc[12][14][16]=5.93708e-36;
Tnc[10][16][16]=3.2807e-36;
Tnc[8][18][16]=9.74107e-37;
Tnc[18][6][18]=1.80281e-37;
Tnc[16][8][18]=9.74107e-37;
Tnc[14][10][18]=2.58048e-36;
Tnc[12][12][18]=3.55017e-36;
Tnc[10][14][18]=2.58048e-36;
Tnc[8][16][18]=9.74107e-37;
Tnc[6][18][18]=1.80281e-37;
Tnc[19][19][3]=6.35428e-38;
Tnc[19][17][5]=1.0691e-36;
Tnc[17][19][5]=1.0691e-36;
Tnc[19][15][7]=6.84611e-36;
Tnc[17][17][7]=8.57921e-36;
Tnc[15][19][7]=6.84611e-36;
Tnc[19][13][9]=1.98356e-35;
Tnc[17][15][9]=3.21035e-35;
Tnc[15][17][9]=3.21035e-35;
Tnc[13][19][9]=1.98356e-35;
Tnc[19][11][11]=2.80689e-35;
Tnc[17][13][11]=6.10019e-35;
Tnc[15][15][11]=7.87855e-35;
Tnc[13][17][11]=6.10019e-35;
Tnc[11][19][11]=2.80689e-35;
Tnc[19][9][13]=1.98356e-35;
Tnc[17][11][13]=6.10019e-35;
Tnc[15][13][13]=1.05793e-34;
Tnc[13][15][13]=1.05793e-34;
Tnc[11][17][13]=6.10019e-35;
Tnc[9][19][13]=1.98356e-35;
Tnc[19][7][15]=6.84611e-36;
Tnc[17][9][15]=3.21035e-35;
Tnc[15][11][15]=7.87855e-35;
Tnc[13][13][15]=1.05793e-34;
Tnc[11][15][15]=7.87855e-35;
Tnc[9][17][15]=3.21035e-35;
Tnc[7][19][15]=6.84611e-36;
Tnc[19][5][17]=1.0691e-36;
Tnc[17][7][17]=8.57921e-36;
Tnc[15][9][17]=3.21035e-35;
Tnc[13][11][17]=6.10019e-35;
Tnc[11][13][17]=6.10019e-35;
Tnc[9][15][17]=3.21035e-35;
Tnc[7][17][17]=8.57921e-36;
Tnc[5][19][17]=1.0691e-36;
Tnc[19][3][19]=6.35428e-38;
Tnc[17][5][19]=1.0691e-36;
Tnc[15][7][19]=6.84611e-36;
Tnc[13][9][19]=1.98356e-35;
Tnc[11][11][19]=2.80689e-35;
Tnc[9][13][19]=1.98356e-35;
Tnc[7][15][19]=6.84611e-36;
Tnc[5][17][19]=1.0691e-36;
Tnc[3][19][19]=6.35428e-38;
Tnc[18][18][4]=5.88673e-36;
Tnc[18][16][6]=5.91702e-35;
Tnc[16][18][6]=5.91702e-35;
Tnc[18][14][8]=2.51178e-34;
Tnc[16][16][8]=3.1918e-34;
Tnc[14][18][8]=2.51178e-34;
Tnc[18][12][10]=5.05553e-34;
Tnc[16][14][10]=8.44721e-34;
Tnc[14][16][10]=8.44721e-34;
Tnc[12][18][10]=5.05553e-34;
Tnc[18][10][12]=5.05553e-34;
Tnc[16][12][12]=1.16178e-33;
Tnc[14][14][12]=1.52762e-33;
Tnc[12][16][12]=1.16178e-33;
Tnc[10][18][12]=5.05553e-34;
Tnc[18][8][14]=2.51178e-34;
Tnc[16][10][14]=8.44721e-34;
Tnc[14][12][14]=1.52762e-33;
Tnc[12][14][14]=1.52762e-33;
Tnc[10][16][14]=8.44721e-34;
Tnc[8][18][14]=2.51178e-34;
Tnc[18][6][16]=5.91702e-35;
Tnc[16][8][16]=3.1918e-34;
Tnc[14][10][16]=8.44721e-34;
Tnc[12][12][16]=1.16178e-33;
Tnc[10][14][16]=8.44721e-34;
Tnc[8][16][16]=3.1918e-34;
Tnc[6][18][16]=5.91702e-35;
Tnc[18][4][18]=5.88673e-36;
Tnc[16][6][18]=5.91702e-35;
Tnc[14][8][18]=2.51178e-34;
Tnc[12][10][18]=5.05553e-34;
Tnc[10][12][18]=5.05553e-34;
Tnc[8][14][18]=2.51178e-34;
Tnc[6][16][18]=5.91702e-35;
Tnc[4][18][18]=5.88673e-36;
Tnc[19][19][1]=4.18362e-37;
Tnc[19][17][3]=2.33662e-35;
Tnc[17][19][3]=2.33662e-35;
Tnc[19][15][5]=3.13132e-34;
Tnc[17][17][5]=3.9216e-34;
Tnc[15][19][5]=3.13132e-34;
Tnc[19][13][7]=1.55073e-33;
Tnc[17][15][7]=2.50705e-33;
Tnc[15][17][7]=2.50705e-33;
Tnc[13][19][7]=1.55073e-33;
Tnc[19][11][9]=3.34407e-33;
Tnc[17][13][9]=7.25682e-33;
Tnc[15][15][9]=9.36783e-33;
Tnc[13][17][9]=7.25682e-33;
Tnc[11][19][9]=3.34407e-33;
Tnc[19][9][11]=3.34407e-33;
Tnc[17][11][11]=1.02657e-32;
Tnc[15][13][11]=1.77879e-32;
Tnc[13][15][11]=1.77879e-32;
Tnc[11][17][11]=1.02657e-32;
Tnc[9][19][11]=3.34407e-33;
Tnc[19][7][13]=1.55073e-33;
Tnc[17][9][13]=7.25682e-33;
Tnc[15][11][13]=1.77879e-32;
Tnc[13][13][13]=2.38763e-32;
Tnc[11][15][13]=1.77879e-32;
Tnc[9][17][13]=7.25682e-33;
Tnc[7][19][13]=1.55073e-33;
Tnc[19][5][15]=3.13132e-34;
Tnc[17][7][15]=2.50705e-33;
Tnc[15][9][15]=9.36783e-33;
Tnc[13][11][15]=1.77879e-32;
Tnc[11][13][15]=1.77879e-32;
Tnc[9][15][15]=9.36783e-33;
Tnc[7][17][15]=2.50705e-33;
Tnc[5][19][15]=3.13132e-34;
Tnc[19][3][17]=2.33662e-35;
Tnc[17][5][17]=3.9216e-34;
Tnc[15][7][17]=2.50705e-33;
Tnc[13][9][17]=7.25682e-33;
Tnc[11][11][17]=1.02657e-32;
Tnc[9][13][17]=7.25682e-33;
Tnc[7][15][17]=2.50705e-33;
Tnc[5][17][17]=3.9216e-34;
Tnc[3][19][17]=2.33662e-35;
Tnc[19][1][19]=4.18362e-37;
Tnc[17][3][19]=2.33662e-35;
Tnc[15][5][19]=3.13132e-34;
Tnc[13][7][19]=1.55073e-33;
Tnc[11][9][19]=3.34407e-33;
Tnc[9][11][19]=3.34407e-33;
Tnc[7][13][19]=1.55073e-33;
Tnc[5][15][19]=3.13132e-34;
Tnc[3][17][19]=2.33662e-35;
Tnc[1][19][19]=4.18362e-37;
Tnc[18][18][2]=7.7507e-35;
Tnc[18][16][4]=1.93965e-33;
Tnc[16][18][4]=1.93965e-33;
Tnc[18][14][6]=1.53168e-32;
Tnc[16][16][6]=1.94516e-32;
Tnc[14][18][6]=1.53168e-32;
Tnc[18][12][8]=4.94009e-32;
Tnc[16][14][8]=8.24521e-32;
Tnc[14][16][8]=8.24521e-32;
Tnc[12][18][8]=4.94009e-32;
Tnc[18][10][10]=7.22723e-32;
Tnc[16][12][10]=1.65837e-31;
Tnc[14][14][10]=2.17953e-31;
Tnc[12][16][10]=1.65837e-31;
Tnc[10][18][10]=7.22723e-32;
Tnc[18][8][12]=4.94009e-32;
Tnc[16][10][12]=1.65837e-31;
Tnc[14][12][12]=2.99644e-31;
Tnc[12][14][12]=2.99644e-31;
Tnc[10][16][12]=1.65837e-31;
Tnc[8][18][12]=4.94009e-32;
Tnc[18][6][14]=1.53168e-32;
Tnc[16][8][14]=8.24521e-32;
Tnc[14][10][14]=2.17953e-31;
Tnc[12][12][14]=2.99644e-31;
Tnc[10][14][14]=2.17953e-31;
Tnc[8][16][14]=8.24521e-32;
Tnc[6][18][14]=1.53168e-32;
Tnc[18][4][16]=1.93965e-33;
Tnc[16][6][16]=1.94516e-32;
Tnc[14][8][16]=8.24521e-32;
Tnc[12][10][16]=1.65837e-31;
Tnc[10][12][16]=1.65837e-31;
Tnc[8][14][16]=8.24521e-32;
Tnc[6][16][16]=1.94516e-32;
Tnc[4][18][16]=1.93965e-33;
Tnc[18][2][18]=7.7507e-35;
Tnc[16][4][18]=1.93965e-33;
Tnc[14][6][18]=1.53168e-32;
Tnc[12][8][18]=4.94009e-32;
Tnc[10][10][18]=7.22723e-32;
Tnc[8][12][18]=4.94009e-32;
Tnc[6][14][18]=1.53168e-32;
Tnc[4][16][18]=1.93965e-33;
Tnc[2][18][18]=7.7507e-35;
Tnc[19][17][1]=1.54574e-34;
Tnc[17][19][1]=1.54574e-34;
Tnc[19][15][3]=6.87619e-33;
Tnc[17][17][3]=8.6046e-33;
Tnc[15][19][3]=6.87619e-33;
Tnc[19][13][5]=7.12626e-32;
Tnc[17][15][5]=1.15045e-31;
Tnc[15][17][5]=1.15045e-31;
Tnc[13][19][5]=7.12626e-32;
Tnc[19][11][7]=2.62665e-31;
Tnc[17][13][7]=5.68909e-31;
Tnc[15][15][7]=7.3395e-31;
Tnc[13][17][7]=5.68909e-31;
Tnc[11][19][7]=2.62665e-31;
Tnc[19][9][9]=4.00278e-31;
Tnc[17][11][9]=1.22596e-30;
Tnc[15][13][9]=2.12194e-30;
Tnc[13][15][9]=2.12194e-30;
Tnc[11][17][9]=1.22596e-30;
Tnc[9][19][9]=4.00278e-31;
Tnc[19][7][11]=2.62665e-31;
Tnc[17][9][11]=1.22596e-30;
Tnc[15][11][11]=3.00058e-30;
Tnc[13][13][11]=4.02567e-30;
Tnc[11][15][11]=3.00058e-30;
Tnc[9][17][11]=1.22596e-30;
Tnc[7][19][11]=2.62665e-31;
Tnc[19][5][13]=7.12626e-32;
Tnc[17][7][13]=5.68909e-31;
Tnc[15][9][13]=2.12194e-30;
Tnc[13][11][13]=4.02567e-30;
Tnc[11][13][13]=4.02567e-30;
Tnc[9][15][13]=2.12194e-30;
Tnc[7][17][13]=5.68909e-31;
Tnc[5][19][13]=7.12626e-32;
Tnc[19][3][15]=6.87619e-33;
Tnc[17][5][15]=1.15045e-31;
Tnc[15][7][15]=7.3395e-31;
Tnc[13][9][15]=2.12194e-30;
Tnc[11][11][15]=3.00058e-30;
Tnc[9][13][15]=2.12194e-30;
Tnc[7][15][15]=7.3395e-31;
Tnc[5][17][15]=1.15045e-31;
Tnc[3][19][15]=6.87619e-33;
Tnc[19][1][17]=1.54574e-34;
Tnc[17][3][17]=8.6046e-33;
Tnc[15][5][17]=1.15045e-31;
Tnc[13][7][17]=5.68909e-31;
Tnc[11][9][17]=1.22596e-30;
Tnc[9][11][17]=1.22596e-30;
Tnc[7][13][17]=5.68909e-31;
Tnc[5][15][17]=1.15045e-31;
Tnc[3][17][17]=8.6046e-33;
Tnc[1][19][17]=1.54574e-34;
Tnc[17][1][19]=1.54574e-34;
Tnc[15][3][19]=6.87619e-33;
Tnc[13][5][19]=7.12626e-32;
Tnc[11][7][19]=2.62665e-31;
Tnc[9][9][19]=4.00278e-31;
Tnc[7][11][19]=2.62665e-31;
Tnc[5][13][19]=7.12626e-32;
Tnc[3][15][19]=6.87619e-33;
Tnc[1][17][19]=1.54574e-34;
Tnc[18][18][0]=1.71749e-34;
Tnc[18][16][2]=2.56592e-32;
Tnc[16][18][2]=2.56592e-32;
Tnc[18][14][4]=5.04466e-31;
Tnc[16][16][4]=6.40127e-31;
Tnc[14][18][4]=5.04466e-31;
Tnc[18][12][6]=3.02663e-30;
Tnc[16][14][6]=5.04436e-30;
Tnc[14][16][6]=5.04436e-30;
Tnc[12][18][6]=3.02663e-30;
Tnc[18][10][8]=7.09536e-30;
Tnc[16][12][8]=1.625e-29;
Tnc[14][14][8]=2.13435e-29;
Tnc[12][16][8]=1.625e-29;
Tnc[10][18][8]=7.09536e-30;
Tnc[18][8][10]=7.09536e-30;
Tnc[16][10][10]=2.37641e-29;
Tnc[14][12][10]=4.28908e-29;
Tnc[12][14][10]=4.28908e-29;
Tnc[10][16][10]=2.37641e-29;
Tnc[8][18][10]=7.09536e-30;
Tnc[18][6][12]=3.02663e-30;
Tnc[16][8][12]=1.625e-29;
Tnc[14][10][12]=4.28908e-29;
Tnc[12][12][12]=5.8938e-29;
Tnc[10][14][12]=4.28908e-29;
Tnc[8][16][12]=1.625e-29;
Tnc[6][18][12]=3.02663e-30;
Tnc[18][4][14]=5.04466e-31;
Tnc[16][6][14]=5.04436e-30;
Tnc[14][8][14]=2.13435e-29;
Tnc[12][10][14]=4.28908e-29;
Tnc[10][12][14]=4.28908e-29;
Tnc[8][14][14]=2.13435e-29;
Tnc[6][16][14]=5.04436e-30;
Tnc[4][18][14]=5.04466e-31;
Tnc[18][2][16]=2.56592e-32;
Tnc[16][4][16]=6.40127e-31;
Tnc[14][6][16]=5.04436e-30;
Tnc[12][8][16]=1.625e-29;
Tnc[10][10][16]=2.37641e-29;
Tnc[8][12][16]=1.625e-29;
Tnc[6][14][16]=5.04436e-30;
Tnc[4][16][16]=6.40127e-31;
Tnc[2][18][16]=2.56592e-32;
Tnc[18][0][18]=1.71749e-34;
Tnc[16][2][18]=2.56592e-32;
Tnc[14][4][18]=5.04466e-31;
Tnc[12][6][18]=3.02663e-30;
Tnc[10][8][18]=7.09536e-30;
Tnc[8][10][18]=7.09536e-30;
Tnc[6][12][18]=3.02663e-30;
Tnc[4][14][18]=5.04466e-31;
Tnc[2][16][18]=2.56592e-32;
Tnc[0][18][18]=1.71749e-34;
Tnc[19][15][1]=4.57538e-32;
Tnc[17][17][1]=5.71923e-32;
Tnc[15][19][1]=4.57538e-32;
Tnc[19][13][3]=1.57398e-30;
Tnc[17][15][3]=2.5362e-30;
Tnc[15][17][3]=2.5362e-30;
Tnc[13][19][3]=1.57398e-30;
Tnc[19][11][5]=1.21405e-29;
Tnc[17][13][5]=2.62293e-29;
Tnc[15][15][5]=3.38111e-29;
Tnc[13][17][5]=2.62293e-29;
Tnc[11][19][5]=1.21405e-29;
Tnc[19][9][7]=3.16223e-29;
Tnc[17][11][7]=9.65617e-29;
Tnc[15][13][7]=1.66894e-28;
Tnc[13][15][7]=1.66894e-28;
Tnc[11][17][7]=9.65617e-29;
Tnc[9][19][7]=3.16223e-29;
Tnc[19][7][9]=3.16223e-29;
Tnc[17][9][9]=1.47094e-28;
Tnc[15][11][9]=3.59329e-28;
Tnc[13][13][9]=4.81788e-28;
Tnc[11][15][9]=3.59329e-28;
Tnc[9][17][9]=1.47094e-28;
Tnc[7][19][9]=3.16223e-29;
Tnc[19][5][11]=1.21405e-29;
Tnc[17][7][11]=9.65617e-29;
Tnc[15][9][11]=3.59329e-28;
Tnc[13][11][11]=6.80953e-28;
Tnc[11][13][11]=6.80953e-28;
Tnc[9][15][11]=3.59329e-28;
Tnc[7][17][11]=9.65617e-29;
Tnc[5][19][11]=1.21405e-29;
Tnc[19][3][13]=1.57398e-30;
Tnc[17][5][13]=2.62293e-29;
Tnc[15][7][13]=1.66894e-28;
Tnc[13][9][13]=4.81788e-28;
Tnc[11][11][13]=6.80953e-28;
Tnc[9][13][13]=4.81788e-28;
Tnc[7][15][13]=1.66894e-28;
Tnc[5][17][13]=2.62293e-29;
Tnc[3][19][13]=1.57398e-30;
Tnc[19][1][15]=4.57538e-32;
Tnc[17][3][15]=2.5362e-30;
Tnc[15][5][15]=3.38111e-29;
Tnc[13][7][15]=1.66894e-28;
Tnc[11][9][15]=3.59329e-28;
Tnc[9][11][15]=3.59329e-28;
Tnc[7][13][15]=1.66894e-28;
Tnc[5][15][15]=3.38111e-29;
Tnc[3][17][15]=2.5362e-30;
Tnc[1][19][15]=4.57538e-32;
Tnc[17][1][17]=5.71923e-32;
Tnc[15][3][17]=2.5362e-30;
Tnc[13][5][17]=2.62293e-29;
Tnc[11][7][17]=9.65617e-29;
Tnc[9][9][17]=1.47094e-28;
Tnc[7][11][17]=9.65617e-29;
Tnc[5][13][17]=2.62293e-29;
Tnc[3][15][17]=2.5362e-30;
Tnc[1][17][17]=5.71923e-32;
Tnc[15][1][19]=4.57538e-32;
Tnc[13][3][19]=1.57398e-30;
Tnc[11][5][19]=1.21405e-29;
Tnc[9][7][19]=3.16223e-29;
Tnc[7][9][19]=3.16223e-29;
Tnc[5][11][19]=1.21405e-29;
Tnc[3][13][19]=1.57398e-30;
Tnc[1][15][19]=4.57538e-32;
Tnc[18][16][0]=5.71923e-32;
Tnc[16][18][0]=5.71923e-32;
Tnc[18][14][2]=6.71235e-30;
Tnc[16][16][2]=8.50819e-30;
Tnc[14][18][2]=6.71235e-30;
Tnc[18][12][4]=1.00261e-28;
Tnc[16][14][4]=1.66785e-28;
Tnc[14][16][4]=1.66785e-28;
Tnc[12][18][4]=1.00261e-28;
Tnc[18][10][6]=4.37222e-28;
Tnc[16][12][6]=9.98824e-28;
Tnc[14][14][6]=1.31084e-27;
Tnc[12][16][6]=9.98824e-28;
Tnc[10][18][6]=4.37222e-28;
Tnc[18][8][8]=7.00612e-28;
Tnc[16][10][8]=2.33948e-27;
Tnc[14][12][8]=4.21641e-27;
Tnc[12][14][8]=4.21641e-27;
Tnc[10][16][8]=2.33948e-27;
Tnc[8][18][8]=7.00612e-28;
Tnc[18][6][10]=4.37222e-28;
Tnc[16][8][10]=2.33948e-27;
Tnc[14][10][10]=6.16308e-27;
Tnc[12][12][10]=8.46369e-27;
Tnc[10][14][10]=6.16308e-27;
Tnc[8][16][10]=2.33948e-27;
Tnc[6][18][10]=4.37222e-28;
Tnc[18][4][12]=1.00261e-28;
Tnc[16][6][12]=9.98824e-28;
Tnc[14][8][12]=4.21641e-27;
Tnc[12][10][12]=8.46369e-27;
Tnc[10][12][12]=8.46369e-27;
Tnc[8][14][12]=4.21641e-27;
Tnc[6][16][12]=9.98824e-28;
Tnc[4][18][12]=1.00261e-28;
Tnc[18][2][14]=6.71235e-30;
Tnc[16][4][14]=1.66785e-28;
Tnc[14][6][14]=1.31084e-27;
Tnc[12][8][14]=4.21641e-27;
Tnc[10][10][14]=6.16308e-27;
Tnc[8][12][14]=4.21641e-27;
Tnc[6][14][14]=1.31084e-27;
Tnc[4][16][14]=1.66785e-28;
Tnc[2][18][14]=6.71235e-30;
Tnc[18][0][16]=5.71923e-32;
Tnc[16][2][16]=8.50819e-30;
Tnc[14][4][16]=1.66785e-28;
Tnc[12][6][16]=9.98824e-28;
Tnc[10][8][16]=2.33948e-27;
Tnc[8][10][16]=2.33948e-27;
Tnc[6][12][16]=9.98824e-28;
Tnc[4][14][16]=1.66785e-28;
Tnc[2][16][16]=8.50819e-30;
Tnc[0][18][16]=5.71923e-32;
Tnc[16][0][18]=5.71923e-32;
Tnc[14][2][18]=6.71235e-30;
Tnc[12][4][18]=1.00261e-28;
Tnc[10][6][18]=4.37222e-28;
Tnc[8][8][18]=7.00612e-28;
Tnc[6][10][18]=4.37222e-28;
Tnc[4][12][18]=1.00261e-28;
Tnc[2][14][18]=6.71235e-30;
Tnc[0][16][18]=5.71923e-32;
Tnc[19][13][1]=1.05503e-29;
Tnc[17][15][1]=1.69558e-29;
Tnc[15][17][1]=1.69558e-29;
Tnc[13][19][1]=1.05503e-29;
Tnc[19][11][3]=2.70111e-28;
Tnc[17][13][3]=5.81585e-28;
Tnc[15][15][3]=7.48882e-28;
Tnc[13][17][3]=5.81585e-28;
Tnc[11][19][3]=2.70111e-28;
Tnc[19][9][5]=1.47227e-27;
Tnc[17][11][5]=4.47766e-27;
Tnc[15][13][5]=7.72439e-27;
Tnc[13][15][5]=7.72439e-27;
Tnc[11][17][5]=4.47766e-27;
Tnc[9][19][5]=1.47227e-27;
Tnc[19][7][7]=2.5164e-27;
Tnc[17][9][7]=1.16526e-26;
Tnc[15][11][7]=2.8394e-26;
Tnc[13][13][7]=3.80398e-26;
Tnc[11][15][7]=2.8394e-26;
Tnc[9][17][7]=1.16526e-26;
Tnc[7][19][7]=2.5164e-27;
Tnc[19][5][9]=1.47227e-27;
Tnc[17][7][9]=1.16526e-26;
Tnc[15][9][9]=4.32318e-26;
Tnc[13][11][9]=8.181e-26;
Tnc[11][13][9]=8.181e-26;
Tnc[9][15][9]=4.32318e-26;
Tnc[7][17][9]=1.16526e-26;
Tnc[5][19][9]=1.47227e-27;
Tnc[19][3][11]=2.70111e-28;
Tnc[17][5][11]=4.47766e-27;
Tnc[15][7][11]=2.8394e-26;
Tnc[13][9][11]=8.181e-26;
Tnc[11][11][11]=1.15557e-25;
Tnc[9][13][11]=8.181e-26;
Tnc[7][15][11]=2.8394e-26;
Tnc[5][17][11]=4.47766e-27;
Tnc[3][19][11]=2.70111e-28;
Tnc[19][1][13]=1.05503e-29;
Tnc[17][3][13]=5.81585e-28;
Tnc[15][5][13]=7.72439e-27;
Tnc[13][7][13]=3.80398e-26;
Tnc[11][9][13]=8.181e-26;
Tnc[9][11][13]=8.181e-26;
Tnc[7][13][13]=3.80398e-26;
Tnc[5][15][13]=7.72439e-27;
Tnc[3][17][13]=5.81585e-28;
Tnc[1][19][13]=1.05503e-29;
Tnc[17][1][15]=1.69558e-29;
Tnc[15][3][15]=7.48882e-28;
Tnc[13][5][15]=7.72439e-27;
Tnc[11][7][15]=2.8394e-26;
Tnc[9][9][15]=4.32318e-26;
Tnc[7][11][15]=2.8394e-26;
Tnc[5][13][15]=7.72439e-27;
Tnc[3][15][15]=7.48882e-28;
Tnc[1][17][15]=1.69558e-29;
Tnc[15][1][17]=1.69558e-29;
Tnc[13][3][17]=5.81585e-28;
Tnc[11][5][17]=4.47766e-27;
Tnc[9][7][17]=1.16526e-26;
Tnc[7][9][17]=1.16526e-26;
Tnc[5][11][17]=4.47766e-27;
Tnc[3][13][17]=5.81585e-28;
Tnc[1][15][17]=1.69558e-29;
Tnc[13][1][19]=1.05503e-29;
Tnc[11][3][19]=2.70111e-28;
Tnc[9][5][19]=1.47227e-27;
Tnc[7][7][19]=2.5164e-27;
Tnc[5][9][19]=1.47227e-27;
Tnc[3][11][19]=2.70111e-28;
Tnc[1][13][19]=1.05503e-29;
Tnc[18][14][0]=1.50718e-29;
Tnc[16][16][0]=1.90753e-29;
Tnc[14][18][0]=1.50718e-29;
Tnc[18][12][2]=1.34384e-27;
Tnc[16][14][2]=2.22969e-27;
Tnc[14][16][2]=2.22969e-27;
Tnc[12][18][2]=1.34384e-27;
Tnc[18][10][4]=1.45893e-26;
Tnc[16][12][4]=3.32156e-26;
Tnc[14][14][4]=4.35443e-26;
Tnc[12][16][4]=3.32156e-26;
Tnc[10][18][4]=1.45893e-26;
Tnc[18][8][6]=4.34867e-26;
Tnc[16][10][6]=1.44628e-25;
Tnc[14][12][6]=2.60166e-25;
Tnc[12][14][6]=2.60166e-25;
Tnc[10][16][6]=1.44628e-25;
Tnc[8][18][6]=4.34867e-26;
Tnc[18][6][8]=4.34867e-26;
Tnc[16][8][8]=2.3164e-25;
Tnc[14][10][8]=6.08691e-25;
Tnc[12][12][8]=8.35231e-25;
Tnc[10][14][8]=6.08691e-25;
Tnc[8][16][8]=2.3164e-25;
Tnc[6][18][8]=4.34867e-26;
Tnc[18][4][10]=1.45893e-26;
Tnc[16][6][10]=1.44628e-25;
Tnc[14][8][10]=6.08691e-25;
Tnc[12][10][10]=1.22009e-24;
Tnc[10][12][10]=1.22009e-24;
Tnc[8][14][10]=6.08691e-25;
Tnc[6][16][10]=1.44628e-25;
Tnc[4][18][10]=1.45893e-26;
Tnc[18][2][12]=1.34384e-27;
Tnc[16][4][12]=3.32156e-26;
Tnc[14][6][12]=2.60166e-25;
Tnc[12][8][12]=8.35231e-25;
Tnc[10][10][12]=1.22009e-24;
Tnc[8][12][12]=8.35231e-25;
Tnc[6][14][12]=2.60166e-25;
Tnc[4][16][12]=3.32156e-26;
Tnc[2][18][12]=1.34384e-27;
Tnc[18][0][14]=1.50718e-29;
Tnc[16][2][14]=2.22969e-27;
Tnc[14][4][14]=4.35443e-26;
Tnc[12][6][14]=2.60166e-25;
Tnc[10][8][14]=6.08691e-25;
Tnc[8][10][14]=6.08691e-25;
Tnc[6][12][14]=2.60166e-25;
Tnc[4][14][14]=4.35443e-26;
Tnc[2][16][14]=2.22969e-27;
Tnc[0][18][14]=1.50718e-29;
Tnc[16][0][16]=1.90753e-29;
Tnc[14][2][16]=2.22969e-27;
Tnc[12][4][16]=3.32156e-26;
Tnc[10][6][16]=1.44628e-25;
Tnc[8][8][16]=2.3164e-25;
Tnc[6][10][16]=1.44628e-25;
Tnc[4][12][16]=3.32156e-26;
Tnc[2][14][16]=2.22969e-27;
Tnc[0][16][16]=1.90753e-29;
Tnc[14][0][18]=1.50718e-29;
Tnc[12][2][18]=1.34384e-27;
Tnc[10][4][18]=1.45893e-26;
Tnc[8][6][18]=4.34867e-26;
Tnc[6][8][18]=4.34867e-26;
Tnc[4][10][18]=1.45893e-26;
Tnc[2][12][18]=1.34384e-27;
Tnc[0][14][18]=1.50718e-29;
Tnc[19][11][1]=1.82784e-27;
Tnc[17][13][1]=3.9168e-27;
Tnc[15][15][1]=5.03588e-27;
Tnc[13][17][1]=3.9168e-27;
Tnc[11][19][1]=1.82784e-27;
Tnc[19][9][3]=3.30672e-26;
Tnc[17][11][3]=1.00009e-25;
Tnc[15][13][3]=1.72078e-25;
Tnc[13][15][3]=1.72078e-25;
Tnc[11][17][3]=1.00009e-25;
Tnc[9][19][3]=3.30672e-26;
Tnc[19][7][5]=1.18268e-25;
Tnc[17][9][5]=5.44273e-25;
Tnc[15][11][5]=1.32174e-24;
Tnc[13][13][5]=1.76882e-24;
Tnc[11][15][5]=1.32174e-24;
Tnc[9][17][5]=5.44273e-25;
Tnc[7][19][5]=1.18268e-25;
Tnc[19][5][7]=1.18268e-25;
Tnc[17][7][7]=9.29809e-25;
Tnc[15][9][7]=3.43579e-24;
Tnc[13][11][7]=6.48942e-24;
Tnc[11][13][7]=6.48942e-24;
Tnc[9][15][7]=3.43579e-24;
Tnc[7][17][7]=9.29809e-25;
Tnc[5][19][7]=1.18268e-25;
Tnc[19][3][9]=3.30672e-26;
Tnc[17][5][9]=5.44273e-25;
Tnc[15][7][9]=3.43579e-24;
Tnc[13][9][9]=9.8744e-24;
Tnc[11][11][9]=1.39364e-23;
Tnc[9][13][9]=9.8744e-24;
Tnc[7][15][9]=3.43579e-24;
Tnc[5][17][9]=5.44273e-25;
Tnc[3][19][9]=3.30672e-26;
Tnc[19][1][11]=1.82784e-27;
Tnc[17][3][11]=1.00009e-25;
Tnc[15][5][11]=1.32174e-24;
Tnc[13][7][11]=6.48942e-24;
Tnc[11][9][11]=1.39364e-23;
Tnc[9][11][11]=1.39364e-23;
Tnc[7][13][11]=6.48942e-24;
Tnc[5][15][11]=1.32174e-24;
Tnc[3][17][11]=1.00009e-25;
Tnc[1][19][11]=1.82784e-27;
Tnc[17][1][13]=3.9168e-27;
Tnc[15][3][13]=1.72078e-25;
Tnc[13][5][13]=1.76882e-24;
Tnc[11][7][13]=6.48942e-24;
Tnc[9][9][13]=9.8744e-24;
Tnc[7][11][13]=6.48942e-24;
Tnc[5][13][13]=1.76882e-24;
Tnc[3][15][13]=1.72078e-25;
Tnc[1][17][13]=3.9168e-27;
Tnc[15][1][15]=5.03588e-27;
Tnc[13][3][15]=1.72078e-25;
Tnc[11][5][15]=1.32174e-24;
Tnc[9][7][15]=3.43579e-24;
Tnc[7][9][15]=3.43579e-24;
Tnc[5][11][15]=1.32174e-24;
Tnc[3][13][15]=1.72078e-25;
Tnc[1][15][15]=5.03588e-27;
Tnc[13][1][17]=3.9168e-27;
Tnc[11][3][17]=1.00009e-25;
Tnc[9][5][17]=5.44273e-25;
Tnc[7][7][17]=9.29809e-25;
Tnc[5][9][17]=5.44273e-25;
Tnc[3][11][17]=1.00009e-25;
Tnc[1][13][17]=3.9168e-27;
Tnc[11][1][19]=1.82784e-27;
Tnc[9][3][19]=3.30672e-26;
Tnc[7][5][19]=1.18268e-25;
Tnc[5][7][19]=1.18268e-25;
Tnc[3][9][19]=3.30672e-26;
Tnc[1][11][19]=1.82784e-27;
Tnc[18][12][0]=3.0464e-27;
Tnc[16][14][0]=5.03588e-27;
Tnc[14][16][0]=5.03588e-27;
Tnc[12][18][0]=3.0464e-27;
Tnc[18][10][2]=1.97406e-25;
Tnc[16][12][2]=4.47298e-25;
Tnc[14][14][2]=5.85505e-25;
Tnc[12][16][2]=4.47298e-25;
Tnc[10][18][2]=1.97406e-25;
Tnc[18][8][4]=1.46481e-24;
Tnc[16][10][4]=4.84455e-24;
Tnc[14][12][4]=8.69215e-24;
Tnc[12][14][4]=8.69215e-24;
Tnc[10][16][4]=4.84455e-24;
Tnc[8][18][4]=1.46481e-24;
Tnc[18][6][6]=2.72473e-24;
Tnc[16][8][6]=1.4424e-23;
Tnc[14][10][6]=3.77738e-23;
Tnc[12][12][6]=5.17761e-23;
Tnc[10][14][6]=3.77738e-23;
Tnc[8][16][6]=1.4424e-23;
Tnc[6][18][6]=2.72473e-24;
Tnc[18][4][8]=1.46481e-24;
Tnc[16][6][8]=1.4424e-23;
Tnc[14][8][8]=6.04614e-23;
Tnc[12][10][8]=1.20962e-22;
Tnc[10][12][8]=1.20962e-22;
Tnc[8][14][8]=6.04614e-23;
Tnc[6][16][8]=1.4424e-23;
Tnc[4][18][8]=1.46481e-24;
Tnc[18][2][10]=1.97406e-25;
Tnc[16][4][10]=4.84455e-24;
Tnc[14][6][10]=3.77738e-23;
Tnc[12][8][10]=1.20962e-22;
Tnc[10][10][10]=1.76555e-22;
Tnc[8][12][10]=1.20962e-22;
Tnc[6][14][10]=3.77738e-23;
Tnc[4][16][10]=4.84455e-24;
Tnc[2][18][10]=1.97406e-25;
Tnc[18][0][12]=3.0464e-27;
Tnc[16][2][12]=4.47298e-25;
Tnc[14][4][12]=8.69215e-24;
Tnc[12][6][12]=5.17761e-23;
Tnc[10][8][12]=1.20962e-22;
Tnc[8][10][12]=1.20962e-22;
Tnc[6][12][12]=5.17761e-23;
Tnc[4][14][12]=8.69215e-24;
Tnc[2][16][12]=4.47298e-25;
Tnc[0][18][12]=3.0464e-27;
Tnc[16][0][14]=5.03588e-27;
Tnc[14][2][14]=5.85505e-25;
Tnc[12][4][14]=8.69215e-24;
Tnc[10][6][14]=3.77738e-23;
Tnc[8][8][14]=6.04614e-23;
Tnc[6][10][14]=3.77738e-23;
Tnc[4][12][14]=8.69215e-24;
Tnc[2][14][14]=5.85505e-25;
Tnc[0][16][14]=5.03588e-27;
Tnc[14][0][16]=5.03588e-27;
Tnc[12][2][16]=4.47298e-25;
Tnc[10][4][16]=4.84455e-24;
Tnc[8][6][16]=1.4424e-23;
Tnc[6][8][16]=1.4424e-23;
Tnc[4][10][16]=4.84455e-24;
Tnc[2][12][16]=4.47298e-25;
Tnc[0][14][16]=5.03588e-27;
Tnc[12][0][18]=3.0464e-27;
Tnc[10][2][18]=1.97406e-25;
Tnc[8][4][18]=1.46481e-24;
Tnc[6][6][18]=2.72473e-24;
Tnc[4][8][18]=1.46481e-24;
Tnc[2][10][18]=1.97406e-25;
Tnc[0][12][18]=3.0464e-27;
Tnc[19][9][1]=2.26652e-25;
Tnc[17][11][1]=6.79956e-25;
Tnc[15][13][1]=1.16564e-24;
Tnc[13][15][1]=1.16564e-24;
Tnc[11][17][1]=6.79956e-25;
Tnc[9][19][1]=2.26652e-25;
Tnc[19][7][3]=2.69039e-24;
Tnc[17][9][3]=1.22716e-23;
Tnc[15][11][3]=2.9659e-23;
Tnc[13][13][3]=3.96317e-23;
Tnc[11][15][3]=2.9659e-23;
Tnc[9][17][3]=1.22716e-23;
Tnc[7][19][3]=2.69039e-24;
Tnc[19][5][5]=5.62966e-24;
Tnc[17][7][5]=4.38402e-23;
Tnc[15][9][5]=1.61095e-22;
Tnc[13][11][5]=3.03484e-22;
Tnc[11][13][5]=3.03484e-22;
Tnc[9][15][5]=1.61095e-22;
Tnc[7][17][5]=4.38402e-23;
Tnc[5][19][5]=5.62966e-24;
Tnc[19][3][7]=2.69039e-24;
Tnc[17][5][7]=4.38402e-23;
Tnc[15][7][7]=2.75032e-22;
Tnc[13][9][7]=7.87749e-22;
Tnc[11][11][7]=1.11059e-21;
Tnc[9][13][7]=7.87749e-22;
Tnc[7][15][7]=2.75032e-22;
Tnc[5][17][7]=4.38402e-23;
Tnc[3][19][7]=2.69039e-24;
Tnc[19][1][9]=2.26652e-25;
Tnc[17][3][9]=1.22716e-23;
Tnc[15][5][9]=1.61095e-22;
Tnc[13][7][9]=7.87749e-22;
Tnc[11][9][9]=1.68852e-21;
Tnc[9][11][9]=1.68852e-21;
Tnc[7][13][9]=7.87749e-22;
Tnc[5][15][9]=1.61095e-22;
Tnc[3][17][9]=1.22716e-23;
Tnc[1][19][9]=2.26652e-25;
Tnc[17][1][11]=6.79956e-25;
Tnc[15][3][11]=2.9659e-23;
Tnc[13][5][11]=3.03484e-22;
Tnc[11][7][11]=1.11059e-21;
Tnc[9][9][11]=1.68852e-21;
Tnc[7][11][11]=1.11059e-21;
Tnc[5][13][11]=3.03484e-22;
Tnc[3][15][11]=2.9659e-23;
Tnc[1][17][11]=6.79956e-25;
Tnc[15][1][13]=1.16564e-24;
Tnc[13][3][13]=3.96317e-23;
Tnc[11][5][13]=3.03484e-22;
Tnc[9][7][13]=7.87749e-22;
Tnc[7][9][13]=7.87749e-22;
Tnc[5][11][13]=3.03484e-22;
Tnc[3][13][13]=3.96317e-23;
Tnc[1][15][13]=1.16564e-24;
Tnc[13][1][15]=1.16564e-24;
Tnc[11][3][15]=2.9659e-23;
Tnc[9][5][15]=1.61095e-22;
Tnc[7][7][15]=2.75032e-22;
Tnc[5][9][15]=1.61095e-22;
Tnc[3][11][15]=2.9659e-23;
Tnc[1][13][15]=1.16564e-24;
Tnc[11][1][17]=6.79956e-25;
Tnc[9][3][17]=1.22716e-23;
Tnc[7][5][17]=4.38402e-23;
Tnc[5][7][17]=4.38402e-23;
Tnc[3][9][17]=1.22716e-23;
Tnc[1][11][17]=6.79956e-25;
Tnc[9][1][19]=2.26652e-25;
Tnc[7][3][19]=2.69039e-24;
Tnc[5][5][19]=5.62966e-24;
Tnc[3][7][19]=2.69039e-24;
Tnc[1][9][19]=2.26652e-25;
Tnc[18][10][0]=4.53304e-25;
Tnc[16][12][0]=1.01993e-24;
Tnc[14][14][0]=1.33216e-24;
Tnc[12][16][0]=1.01993e-24;
Tnc[10][18][0]=4.53304e-25;
Tnc[18][8][2]=2.00749e-23;
Tnc[16][10][2]=6.58586e-23;
Tnc[14][12][2]=1.17729e-22;
Tnc[12][14][2]=1.17729e-22;
Tnc[10][16][2]=6.58586e-23;
Tnc[8][18][2]=2.00749e-23;
Tnc[18][6][4]=9.29544e-23;
Tnc[16][8][4]=4.87719e-22;
Tnc[14][10][4]=1.27117e-21;
Tnc[12][12][4]=1.73976e-21;
Tnc[10][14][4]=1.27117e-21;
Tnc[8][16][4]=4.87719e-22;
Tnc[6][18][4]=9.29544e-23;
Tnc[18][4][6]=9.29544e-23;
Tnc[16][6][6]=9.06636e-22;
Tnc[14][8][6]=3.77921e-21;
Tnc[12][10][6]=7.54126e-21;
Tnc[10][12][6]=7.54126e-21;
Tnc[8][14][6]=3.77921e-21;
Tnc[6][16][6]=9.06636e-22;
Tnc[4][18][6]=9.29544e-23;
Tnc[18][2][8]=2.00749e-23;
Tnc[16][4][8]=4.87719e-22;
Tnc[14][6][8]=3.77921e-21;
Tnc[12][8][8]=1.20608e-20;
Tnc[10][10][8]=1.75846e-20;
Tnc[8][12][8]=1.20608e-20;
Tnc[6][14][8]=3.77921e-21;
Tnc[4][16][8]=4.87719e-22;
Tnc[2][18][8]=2.00749e-23;
Tnc[18][0][10]=4.53304e-25;
Tnc[16][2][10]=6.58586e-23;
Tnc[14][4][10]=1.27117e-21;
Tnc[12][6][10]=7.54126e-21;
Tnc[10][8][10]=1.75846e-20;
Tnc[8][10][10]=1.75846e-20;
Tnc[6][12][10]=7.54126e-21;
Tnc[4][14][10]=1.27117e-21;
Tnc[2][16][10]=6.58586e-23;
Tnc[0][18][10]=4.53304e-25;
Tnc[16][0][12]=1.01993e-24;
Tnc[14][2][12]=1.17729e-22;
Tnc[12][4][12]=1.73976e-21;
Tnc[10][6][12]=7.54126e-21;
Tnc[8][8][12]=1.20608e-20;
Tnc[6][10][12]=7.54126e-21;
Tnc[4][12][12]=1.73976e-21;
Tnc[2][14][12]=1.17729e-22;
Tnc[0][16][12]=1.01993e-24;
Tnc[14][0][14]=1.33216e-24;
Tnc[12][2][14]=1.17729e-22;
Tnc[10][4][14]=1.27117e-21;
Tnc[8][6][14]=3.77921e-21;
Tnc[6][8][14]=3.77921e-21;
Tnc[4][10][14]=1.27117e-21;
Tnc[2][12][14]=1.17729e-22;
Tnc[0][14][14]=1.33216e-24;
Tnc[12][0][16]=1.01993e-24;
Tnc[10][2][16]=6.58586e-23;
Tnc[8][4][16]=4.87719e-22;
Tnc[6][6][16]=9.06636e-22;
Tnc[4][8][16]=4.87719e-22;
Tnc[2][10][16]=6.58586e-23;
Tnc[0][12][16]=1.01993e-24;
Tnc[10][0][18]=4.53304e-25;
Tnc[8][2][18]=2.00749e-23;
Tnc[6][4][18]=9.29544e-23;
Tnc[4][6][18]=9.29544e-23;
Tnc[2][8][18]=2.00749e-23;
Tnc[0][10][18]=4.53304e-25;
Tnc[19][7][1]=1.87797e-23;
Tnc[17][9][1]=8.45088e-23;
Tnc[15][11][1]=2.02821e-22;
Tnc[13][13][1]=2.70428e-22;
Tnc[11][15][1]=2.02821e-22;
Tnc[9][17][1]=8.45088e-23;
Tnc[7][19][1]=1.87797e-23;
Tnc[19][5][3]=1.30407e-22;
Tnc[17][7][3]=1.0011e-21;
Tnc[15][9][3]=3.64905e-21;
Tnc[13][11][3]=6.84911e-21;
Tnc[11][13][3]=6.84911e-21;
Tnc[9][15][3]=3.64905e-21;
Tnc[7][17][3]=1.0011e-21;
Tnc[5][19][3]=1.30407e-22;
Tnc[19][3][5]=1.30407e-22;
Tnc[17][5][5]=2.09345e-21;
Tnc[15][7][5]=1.3017e-20;
Tnc[13][9][5]=3.71059e-20;
Tnc[11][11][5]=5.22342e-20;
Tnc[9][13][5]=3.71059e-20;
Tnc[7][15][5]=1.3017e-20;
Tnc[5][17][5]=2.09345e-21;
Tnc[3][19][5]=1.30407e-22;
Tnc[19][1][7]=1.87797e-23;
Tnc[17][3][7]=1.0011e-21;
Tnc[15][5][7]=1.3017e-20;
Tnc[13][7][7]=6.32972e-20;
Tnc[11][9][7]=1.35323e-19;
Tnc[9][11][7]=1.35323e-19;
Tnc[7][13][7]=6.32972e-20;
Tnc[5][15][7]=1.3017e-20;
Tnc[3][17][7]=1.0011e-21;
Tnc[1][19][7]=1.87797e-23;
Tnc[17][1][9]=8.45088e-23;
Tnc[15][3][9]=3.64905e-21;
Tnc[13][5][9]=3.71059e-20;
Tnc[11][7][9]=1.35323e-19;
Tnc[9][9][9]=2.05517e-19;
Tnc[7][11][9]=1.35323e-19;
Tnc[5][13][9]=3.71059e-20;
Tnc[3][15][9]=3.64905e-21;
Tnc[1][17][9]=8.45088e-23;
Tnc[15][1][11]=2.02821e-22;
Tnc[13][3][11]=6.84911e-21;
Tnc[11][5][11]=5.22342e-20;
Tnc[9][7][11]=1.35323e-19;
Tnc[7][9][11]=1.35323e-19;
Tnc[5][11][11]=5.22342e-20;
Tnc[3][13][11]=6.84911e-21;
Tnc[1][15][11]=2.02821e-22;
Tnc[13][1][13]=2.70428e-22;
Tnc[11][3][13]=6.84911e-21;
Tnc[9][5][13]=3.71059e-20;
Tnc[7][7][13]=6.32972e-20;
Tnc[5][9][13]=3.71059e-20;
Tnc[3][11][13]=6.84911e-21;
Tnc[1][13][13]=2.70428e-22;
Tnc[11][1][15]=2.02821e-22;
Tnc[9][3][15]=3.64905e-21;
Tnc[7][5][15]=1.3017e-20;
Tnc[5][7][15]=1.3017e-20;
Tnc[3][9][15]=3.64905e-21;
Tnc[1][11][15]=2.02821e-22;
Tnc[9][1][17]=8.45088e-23;
Tnc[7][3][17]=1.0011e-21;
Tnc[5][5][17]=2.09345e-21;
Tnc[3][7][17]=1.0011e-21;
Tnc[1][9][17]=8.45088e-23;
Tnc[7][1][19]=1.87797e-23;
Tnc[5][3][19]=1.30407e-22;
Tnc[3][5][19]=1.30407e-22;
Tnc[1][7][19]=1.87797e-23;
Tnc[18][8][0]=4.69493e-23;
Tnc[16][10][0]=1.52116e-22;
Tnc[14][12][0]=2.70428e-22;
Tnc[12][14][0]=2.70428e-22;
Tnc[10][16][0]=1.52116e-22;
Tnc[8][18][0]=4.69493e-23;
Tnc[18][6][2]=1.29725e-21;
Tnc[16][8][2]=6.7152e-21;
Tnc[14][10][2]=1.73802e-20;
Tnc[12][12][2]=2.37353e-20;
Tnc[10][14][2]=1.73802e-20;
Tnc[8][16][2]=6.7152e-21;
Tnc[6][18][2]=1.29725e-21;
Tnc[18][4][4]=3.22906e-21;
Tnc[16][6][4]=3.10472e-20;
Tnc[14][8][4]=1.28375e-19;
Tnc[12][10][4]=2.55227e-19;
Tnc[10][12][4]=2.55227e-19;
Tnc[8][14][4]=1.28375e-19;
Tnc[6][16][4]=3.10472e-20;
Tnc[4][18][4]=3.22906e-21;
Tnc[18][2][6]=1.29725e-21;
Tnc[16][4][6]=3.10472e-20;
Tnc[14][6][6]=2.38442e-19;
Tnc[12][8][6]=7.57325e-19;
Tnc[10][10][6]=1.10252e-18;
Tnc[8][12][6]=7.57325e-19;
Tnc[6][14][6]=2.38442e-19;
Tnc[4][16][6]=3.10472e-20;
Tnc[2][18][6]=1.29725e-21;
Tnc[18][0][8]=4.69493e-23;
Tnc[16][2][8]=6.7152e-21;
Tnc[14][4][8]=1.28375e-19;
Tnc[12][6][8]=7.57325e-19;
Tnc[10][8][8]=1.76133e-18;
Tnc[8][10][8]=1.76133e-18;
Tnc[6][12][8]=7.57325e-19;
Tnc[4][14][8]=1.28375e-19;
Tnc[2][16][8]=6.7152e-21;
Tnc[0][18][8]=4.69493e-23;
Tnc[16][0][10]=1.52116e-22;
Tnc[14][2][10]=1.73802e-20;
Tnc[12][4][10]=2.55227e-19;
Tnc[10][6][10]=1.10252e-18;
Tnc[8][8][10]=1.76133e-18;
Tnc[6][10][10]=1.10252e-18;
Tnc[4][12][10]=2.55227e-19;
Tnc[2][14][10]=1.73802e-20;
Tnc[0][16][10]=1.52116e-22;
Tnc[14][0][12]=2.70428e-22;
Tnc[12][2][12]=2.37353e-20;
Tnc[10][4][12]=2.55227e-19;
Tnc[8][6][12]=7.57325e-19;
Tnc[6][8][12]=7.57325e-19;
Tnc[4][10][12]=2.55227e-19;
Tnc[2][12][12]=2.37353e-20;
Tnc[0][14][12]=2.70428e-22;
Tnc[12][0][14]=2.70428e-22;
Tnc[10][2][14]=1.73802e-20;
Tnc[8][4][14]=1.28375e-19;
Tnc[6][6][14]=2.38442e-19;
Tnc[4][8][14]=1.28375e-19;
Tnc[2][10][14]=1.73802e-20;
Tnc[0][12][14]=2.70428e-22;
Tnc[10][0][16]=1.52116e-22;
Tnc[8][2][16]=6.7152e-21;
Tnc[6][4][16]=3.10472e-20;
Tnc[4][6][16]=3.10472e-20;
Tnc[2][8][16]=6.7152e-21;
Tnc[0][10][16]=1.52116e-22;
Tnc[8][0][18]=4.69493e-23;
Tnc[6][2][18]=1.29725e-21;
Tnc[4][4][18]=3.22906e-21;
Tnc[2][6][18]=1.29725e-21;
Tnc[0][8][18]=4.69493e-23;
Tnc[19][5][1]=9.36097e-22;
Tnc[17][7][1]=7.02073e-21;
Tnc[15][9][1]=2.52746e-20;
Tnc[13][11][1]=4.71793e-20;
Tnc[11][13][1]=4.71793e-20;
Tnc[9][15][1]=2.52746e-20;
Tnc[7][17][1]=7.02073e-21;
Tnc[5][19][1]=9.36097e-22;
Tnc[19][3][3]=3.10614e-21;
Tnc[17][5][3]=4.86771e-20;
Tnc[15][7][3]=2.98615e-19;
Tnc[13][9][3]=8.45296e-19;
Tnc[11][11][3]=1.18735e-18;
Tnc[9][13][3]=8.45296e-19;
Tnc[7][15][3]=2.98615e-19;
Tnc[5][17][3]=4.86771e-20;
Tnc[3][19][3]=3.10614e-21;
Tnc[19][1][5]=9.36097e-22;
Tnc[17][3][5]=4.86771e-20;
Tnc[15][5][5]=6.23917e-19;
Tnc[13][7][5]=3.00947e-18;
Tnc[11][9][5]=6.41031e-18;
Tnc[9][11][5]=6.41031e-18;
Tnc[7][13][5]=3.00947e-18;
Tnc[5][15][5]=6.23917e-19;
Tnc[3][17][5]=4.86771e-20;
Tnc[1][19][5]=9.36097e-22;
Tnc[17][1][7]=7.02073e-21;
Tnc[15][3][7]=2.98615e-19;
Tnc[13][5][7]=3.00947e-18;
Tnc[11][7][7]=1.0923e-17;
Tnc[9][9][7]=1.65638e-17;
Tnc[7][11][7]=1.0923e-17;
Tnc[5][13][7]=3.00947e-18;
Tnc[3][15][7]=2.98615e-19;
Tnc[1][17][7]=7.02073e-21;
Tnc[15][1][9]=2.52746e-20;
Tnc[13][3][9]=8.45296e-19;
Tnc[11][5][9]=6.41031e-18;
Tnc[9][7][9]=1.65638e-17;
Tnc[7][9][9]=1.65638e-17;
Tnc[5][11][9]=6.41031e-18;
Tnc[3][13][9]=8.45296e-19;
Tnc[1][15][9]=2.52746e-20;
Tnc[13][1][11]=4.71793e-20;
Tnc[11][3][11]=1.18735e-18;
Tnc[9][5][11]=6.41031e-18;
Tnc[7][7][11]=1.0923e-17;
Tnc[5][9][11]=6.41031e-18;
Tnc[3][11][11]=1.18735e-18;
Tnc[1][13][11]=4.71793e-20;
Tnc[11][1][13]=4.71793e-20;
Tnc[9][3][13]=8.45296e-19;
Tnc[7][5][13]=3.00947e-18;
Tnc[5][7][13]=3.00947e-18;
Tnc[3][9][13]=8.45296e-19;
Tnc[1][11][13]=4.71793e-20;
Tnc[9][1][15]=2.52746e-20;
Tnc[7][3][15]=2.98615e-19;
Tnc[5][5][15]=6.23917e-19;
Tnc[3][7][15]=2.98615e-19;
Tnc[1][9][15]=2.52746e-20;
Tnc[7][1][17]=7.02073e-21;
Tnc[5][3][17]=4.86771e-20;
Tnc[3][5][17]=4.86771e-20;
Tnc[1][7][17]=7.02073e-21;
Tnc[5][1][19]=9.36097e-22;
Tnc[3][3][19]=3.10614e-21;
Tnc[1][5][19]=9.36097e-22;
Tnc[18][6][0]=3.12032e-21;
Tnc[16][8][0]=1.57966e-20;
Tnc[14][10][0]=4.04394e-20;
Tnc[12][12][0]=5.50425e-20;
Tnc[10][14][0]=4.04394e-20;
Tnc[8][16][0]=1.57966e-20;
Tnc[6][18][0]=3.12032e-21;
Tnc[18][4][2]=4.63368e-20;
Tnc[16][6][2]=4.35285e-19;
Tnc[14][8][2]=1.77765e-18;
Tnc[12][10][2]=3.51486e-18;
Tnc[10][12][2]=3.51486e-18;
Tnc[8][14][2]=1.77765e-18;
Tnc[6][16][2]=4.35285e-19;
Tnc[4][18][2]=4.63368e-20;
Tnc[18][2][4]=4.63368e-20;
Tnc[16][4][4]=1.08255e-18;
Tnc[14][6][4]=8.20251e-18;
Tnc[12][8][4]=2.58709e-17;
Tnc[10][10][4]=3.75812e-17;
Tnc[8][12][4]=2.58709e-17;
Tnc[6][14][4]=8.20251e-18;
Tnc[4][16][4]=1.08255e-18;
Tnc[2][18][4]=4.63368e-20;
Tnc[18][0][6]=3.12032e-21;
Tnc[16][2][6]=4.35285e-19;
Tnc[14][4][6]=8.20251e-18;
Tnc[12][6][6]=4.79984e-17;
Tnc[10][8][6]=1.1122e-16;
Tnc[8][10][6]=1.1122e-16;
Tnc[6][12][6]=4.79984e-17;
Tnc[4][14][6]=8.20251e-18;
Tnc[2][16][6]=4.35285e-19;
Tnc[0][18][6]=3.12032e-21;
Tnc[16][0][8]=1.57966e-20;
Tnc[14][2][8]=1.77765e-18;
Tnc[12][4][8]=2.58709e-17;
Tnc[10][6][8]=1.1122e-16;
Tnc[8][8][8]=1.7741e-16;
Tnc[6][10][8]=1.1122e-16;
Tnc[4][12][8]=2.58709e-17;
Tnc[2][14][8]=1.77765e-18;
Tnc[0][16][8]=1.57966e-20;
Tnc[14][0][10]=4.04394e-20;
Tnc[12][2][10]=3.51486e-18;
Tnc[10][4][10]=3.75812e-17;
Tnc[8][6][10]=1.1122e-16;
Tnc[6][8][10]=1.1122e-16;
Tnc[4][10][10]=3.75812e-17;
Tnc[2][12][10]=3.51486e-18;
Tnc[0][14][10]=4.04394e-20;
Tnc[12][0][12]=5.50425e-20;
Tnc[10][2][12]=3.51486e-18;
Tnc[8][4][12]=2.58709e-17;
Tnc[6][6][12]=4.79984e-17;
Tnc[4][8][12]=2.58709e-17;
Tnc[2][10][12]=3.51486e-18;
Tnc[0][12][12]=5.50425e-20;
Tnc[10][0][14]=4.04394e-20;
Tnc[8][2][14]=1.77765e-18;
Tnc[6][4][14]=8.20251e-18;
Tnc[4][6][14]=8.20251e-18;
Tnc[2][8][14]=1.77765e-18;
Tnc[0][10][14]=4.04394e-20;
Tnc[8][0][16]=1.57966e-20;
Tnc[6][2][16]=4.35285e-19;
Tnc[4][4][16]=1.08255e-18;
Tnc[2][6][16]=4.35285e-19;
Tnc[0][8][16]=1.57966e-20;
Tnc[6][0][18]=3.12032e-21;
Tnc[4][2][18]=4.63368e-20;
Tnc[2][4][18]=4.63368e-20;
Tnc[0][6][18]=3.12032e-21;
Tnc[19][3][1]=2.34024e-20;
Tnc[17][5][1]=3.51036e-19;
Tnc[15][7][1]=2.10622e-18;
Tnc[13][9][1]=5.89741e-18;
Tnc[11][11][1]=8.25638e-18;
Tnc[9][13][1]=5.89741e-18;
Tnc[7][15][1]=2.10622e-18;
Tnc[5][17][1]=3.51036e-19;
Tnc[3][19][1]=2.34024e-20;
Tnc[19][1][3]=2.34024e-20;
Tnc[17][3][3]=1.16374e-18;
Tnc[15][5][3]=1.45733e-17;
Tnc[13][7][3]=6.94286e-17;
Tnc[11][9][3]=1.47078e-16;
Tnc[9][11][3]=1.47078e-16;
Tnc[7][13][3]=6.94286e-17;
Tnc[5][15][3]=1.45733e-17;
Tnc[3][17][3]=1.16374e-18;
Tnc[1][19][3]=2.34024e-20;
Tnc[17][1][5]=3.51036e-19;
Tnc[15][3][5]=1.45733e-17;
Tnc[13][5][5]=1.44898e-16;
Tnc[11][7][5]=5.22243e-16;
Tnc[9][9][5]=7.90217e-16;
Tnc[7][11][5]=5.22243e-16;
Tnc[5][13][5]=1.44898e-16;
Tnc[3][15][5]=1.45733e-17;
Tnc[1][17][5]=3.51036e-19;
Tnc[15][1][7]=2.10622e-18;
Tnc[13][3][7]=6.94286e-17;
Tnc[11][5][7]=5.22243e-16;
Tnc[9][7][7]=1.34445e-15;
Tnc[7][9][7]=1.34445e-15;
Tnc[5][11][7]=5.22243e-16;
Tnc[3][13][7]=6.94286e-17;
Tnc[1][15][7]=2.10622e-18;
Tnc[13][1][9]=5.89741e-18;
Tnc[11][3][9]=1.47078e-16;
Tnc[9][5][9]=7.90217e-16;
Tnc[7][7][9]=1.34445e-15;
Tnc[5][9][9]=7.90217e-16;
Tnc[3][11][9]=1.47078e-16;
Tnc[1][13][9]=5.89741e-18;
Tnc[11][1][11]=8.25638e-18;
Tnc[9][3][11]=1.47078e-16;
Tnc[7][5][11]=5.22243e-16;
Tnc[5][7][11]=5.22243e-16;
Tnc[3][9][11]=1.47078e-16;
Tnc[1][11][11]=8.25638e-18;
Tnc[9][1][13]=5.89741e-18;
Tnc[7][3][13]=6.94286e-17;
Tnc[5][5][13]=1.44898e-16;
Tnc[3][7][13]=6.94286e-17;
Tnc[1][9][13]=5.89741e-18;
Tnc[7][1][15]=2.10622e-18;
Tnc[5][3][15]=1.45733e-17;
Tnc[3][5][15]=1.45733e-17;
Tnc[1][7][15]=2.10622e-18;
Tnc[5][1][17]=3.51036e-19;
Tnc[3][3][17]=1.16374e-18;
Tnc[1][5][17]=3.51036e-19;
Tnc[3][1][19]=2.34024e-20;
Tnc[1][3][19]=2.34024e-20;
Tnc[18][4][0]=1.17012e-19;
Tnc[16][6][0]=1.05311e-18;
Tnc[14][8][0]=4.21244e-18;
Tnc[12][10][0]=8.25638e-18;
Tnc[10][12][0]=8.25638e-18;
Tnc[8][14][0]=4.21244e-18;
Tnc[6][16][0]=1.05311e-18;
Tnc[4][18][0]=1.17012e-19;
Tnc[18][2][2]=6.97818e-19;
Tnc[16][4][2]=1.56052e-17;
Tnc[14][6][2]=1.15651e-16;
Tnc[12][8][2]=3.60814e-16;
Tnc[10][10][2]=5.22403e-16;
Tnc[8][12][2]=3.60814e-16;
Tnc[6][14][2]=1.15651e-16;
Tnc[4][16][2]=1.56052e-17;
Tnc[2][18][2]=6.97818e-19;
Tnc[18][0][4]=1.17012e-19;
Tnc[16][2][4]=1.56052e-17;
Tnc[14][4][4]=2.87288e-16;
Tnc[12][6][4]=1.66039e-15;
Tnc[10][8][4]=3.82635e-15;
Tnc[8][10][4]=3.82635e-15;
Tnc[6][12][4]=1.66039e-15;
Tnc[4][14][4]=2.87288e-16;
Tnc[2][16][4]=1.56052e-17;
Tnc[0][18][4]=1.17012e-19;
Tnc[16][0][6]=1.05311e-18;
Tnc[14][2][6]=1.15651e-16;
Tnc[12][4][6]=1.66039e-15;
Tnc[10][6][6]=7.08812e-15;
Tnc[8][8][6]=1.12818e-14;
Tnc[6][10][6]=7.08812e-15;
Tnc[4][12][6]=1.66039e-15;
Tnc[2][14][6]=1.15651e-16;
Tnc[0][16][6]=1.05311e-18;
Tnc[14][0][8]=4.21244e-18;
Tnc[12][2][8]=3.60814e-16;
Tnc[10][4][8]=3.82635e-15;
Tnc[8][6][8]=1.12818e-14;
Tnc[6][8][8]=1.12818e-14;
Tnc[4][10][8]=3.82635e-15;
Tnc[2][12][8]=3.60814e-16;
Tnc[0][14][8]=4.21244e-18;
Tnc[12][0][10]=8.25638e-18;
Tnc[10][2][10]=5.22403e-16;
Tnc[8][4][10]=3.82635e-15;
Tnc[6][6][10]=7.08812e-15;
Tnc[4][8][10]=3.82635e-15;
Tnc[2][10][10]=5.22403e-16;
Tnc[0][12][10]=8.25638e-18;
Tnc[10][0][12]=8.25638e-18;
Tnc[8][2][12]=3.60814e-16;
Tnc[6][4][12]=1.66039e-15;
Tnc[4][6][12]=1.66039e-15;
Tnc[2][8][12]=3.60814e-16;
Tnc[0][10][12]=8.25638e-18;
Tnc[8][0][14]=4.21244e-18;
Tnc[6][2][14]=1.15651e-16;
Tnc[4][4][14]=2.87288e-16;
Tnc[2][6][14]=1.15651e-16;
Tnc[0][8][14]=4.21244e-18;
Tnc[6][0][16]=1.05311e-18;
Tnc[4][2][16]=1.56052e-17;
Tnc[2][4][16]=1.56052e-17;
Tnc[0][6][16]=1.05311e-18;
Tnc[4][0][18]=1.17012e-19;
Tnc[2][2][18]=6.97818e-19;
Tnc[0][4][18]=1.17012e-19;
Tnc[19][1][1]=1.95729e-19;
Tnc[17][3][1]=8.80782e-18;
Tnc[15][5][1]=1.05694e-16;
Tnc[13][7][1]=4.93238e-16;
Tnc[11][9][1]=1.0358e-15;
Tnc[9][11][1]=1.0358e-15;
Tnc[7][13][1]=4.93238e-16;
Tnc[5][15][1]=1.05694e-16;
Tnc[3][17][1]=8.80782e-18;
Tnc[1][19][1]=1.95729e-19;
Tnc[17][1][3]=8.80782e-18;
Tnc[15][3][3]=3.49964e-16;
Tnc[13][5][3]=3.40334e-15;
Tnc[11][7][3]=1.21337e-14;
Tnc[9][9][3]=1.82991e-14;
Tnc[7][11][3]=1.21337e-14;
Tnc[5][13][3]=3.40334e-15;
Tnc[3][15][3]=3.49964e-16;
Tnc[1][17][3]=8.80782e-18;
Tnc[15][1][5]=1.05694e-16;
Tnc[13][3][5]=3.40334e-15;
Tnc[11][5][5]=2.52834e-14;
Tnc[9][7][5]=6.47326e-14;
Tnc[7][9][5]=6.47326e-14;
Tnc[5][11][5]=2.52834e-14;
Tnc[3][13][5]=3.40334e-15;
Tnc[1][15][5]=1.05694e-16;
Tnc[13][1][7]=4.93238e-16;
Tnc[11][3][7]=1.21337e-14;
Tnc[9][5][7]=6.47326e-14;
Tnc[7][7][7]=1.09893e-13;
Tnc[5][9][7]=6.47326e-14;
Tnc[3][11][7]=1.21337e-14;
Tnc[1][13][7]=4.93238e-16;
Tnc[11][1][9]=1.0358e-15;
Tnc[9][3][9]=1.82991e-14;
Tnc[7][5][9]=6.47326e-14;
Tnc[5][7][9]=6.47326e-14;
Tnc[3][9][9]=1.82991e-14;
Tnc[1][11][9]=1.0358e-15;
Tnc[9][1][11]=1.0358e-15;
Tnc[7][3][11]=1.21337e-14;
Tnc[5][5][11]=2.52834e-14;
Tnc[3][7][11]=1.21337e-14;
Tnc[1][9][11]=1.0358e-15;
Tnc[7][1][13]=4.93238e-16;
Tnc[5][3][13]=3.40334e-15;
Tnc[3][5][13]=3.40334e-15;
Tnc[1][7][13]=4.93238e-16;
Tnc[5][1][15]=1.05694e-16;
Tnc[3][3][15]=3.49964e-16;
Tnc[1][5][15]=1.05694e-16;
Tnc[3][1][17]=8.80782e-18;
Tnc[1][3][17]=8.80782e-18;
Tnc[1][1][19]=1.95729e-19;
Tnc[18][2][0]=1.95729e-18;
Tnc[16][4][0]=3.96352e-17;
Tnc[14][6][0]=2.8185e-16;
Tnc[12][8][0]=8.63167e-16;
Tnc[10][10][0]=1.24296e-15;
Tnc[8][12][0]=8.63167e-16;
Tnc[6][14][0]=2.8185e-16;
Tnc[4][16][0]=3.96352e-17;
Tnc[2][18][0]=1.95729e-18;
Tnc[18][0][2]=1.95729e-18;
Tnc[16][2][2]=2.3605e-16;
Tnc[14][4][2]=4.16434e-15;
Tnc[12][6][2]=2.35768e-14;
Tnc[10][8][2]=5.38616e-14;
Tnc[8][10][2]=5.38616e-14;
Tnc[6][12][2]=2.35768e-14;
Tnc[4][14][2]=4.16434e-15;
Tnc[2][16][2]=2.3605e-16;
Tnc[0][18][2]=1.95729e-18;
Tnc[16][0][4]=3.96352e-17;
Tnc[14][2][4]=4.16434e-15;
Tnc[12][4][4]=5.84734e-14;
Tnc[10][6][4]=2.46915e-13;
Tnc[8][8][4]=3.91705e-13;
Tnc[6][10][4]=2.46915e-13;
Tnc[4][12][4]=5.84734e-14;
Tnc[2][14][4]=4.16434e-15;
Tnc[0][16][4]=3.96352e-17;
Tnc[14][0][6]=2.8185e-16;
Tnc[12][2][6]=2.35768e-14;
Tnc[10][4][6]=2.46915e-13;
Tnc[8][6][6]=7.24011e-13;
Tnc[6][8][6]=7.24011e-13;
Tnc[4][10][6]=2.46915e-13;
Tnc[2][12][6]=2.35768e-14;
Tnc[0][14][6]=2.8185e-16;
Tnc[12][0][8]=8.63167e-16;
Tnc[10][2][8]=5.38616e-14;
Tnc[8][4][8]=3.91705e-13;
Tnc[6][6][8]=7.24011e-13;
Tnc[4][8][8]=3.91705e-13;
Tnc[2][10][8]=5.38616e-14;
Tnc[0][12][8]=8.63167e-16;
Tnc[10][0][10]=1.24296e-15;
Tnc[8][2][10]=5.38616e-14;
Tnc[6][4][10]=2.46915e-13;
Tnc[4][6][10]=2.46915e-13;
Tnc[2][8][10]=5.38616e-14;
Tnc[0][10][10]=1.24296e-15;
Tnc[8][0][12]=8.63167e-16;
Tnc[6][2][12]=2.35768e-14;
Tnc[4][4][12]=5.84734e-14;
Tnc[2][6][12]=2.35768e-14;
Tnc[0][8][12]=8.63167e-16;
Tnc[6][0][14]=2.8185e-16;
Tnc[4][2][14]=4.16434e-15;
Tnc[2][4][14]=4.16434e-15;
Tnc[0][6][14]=2.8185e-16;
Tnc[4][0][16]=3.96352e-17;
Tnc[2][2][16]=2.3605e-16;
Tnc[0][4][16]=3.96352e-17;
Tnc[2][0][18]=1.95729e-18;
Tnc[0][2][18]=1.95729e-18;
Tnc[17][1][1]=7.39857e-17;
Tnc[15][3][1]=2.66349e-15;
Tnc[13][5][1]=2.48592e-14;
Tnc[11][7][1]=8.70072e-14;
Tnc[9][9][1]=1.30511e-13;
Tnc[7][11][1]=8.70072e-14;
Tnc[5][13][1]=2.48592e-14;
Tnc[3][15][1]=2.66349e-15;
Tnc[1][17][1]=7.39857e-17;
Tnc[15][1][3]=2.66349e-15;
Tnc[13][3][3]=8.21735e-14;
Tnc[11][5][3]=5.98002e-13;
Tnc[9][7][3]=1.51779e-12;
Tnc[7][9][3]=1.51779e-12;
Tnc[5][11][3]=5.98002e-13;
Tnc[3][13][3]=8.21735e-14;
Tnc[1][15][3]=2.66349e-15;
Tnc[13][1][5]=2.48592e-14;
Tnc[11][3][5]=5.98002e-13;
Tnc[9][5][5]=3.1556e-12;
Tnc[7][7][5]=5.33934e-12;
Tnc[5][9][5]=3.1556e-12;
Tnc[3][11][5]=5.98002e-13;
Tnc[1][13][5]=2.48592e-14;
Tnc[11][1][7]=8.70072e-14;
Tnc[9][3][7]=1.51779e-12;
Tnc[7][5][7]=5.33934e-12;
Tnc[5][7][7]=5.33934e-12;
Tnc[3][9][7]=1.51779e-12;
Tnc[1][11][7]=8.70072e-14;
Tnc[9][1][9]=1.30511e-13;
Tnc[7][3][9]=1.51779e-12;
Tnc[5][5][9]=3.1556e-12;
Tnc[3][7][9]=1.51779e-12;
Tnc[1][9][9]=1.30511e-13;
Tnc[7][1][11]=8.70072e-14;
Tnc[5][3][11]=5.98002e-13;
Tnc[3][5][11]=5.98002e-13;
Tnc[1][7][11]=8.70072e-14;
Tnc[5][1][13]=2.48592e-14;
Tnc[3][3][13]=8.21735e-14;
Tnc[1][5][13]=2.48592e-14;
Tnc[3][1][15]=2.66349e-15;
Tnc[1][3][15]=2.66349e-15;
Tnc[1][1][17]=7.39857e-17;
Tnc[18][0][0]=8.22064e-18;
Tnc[16][2][0]=6.65871e-16;
Tnc[14][4][0]=1.06539e-14;
Tnc[12][6][0]=5.80048e-14;
Tnc[10][8][0]=1.30511e-13;
Tnc[8][10][0]=1.30511e-13;
Tnc[6][12][0]=5.80048e-14;
Tnc[4][14][0]=1.06539e-14;
Tnc[2][16][0]=6.65871e-16;
Tnc[0][18][0]=8.22064e-18;
Tnc[16][0][2]=6.65871e-16;
Tnc[14][2][2]=6.33318e-14;
Tnc[12][4][2]=8.53499e-13;
Tnc[10][6][2]=3.53829e-12;
Tnc[8][8][2]=5.58296e-12;
Tnc[6][10][2]=3.53829e-12;
Tnc[4][12][2]=8.53499e-13;
Tnc[2][14][2]=6.33318e-14;
Tnc[0][16][2]=6.65871e-16;
Tnc[14][0][4]=1.06539e-14;
Tnc[12][2][4]=8.53499e-13;
Tnc[10][4][4]=8.75527e-12;
Tnc[8][6][4]=2.54496e-11;
Tnc[6][8][4]=2.54496e-11;
Tnc[4][10][4]=8.75527e-12;
Tnc[2][12][4]=8.53499e-13;
Tnc[0][14][4]=1.06539e-14;
Tnc[12][0][6]=5.80048e-14;
Tnc[10][2][6]=3.53829e-12;
Tnc[8][4][6]=2.54496e-11;
Tnc[6][6][6]=4.68831e-11;
Tnc[4][8][6]=2.54496e-11;
Tnc[2][10][6]=3.53829e-12;
Tnc[0][12][6]=5.80048e-14;
Tnc[10][0][8]=1.30511e-13;
Tnc[8][2][8]=5.58296e-12;
Tnc[6][4][8]=2.54496e-11;
Tnc[4][6][8]=2.54496e-11;
Tnc[2][8][8]=5.58296e-12;
Tnc[0][10][8]=1.30511e-13;
Tnc[8][0][10]=1.30511e-13;
Tnc[6][2][10]=3.53829e-12;
Tnc[4][4][10]=8.75527e-12;
Tnc[2][6][10]=3.53829e-12;
Tnc[0][8][10]=1.30511e-13;
Tnc[6][0][12]=5.80048e-14;
Tnc[4][2][12]=8.53499e-13;
Tnc[2][4][12]=8.53499e-13;
Tnc[0][6][12]=5.80048e-14;
Tnc[4][0][14]=1.06539e-14;
Tnc[2][2][14]=6.33318e-14;
Tnc[0][4][14]=1.06539e-14;
Tnc[2][0][16]=6.65871e-16;
Tnc[0][2][16]=6.65871e-16;
Tnc[0][0][18]=8.22064e-18;
Tnc[15][1][1]=2.24917e-14;
Tnc[13][3][1]=6.29766e-13;
Tnc[11][5][1]=4.40836e-12;
Tnc[9][7][1]=1.10209e-11;
Tnc[7][9][1]=1.10209e-11;
Tnc[5][11][1]=4.40836e-12;
Tnc[3][13][1]=6.29766e-13;
Tnc[1][15][1]=2.24917e-14;
Tnc[13][1][3]=6.29766e-13;
Tnc[11][3][3]=1.45371e-11;
Tnc[9][5][3]=7.53096e-11;
Tnc[7][7][3]=1.2674e-10;
Tnc[5][9][3]=7.53096e-11;
Tnc[3][11][3]=1.45371e-11;
Tnc[1][13][3]=6.29766e-13;
Tnc[11][1][5]=4.40836e-12;
Tnc[9][3][5]=7.53096e-11;
Tnc[7][5][5]=2.62613e-10;
Tnc[5][7][5]=2.62613e-10;
Tnc[3][9][5]=7.53096e-11;
Tnc[1][11][5]=4.40836e-12;
Tnc[9][1][7]=1.10209e-11;
Tnc[7][3][7]=1.2674e-10;
Tnc[5][5][7]=2.62613e-10;
Tnc[3][7][7]=1.2674e-10;
Tnc[1][9][7]=1.10209e-11;
Tnc[7][1][9]=1.10209e-11;
Tnc[5][3][9]=7.53096e-11;
Tnc[3][5][9]=7.53096e-11;
Tnc[1][7][9]=1.10209e-11;
Tnc[5][1][11]=4.40836e-12;
Tnc[3][3][11]=1.45371e-11;
Tnc[1][5][11]=4.40836e-12;
Tnc[3][1][13]=6.29766e-13;
Tnc[1][3][13]=6.29766e-13;
Tnc[1][1][15]=2.24917e-14;
Tnc[16][0][0]=2.81146e-15;
Tnc[14][2][0]=1.79933e-13;
Tnc[12][4][0]=2.20418e-12;
Tnc[10][6][0]=8.81673e-12;
Tnc[8][8][0]=1.37761e-11;
Tnc[6][10][0]=8.81673e-12;
Tnc[4][12][0]=2.20418e-12;
Tnc[2][14][0]=1.79933e-13;
Tnc[0][16][0]=2.81146e-15;
Tnc[14][0][2]=1.79933e-13;
Tnc[12][2][2]=1.30677e-11;
Tnc[10][4][2]=1.28945e-10;
Tnc[8][6][2]=3.69201e-10;
Tnc[6][8][2]=3.69201e-10;
Tnc[4][10][2]=1.28945e-10;
Tnc[2][12][2]=1.30677e-11;
Tnc[0][14][2]=1.79933e-13;
Tnc[12][0][4]=2.20418e-12;
Tnc[10][2][4]=1.28945e-10;
Tnc[8][4][4]=9.10406e-10;
Tnc[6][6][4]=1.66809e-09;
Tnc[4][8][4]=9.10406e-10;
Tnc[2][10][4]=1.28945e-10;
Tnc[0][12][4]=2.20418e-12;
Tnc[10][0][6]=8.81673e-12;
Tnc[8][2][6]=3.69201e-10;
Tnc[6][4][6]=1.66809e-09;
Tnc[4][6][6]=1.66809e-09;
Tnc[2][8][6]=3.69201e-10;
Tnc[0][10][6]=8.81673e-12;
Tnc[8][0][8]=1.37761e-11;
Tnc[6][2][8]=3.69201e-10;
Tnc[4][4][8]=9.10406e-10;
Tnc[2][6][8]=3.69201e-10;
Tnc[0][8][8]=1.37761e-11;
Tnc[6][0][10]=8.81673e-12;
Tnc[4][2][10]=1.28945e-10;
Tnc[2][4][10]=1.28945e-10;
Tnc[0][6][10]=8.81673e-12;
Tnc[4][0][12]=2.20418e-12;
Tnc[2][2][12]=1.30677e-11;
Tnc[0][4][12]=2.20418e-12;
Tnc[2][0][14]=1.79933e-13;
Tnc[0][2][14]=1.79933e-13;
Tnc[0][0][16]=2.81146e-15;
Tnc[13][1][1]=5.35301e-12;
Tnc[11][3][1]=1.12413e-10;
Tnc[9][5][1]=5.62067e-10;
Tnc[7][7][1]=9.36778e-10;
Tnc[5][9][1]=5.62067e-10;
Tnc[3][11][1]=1.12413e-10;
Tnc[1][13][1]=5.35301e-12;
Tnc[11][1][3]=1.12413e-10;
Tnc[9][3][3]=1.84679e-09;
Tnc[7][5][3]=6.34332e-09;
Tnc[5][7][3]=6.34332e-09;
Tnc[3][9][3]=1.84679e-09;
Tnc[1][11][3]=1.12413e-10;
Tnc[9][1][5]=5.62067e-10;
Tnc[7][3][5]=6.34332e-09;
Tnc[5][5][5]=1.30721e-08;
Tnc[3][7][5]=6.34332e-09;
Tnc[1][9][5]=5.62067e-10;
Tnc[7][1][7]=9.36778e-10;
Tnc[5][3][7]=6.34332e-09;
Tnc[3][5][7]=6.34332e-09;
Tnc[1][7][7]=9.36778e-10;
Tnc[5][1][9]=5.62067e-10;
Tnc[3][3][9]=1.84679e-09;
Tnc[1][5][9]=5.62067e-10;
Tnc[3][1][11]=1.12413e-10;
Tnc[1][3][11]=1.12413e-10;
Tnc[1][1][13]=5.35301e-12;
Tnc[14][0][0]=7.64716e-13;
Tnc[12][2][0]=3.74711e-11;
Tnc[10][4][0]=3.3724e-10;
Tnc[8][6][0]=9.36778e-10;
Tnc[6][8][0]=9.36778e-10;
Tnc[4][10][0]=3.3724e-10;
Tnc[2][12][0]=3.74711e-11;
Tnc[0][14][0]=7.64716e-13;
Tnc[12][0][2]=3.74711e-11;
Tnc[10][2][2]=1.99132e-09;
Tnc[8][4][2]=1.35699e-08;
Tnc[6][6][2]=2.46239e-08;
Tnc[4][8][2]=1.35699e-08;
Tnc[2][10][2]=1.99132e-09;
Tnc[0][12][2]=3.74711e-11;
Tnc[10][0][4]=3.3724e-10;
Tnc[8][2][4]=1.35699e-08;
Tnc[6][4][4]=6.0382e-08;
Tnc[4][6][4]=6.0382e-08;
Tnc[2][8][4]=1.35699e-08;
Tnc[0][10][4]=3.3724e-10;
Tnc[8][0][6]=9.36778e-10;
Tnc[6][2][6]=2.46239e-08;
Tnc[4][4][6]=6.0382e-08;
Tnc[2][6][6]=2.46239e-08;
Tnc[0][8][6]=9.36778e-10;
Tnc[6][0][8]=9.36778e-10;
Tnc[4][2][8]=1.35699e-08;
Tnc[2][4][8]=1.35699e-08;
Tnc[0][6][8]=9.36778e-10;
Tnc[4][0][10]=3.3724e-10;
Tnc[2][2][10]=1.99132e-09;
Tnc[0][4][10]=3.3724e-10;
Tnc[2][0][12]=3.74711e-11;
Tnc[0][2][12]=3.74711e-11;
Tnc[0][0][14]=7.64716e-13;
Tnc[11][1][1]=9.63543e-10;
Tnc[9][3][1]=1.44531e-08;
Tnc[7][5][1]=4.81771e-08;
Tnc[5][7][1]=4.81771e-08;
Tnc[3][9][1]=1.44531e-08;
Tnc[1][11][1]=9.63543e-10;
Tnc[9][1][3]=1.44531e-08;
Tnc[7][3][3]=1.57379e-07;
Tnc[5][5][3]=3.21181e-07;
Tnc[3][7][3]=1.57379e-07;
Tnc[1][9][3]=1.44531e-08;
Tnc[7][1][5]=4.81771e-08;
Tnc[5][3][5]=3.21181e-07;
Tnc[3][5][5]=3.21181e-07;
Tnc[1][7][5]=4.81771e-08;
Tnc[5][1][7]=4.81771e-08;
Tnc[3][3][7]=1.57379e-07;
Tnc[1][5][7]=4.81771e-08;
Tnc[3][1][9]=1.44531e-08;
Tnc[1][3][9]=1.44531e-08;
Tnc[1][1][11]=9.63543e-10;
Tnc[12][0][0]=1.6059e-10;
Tnc[10][2][0]=5.78126e-09;
Tnc[8][4][0]=3.61328e-08;
Tnc[6][6][0]=6.42362e-08;
Tnc[4][8][0]=3.61328e-08;
Tnc[2][10][0]=5.78126e-09;
Tnc[0][12][0]=1.6059e-10;
Tnc[10][0][2]=5.78126e-09;
Tnc[8][2][2]=2.11979e-07;
Tnc[6][4][2]=9.15365e-07;
Tnc[4][6][2]=9.15365e-07;
Tnc[2][8][2]=2.11979e-07;
Tnc[0][10][2]=5.78126e-09;
Tnc[8][0][4]=3.61328e-08;
Tnc[6][2][4]=9.15365e-07;
Tnc[4][4][4]=2.22257e-06;
Tnc[2][6][4]=9.15365e-07;
Tnc[0][8][4]=3.61328e-08;
Tnc[6][0][6]=6.42362e-08;
Tnc[4][2][6]=9.15365e-07;
Tnc[2][4][6]=9.15365e-07;
Tnc[0][6][6]=6.42362e-08;
Tnc[4][0][8]=3.61328e-08;
Tnc[2][2][8]=2.11979e-07;
Tnc[0][4][8]=3.61328e-08;
Tnc[2][0][10]=5.78126e-09;
Tnc[0][2][10]=5.78126e-09;
Tnc[0][0][12]=1.6059e-10;
Tnc[9][1][1]=1.25261e-07;
Tnc[7][3][1]=1.25261e-06;
Tnc[5][5][1]=2.50521e-06;
Tnc[3][7][1]=1.25261e-06;
Tnc[1][9][1]=1.25261e-07;
Tnc[7][1][3]=1.25261e-06;
Tnc[5][3][3]=8.10018e-06;
Tnc[3][5][3]=8.10018e-06;
Tnc[1][7][3]=1.25261e-06;
Tnc[5][1][5]=2.50521e-06;
Tnc[3][3][5]=8.10018e-06;
Tnc[1][5][5]=2.50521e-06;
Tnc[3][1][7]=1.25261e-06;
Tnc[1][3][7]=1.25261e-06;
Tnc[1][1][9]=1.25261e-07;
Tnc[10][0][0]=2.50521e-08;
Tnc[8][2][0]=6.26303e-07;
Tnc[6][4][0]=2.50521e-06;
Tnc[4][6][0]=2.50521e-06;
Tnc[2][8][0]=6.26303e-07;
Tnc[0][10][0]=2.50521e-08;
Tnc[8][0][2]=6.26303e-07;
Tnc[6][2][2]=1.45302e-05;
Tnc[4][4][2]=3.45719e-05;
Tnc[2][6][2]=1.45302e-05;
Tnc[0][8][2]=6.26303e-07;
Tnc[6][0][4]=2.50521e-06;
Tnc[4][2][4]=3.45719e-05;
Tnc[2][4][4]=3.45719e-05;
Tnc[0][6][4]=2.50521e-06;
Tnc[4][0][6]=2.50521e-06;
Tnc[2][2][6]=1.45302e-05;
Tnc[0][4][6]=2.50521e-06;
Tnc[2][0][8]=6.26303e-07;
Tnc[0][2][8]=6.26303e-07;
Tnc[0][0][10]=2.50521e-08;
Tnc[7][1][1]=1.10229e-05;
Tnc[5][3][1]=6.61376e-05;
Tnc[3][5][1]=6.61376e-05;
Tnc[1][7][1]=1.10229e-05;
Tnc[5][1][3]=6.61376e-05;
Tnc[3][3][3]=0.000209436;
Tnc[1][5][3]=6.61376e-05;
Tnc[3][1][5]=6.61376e-05;
Tnc[1][3][5]=6.61376e-05;
Tnc[1][1][7]=1.10229e-05;
Tnc[8][0][0]=2.75573e-06;
Tnc[6][2][0]=4.40917e-05;
Tnc[4][4][0]=9.92063e-05;
Tnc[2][6][0]=4.40917e-05;
Tnc[0][8][0]=2.75573e-06;
Tnc[6][0][2]=4.40917e-05;
Tnc[4][2][2]=0.000562169;
Tnc[2][4][2]=0.000562169;
Tnc[0][6][2]=4.40917e-05;
Tnc[4][0][4]=9.92063e-05;
Tnc[2][2][4]=0.000562169;
Tnc[0][4][4]=9.92063e-05;
Tnc[2][0][6]=4.40917e-05;
Tnc[0][2][6]=4.40917e-05;
Tnc[0][0][8]=2.75573e-06;
Tnc[5][1][1]=0.000595238;
Tnc[3][3][1]=0.00178571;
Tnc[1][5][1]=0.000595238;
Tnc[3][1][3]=0.00178571;
Tnc[1][3][3]=0.00178571;
Tnc[1][1][5]=0.000595238;
Tnc[6][0][0]=0.000198413;
Tnc[4][2][0]=0.00178571;
Tnc[2][4][0]=0.00178571;
Tnc[0][6][0]=0.000198413;
Tnc[4][0][2]=0.00178571;
Tnc[2][2][2]=0.00952381;
Tnc[0][4][2]=0.00178571;
Tnc[2][0][4]=0.00178571;
Tnc[0][2][4]=0.00178571;
Tnc[0][0][6]=0.000198413;
Tnc[3][1][1]=0.0166667;
Tnc[1][3][1]=0.0166667;
Tnc[1][1][3]=0.0166667;
Tnc[4][0][0]=0.00833333;
Tnc[2][2][0]=0.0333333;
Tnc[0][4][0]=0.00833333;
Tnc[2][0][2]=0.0333333;
Tnc[0][2][2]=0.0333333;
Tnc[0][0][4]=0.00833333;
Tnc[1][1][1]=0.166667;
Tnc[2][0][0]=0.166667;
Tnc[0][2][0]=0.166667;
Tnc[0][0][2]=0.166667;
Tnc[0][0][0]=1;
/* end of table */
 f[0]=f[1]=f[2]=f[3]=0.0; ex[0][0][0] = 1.0;
 for (i=0; i<deg; i++) {
   for (j=0; j<deg; j++) {
     for (k=0; k<deg; k++) {
       if (i>0) ex[i][j][k] = ex[i-1][j][k]*(a*t);
       else if (j>0) ex[i][j][k] = ex[i][j-1][k]*(b*t);
       else if (k>0) ex[i][j][k] = ex[i][j][k-1]*(c*t);
       f[0] += (ex[i][j][k])*(Tnc[i][j][k]);
       if (i>0) f[1] += ((double) i)*(ex[i-1][j][k])*(Tnc[i][j][k]);
       if (j>0) f[2] += ((double) j)*(ex[i][j-1][k])*(Tnc[i][j][k]);
       if (k>0) f[3] += ((double) k)*(ex[i][j][k-1])*(Tnc[i][j][k]);
     }
   }
 }
}


