% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/row_pair_apply.R
\name{rowPairApply}
\alias{rowPairApply}
\title{Apply functions to all row pairs.}
\usage{
rowPairApply(test_data, ...)
}
\arguments{
\item{test_data}{The data to apply the functions to as a matrix or
data.frame.  Heuristics must have already been fitted to trying data and
must include the same criterion_col and cols_to_fit.}

\item{...}{The functions that generate the functions to apply, such as
heuristics(ttb) and correctGreater(col)-- see example below.}
}
\value{
A matrix of outputs from the functions.  The number of rows is based
  on the number of row pairs in test_data.  If the input has N rows, the
  output will have N x (N-1) rows.  The number of columns will be at least
  the number of functions but may be more as some functions may output more
  than one column.
}
\description{
Apply functions like heuristic predictions to all row pairs in a matrix
or data.frame.  This does not accept arbitrary functions-- they must be
functions designed to be run by rowPairApply.
}
\examples{
## Fit two models to data.
data <- cbind(y=c(30,20,10,5), x1=c(1,1,0,0), x2=c(1,1,0,1))
ttb <- ttbModel(data, 1, c(2:ncol(data)))
lreg <- logRegModel(data, 1, c(2:ncol(data)))

## Generate predictions for all row pairs for these two models:
rowPairApply(data, heuristics(ttb, lreg))
## Returns a matrix of 2 columns, named ttbModel and regModel, and 6 rows.
## The original data had 4 rows, meaning there are 4*3/2 = 6 row pairs.

## To see which row pair is which row, use rowIndexes:
rowPairApply(data, rowIndexes(), heuristics(ttb, lreg))
## Returns a matrix with columns Row1, Row2, ttbModel, logRegModel.
## (RowIndexes returns *two* columns.)

## To see whether the first row was actually greater than the second in the
## row pair, use correctGreater and give it the criterion column index, in
## this case 1.
rowPairApply(data, heuristics(lreg, ttb), correctGreater(1))
## Returns a matrix with columns logRegModel, ttbModel,
## CorrectGreater.  Values are -1, 0, or 1.

## To do the same analysis for the *probabilty* that the first row is
## greater. use heuristicsProb and probGreater.  Warning: Not all heuristica
## models have implemented the prob greater function.
rowPairApply(data, heuristicsProb(lreg, ttb), probGreater(1))
## Returns a matrix with columns logRegModel, ttbModel, ProbGreater.
## Values range from 0.0 to 1.0.

}
\seealso{
\code{\link{heuristics}} and \code{\link{heuristics}} to wrap heuristics
to be applied.

\code{\link{rowIndexes}} to get apply to output row indexes for the pair.

\code{\link{correctGreater}} to get the correct output based on the criterion column.
(CorrectGreater should be used with heuristics while probGreater should be used with
heuristicsProb.)
}
