% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hetGP.R
\name{mleHomGP}
\alias{mleHomGP}
\title{Gaussian process modeling with homogeneous noise}
\usage{
mleHomGP(X, Z, lower, upper, known = NULL, noiseControl = list(g_bounds =
  c(sqrt(.Machine$double.eps), 100)), init = NULL, covtype = c("Gaussian",
  "Matern5_2", "Matern3_2"), maxit = 100, eps = sqrt(.Machine$double.eps),
  settings = list(return.Ki = TRUE))
}
\arguments{
\item{X}{matrix of all designs, one per row, or list with elements:
\itemize{
  \item \code{X0} matrix of unique design locations, one point per row
  \item \code{Z0} vector of averaged observations, of length \code{nrow(X0)}
  \item \code{mult} number of replicates at designs in \code{X0}, of length \code{nrow(X0)}
}}

\item{Z}{vector of all observations. If using a list with \code{X}, \code{Z} has to be ordered with respect to \code{X0}, and of length \code{sum(mult)}}

\item{lower, upper}{bounds for the \code{theta} parameter (see \code{\link[hetGP]{cov_gen}} for the exact parameterization).
In the multivariate case, it is possible to give vectors for bounds (resp. scalars) for anisotropy (resp. isotropy)}

\item{known}{optional list of known parameters, e.g., \code{beta0}, \code{theta} or \code{g}}

\item{noiseControl}{list with element , 
\itemize{
\item \code{g_bounds}, vector providing minimal and maximal noise to signal ratio
}}

\item{init}{optional list specifying starting values for MLE optimization, with elements:
\itemize{
 \item \code{theta_init} initial value of the theta parameters to be optimized over (default to \code{(upper-lower)/2})
 \item \code{g_init} initial value of the nugget parameter to be optimized over (default to \code{0.5})
}}

\item{covtype}{covariance kernel type, either 'Gaussian', 'Matern5_2' or 'Matern3_2', see \code{\link[hetGP]{cov_gen}}}

\item{maxit}{maximum number of iteration for L-BFGS-B of \code{\link[stats]{optim}}}

\item{eps}{jitter used in the inversion of the covariance matrix for numerical stability}

\item{settings}{list with argument \code{return.Ki}, to include the inverse covariance matrix in the object for further use (e.g., prediction).}
}
\value{
a list which is given the S3 class "\code{homGP}", with elements:
\itemize{
\item \code{theta}: maximum likelihood estimate of the lengthscale parameter(s),
\item \code{g}: maximum likelihood estimate of the nugget variance,
\item \code{trendtype}: either "\code{SK}" if \code{beta0} is given, else "\code{OK}" 
\item \code{beta0}: estimated trend unless given in input,
\item \code{nu2_hat}: plugin estimator of the variance,
\item \code{ll}: log-likelihood value,
\item \code{X0}, \code{Z0}, \code{Z}, \code{mult}, \code{eps}, \code{covtype}: values given in input,
\item \code{call}: user call of the function
\item \code{used_args}: list with arguments provided in the call
\item \code{nit_opt}, \code{msg}: \code{counts} and \code{msg} returned by \code{\link[stats]{optim}}
\item \code{Ki}, inverse covariance matrix (if \code{return.Ki} is \code{TRUE} in \code{settings})

}
}
\description{
Gaussian process regression under homogeneous noise based on maximum likelihood estimation of the 
hyperparameters. This function is enhanced to deal with replicated observations.
}
\details{
The global covariance matrix of the model is parameterized as \code{K = nu2_hat * (C + g * diag(1/mult))},
with \code{C} the correlation matrix between unique designs, depending on the family of kernel used (see \code{\link[hetGP]{cov_gen}} for available choices) and values of lengthscale parameters.
\code{nu2_hat} is the plugin estimator of the variance of the process.

It is generally recommended to use \code{\link[hetGP]{find_reps}} to pre-process the data, to rescale the inputs to the unit cube and to normalize the outputs.
}
\examples{
##------------------------------------------------------------
## Example 1: Homoskedastic GP modeling on the motorcycle data
##------------------------------------------------------------
set.seed(32)

## motorcycle data
library(MASS)
X <- matrix(mcycle$times, ncol = 1)
Z <- mcycle$accel
plot(X, Z, ylim = c(-160, 90), ylab = 'acceleration', xlab = "time")

 
model <- mleHomGP(X = X, Z = Z, lower = 0.01, upper = 100)
  
## Display averaged observations
points(model$X0, model$Z0, pch = 20) 
xgrid <- matrix(seq(0, 60, length.out = 301), ncol = 1) 
predictions <- predict(x = xgrid, object =  model)

## Display mean prediction
lines(xgrid, predictions$mean, col = 'red', lwd = 2)
## Display 95\% confidence intervals
lines(xgrid, qnorm(0.05, predictions$mean, sqrt(predictions$sd2)), col = 2, lty = 2)
lines(xgrid, qnorm(0.95, predictions$mean, sqrt(predictions$sd2)), col = 2, lty = 2)
## Display 95\% prediction intervals
lines(xgrid, qnorm(0.05, predictions$mean, sqrt(predictions$sd2 + predictions$nugs)), 
  col = 3, lty = 2)
lines(xgrid, qnorm(0.95, predictions$mean, sqrt(predictions$sd2 + predictions$nugs)), 
  col = 3, lty = 2)
}
\references{
M. Binois, Robert B. Gramacy, M. Ludkovski (2016+), Practical heteroskedastic Gaussian process modeling for large simulation experiments, arXiv preprint arXiv:1611.05902.
}
\seealso{
\code{\link[hetGP]{predict.homGP}} for predictions, \code{\link[hetGP]{update.homGP}} for updating an existing model. A \code{summary} function is available as well.
}
