% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vigitel.R
\name{vigitel_data}
\alias{vigitel_data}
\title{Load VIGITEL microdata}
\usage{
vigitel_data(
  year,
  vars = NULL,
  force_download = FALSE,
  parallel = TRUE,
  lazy = FALSE
)
}
\arguments{
\item{year}{Year(s) of the survey. Can be:
\itemize{
\item Single year: \code{2023}
\item Range: \code{2021:2023}
\item Vector: \code{c(2021, 2023)}
\item Character: \code{c("2021", "2023")}
\item All years: \code{"all"}
}}

\item{vars}{Character vector. Variable names to select, or NULL for all
variables. Default is NULL.}

\item{force_download}{Logical. If TRUE, re-download and reconvert data.
Default is FALSE.}

\item{parallel}{Logical. If TRUE, download and process multiple years in
parallel. Default is TRUE when multiple years are requested.}

\item{lazy}{Logical. If TRUE, return an Arrow Dataset for lazy evaluation
instead of loading all data into memory. Useful for filtering large
datasets before collecting. Use \code{collect()} to retrieve results.
Default is FALSE.}
}
\value{
A tibble with the VIGITEL microdata. When multiple years are
requested, a \code{year} column is added to identify the source year.
If \code{lazy = TRUE}, returns an Arrow Dataset that can be queried
with dplyr verbs before calling \code{collect()}.
}
\description{
Downloads (if necessary) and loads VIGITEL survey microdata into R.
Data is automatically converted to Parquet format for faster subsequent
loading. The data includes survey weights for proper statistical analysis.
}
\details{
On first access, data is downloaded from the Ministry of Health and
converted to Parquet format. Subsequent loads read directly from the
Parquet file, which is significantly faster.

For parallel downloads, the function uses the \code{furrr} and \code{future}
packages if installed. Install them with \code{install.packages(c("furrr", "future"))}
to enable parallel processing. The number of workers is automatically set
based on available CPU cores. If these packages are not installed, processing
falls back to sequential mode.

When \code{lazy = TRUE}, the function returns an Arrow Dataset that supports
dplyr operations (filter, select, mutate, etc.) without loading data into
memory. This is useful for working with large datasets or when you only
need a subset of the data. Call \code{collect()} to retrieve the results
as a tibble.

The VIGITEL survey uses complex sampling weights. For proper statistical
analysis, use survey packages like \code{survey} or \code{srvyr}.
The weight variable is named \code{pesorake}.
}
\examples{
\donttest{
# single year
df <- vigitel_data(2023)

# multiple years
df <- vigitel_data(2021:2023)
df <- vigitel_data(c(2018, 2020, 2023))

# all available years
df <- vigitel_data("all")

# specific variables
df <- vigitel_data(2023, vars = c("cidade", "sexo", "idade", "pesorake"))

# multiple years with specific variables
df <- vigitel_data(2020:2023, vars = c("cidade", "sexo", "idade", "pesorake"))

# lazy evaluation - filter before loading into memory
vigitel_data(2023, lazy = TRUE) |>
  dplyr::filter(cidade == 1) |>
  dplyr::select(pesorake) |>
  dplyr::collect()

# lazy with multiple years
vigitel_data(2020:2023, lazy = TRUE) |>
  dplyr::filter(q6 == 1) |>
  dplyr::collect()
}
}
