/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5LTprivate.h"

#define COL       3
#define LIMIT     512
#define INCREMENT 1024
#define TMP_LEN   256
#define MAX(a, b) (((a) > (b)) ? (a) : (b))
size_t input_len;
char  *myinput;
size_t indent = 0;

typedef struct {
    void    *app_image_ptr;   
    size_t   app_image_size;  
    void    *fapl_image_ptr;  
    size_t   fapl_image_size; 
    int      fapl_ref_count;  
    void    *vfd_image_ptr;   
    size_t   vfd_image_size;  
    int      vfd_ref_count;   
    unsigned flags;           
                              
    int ref_count;            
} H5LT_file_image_ud_t;

static void  *image_malloc(size_t size, H5FD_file_image_op_t file_image_op, void *udata);
static void  *image_memcpy(void *dest, const void *src, size_t size, H5FD_file_image_op_t file_image_op,
                           void *udata);
static void  *image_realloc(void *ptr, size_t size, H5FD_file_image_op_t file_image_op, void *udata);
static herr_t image_free(void *ptr, H5FD_file_image_op_t file_image_op, void *udata);
static void  *udata_copy(void *udata);
static herr_t udata_free(void *udata);

static void *
image_malloc(size_t size, H5FD_file_image_op_t file_image_op, void *_udata)
{
    H5LT_file_image_ud_t *udata        = (H5LT_file_image_ud_t *)_udata;
    void                 *return_value = NULL;

    
    if (!(udata->flags & H5LT_FILE_IMAGE_DONT_COPY))
        goto out;

    switch (file_image_op) {
        
        case H5FD_FILE_IMAGE_OP_PROPERTY_LIST_SET:
            if (udata->app_image_ptr == NULL)
                goto out;
            if (udata->app_image_size != size)
                goto out;
            if (udata->fapl_image_ptr != NULL)
                goto out;
            if (udata->fapl_image_size != 0)
                goto out;
            if (udata->fapl_ref_count != 0)
                goto out;

            udata->fapl_image_ptr  = udata->app_image_ptr;
            udata->fapl_image_size = udata->app_image_size;
            return_value           = udata->fapl_image_ptr;
            udata->fapl_ref_count++;
            break;

        case H5FD_FILE_IMAGE_OP_PROPERTY_LIST_COPY:
            if (udata->fapl_image_ptr == NULL)
                goto out;
            if (udata->fapl_image_size != size)
                goto out;
            if (udata->fapl_ref_count == 0)
                goto out;

            return_value = udata->fapl_image_ptr;
            udata->fapl_ref_count++;
            break;

        case H5FD_FILE_IMAGE_OP_PROPERTY_LIST_GET:
            goto out;

        case H5FD_FILE_IMAGE_OP_FILE_OPEN:
            
            if (udata->vfd_image_ptr != NULL)
                goto out;
            if (udata->vfd_image_size != 0)
                goto out;
            if (udata->vfd_ref_count != 0)
                goto out;
            if (udata->fapl_image_ptr == NULL)
                goto out;
            if (udata->fapl_image_size != size)
                goto out;
            if (udata->fapl_ref_count == 0)
                goto out;

            udata->vfd_image_ptr  = udata->fapl_image_ptr;
            udata->vfd_image_size = size;
            udata->vfd_ref_count++;
            return_value = udata->vfd_image_ptr;
            break;

        
        case H5FD_FILE_IMAGE_OP_NO_OP:
        case H5FD_FILE_IMAGE_OP_PROPERTY_LIST_CLOSE:
        case H5FD_FILE_IMAGE_OP_FILE_RESIZE:
        case H5FD_FILE_IMAGE_OP_FILE_CLOSE:
        default:
            goto out;
    } 

    return (return_value);

out:
    return NULL;
} 

static void *
image_memcpy(void *dest, const void *src, size_t size, H5FD_file_image_op_t file_image_op, void *_udata)
{
    H5LT_file_image_ud_t *udata = (H5LT_file_image_ud_t *)_udata;

    
    if (!(udata->flags & H5LT_FILE_IMAGE_DONT_COPY))
        goto out;

    switch (file_image_op) {
        case H5FD_FILE_IMAGE_OP_PROPERTY_LIST_SET:
            if (dest != udata->fapl_image_ptr)
                goto out;
            if (src != udata->app_image_ptr)
                goto out;
            if (size != udata->fapl_image_size)
                goto out;
            if (size != udata->app_image_size)
                goto out;
            if (udata->fapl_ref_count == 0)
                goto out;
            break;

        case H5FD_FILE_IMAGE_OP_PROPERTY_LIST_COPY:
            if (dest != udata->fapl_image_ptr)
                goto out;
            if (src != udata->fapl_image_ptr)
                goto out;
            if (size != udata->fapl_image_size)
                goto out;
            if (udata->fapl_ref_count < 2)
                goto out;
            break;

        case H5FD_FILE_IMAGE_OP_PROPERTY_LIST_GET:
            goto out;

        case H5FD_FILE_IMAGE_OP_FILE_OPEN:
            if (dest != udata->vfd_image_ptr)
                goto out;
            if (src != udata->fapl_image_ptr)
                goto out;
            if (size != udata->vfd_image_size)
                goto out;
            if (size != udata->fapl_image_size)
                goto out;
            if (udata->fapl_ref_count == 0)
                goto out;
            if (udata->vfd_ref_count != 1)
                goto out;
            break;

        
        case H5FD_FILE_IMAGE_OP_NO_OP:
        case H5FD_FILE_IMAGE_OP_PROPERTY_LIST_CLOSE:
        case H5FD_FILE_IMAGE_OP_FILE_RESIZE:
        case H5FD_FILE_IMAGE_OP_FILE_CLOSE:
        default:
            goto out;
    } 

    return (dest);

out:
    return NULL;
} 

static void *
image_realloc(void *ptr, size_t size, H5FD_file_image_op_t file_image_op, void *_udata)
{
    H5LT_file_image_ud_t *udata        = (H5LT_file_image_ud_t *)_udata;
    void                 *return_value = NULL;

    
    if (!(udata->flags & H5LT_FILE_IMAGE_DONT_COPY))
        goto out;

    
    if (udata->flags & H5LT_FILE_IMAGE_DONT_RELEASE)
        goto out;

    
    if (!(udata->flags & H5LT_FILE_IMAGE_OPEN_RW))
        goto out;

    if (file_image_op == H5FD_FILE_IMAGE_OP_FILE_RESIZE) {
        void *tmp_realloc;

        if (udata->vfd_image_ptr != ptr)
            goto out;

        if (udata->vfd_ref_count != 1)
            goto out;

        
        assert(udata->vfd_image_ptr == udata->app_image_ptr);
        assert(udata->vfd_image_ptr == udata->fapl_image_ptr);

        tmp_realloc = realloc(ptr, size);
        if (tmp_realloc) {
            udata->vfd_image_ptr  = tmp_realloc;
            udata->app_image_ptr  = udata->vfd_image_ptr;
            udata->fapl_image_ptr = udata->vfd_image_ptr;
        }
        else {
            free(ptr);
            udata->vfd_image_ptr  = NULL;
            udata->app_image_ptr  = NULL;
            udata->fapl_image_ptr = NULL;
            goto out;
        }

        udata->vfd_image_size = size;
        return_value          = udata->vfd_image_ptr;
    } 
    else
        goto out;

    return (return_value);

out:
    return NULL;
} 

static herr_t
image_free(void *ptr, H5FD_file_image_op_t file_image_op, void *_udata)
{
    H5LT_file_image_ud_t *udata = (H5LT_file_image_ud_t *)_udata;

    
    if (!(udata->flags & H5LT_FILE_IMAGE_DONT_COPY))
        goto out;

    switch (file_image_op) {
        case H5FD_FILE_IMAGE_OP_PROPERTY_LIST_CLOSE:
            if (udata->fapl_image_ptr != ptr)
                goto out;
            if (udata->fapl_ref_count == 0)
                goto out;

            udata->fapl_ref_count--;

            
            if (udata->fapl_ref_count == 0 && udata->vfd_ref_count == 0 &&
                !(udata->flags & H5LT_FILE_IMAGE_DONT_RELEASE)) {
                free(udata->fapl_image_ptr);
                udata->app_image_ptr  = NULL;
                udata->fapl_image_ptr = NULL;
                udata->vfd_image_ptr  = NULL;
            } 
            break;

        case H5FD_FILE_IMAGE_OP_FILE_CLOSE:
            if (udata->vfd_image_ptr != ptr)
                goto out;
            if (udata->vfd_ref_count != 1)
                goto out;

            udata->vfd_ref_count--;

            
            if (udata->fapl_ref_count == 0 && udata->vfd_ref_count == 0 &&
                !(udata->flags & H5LT_FILE_IMAGE_DONT_RELEASE)) {
                
                assert(udata->app_image_ptr == udata->vfd_image_ptr || udata->app_image_ptr == NULL);
                assert(udata->fapl_image_ptr == udata->vfd_image_ptr || udata->fapl_image_ptr == NULL);

                free(udata->vfd_image_ptr);
                udata->app_image_ptr  = NULL;
                udata->fapl_image_ptr = NULL;
                udata->vfd_image_ptr  = NULL;
            }

            
            if (udata_free(udata) < 0)
                goto out;

            break;

        
        case H5FD_FILE_IMAGE_OP_NO_OP:
        case H5FD_FILE_IMAGE_OP_PROPERTY_LIST_SET:
        case H5FD_FILE_IMAGE_OP_PROPERTY_LIST_COPY:
        case H5FD_FILE_IMAGE_OP_PROPERTY_LIST_GET:
        case H5FD_FILE_IMAGE_OP_FILE_OPEN:
        case H5FD_FILE_IMAGE_OP_FILE_RESIZE:
        default:
            goto out;
    } 

    return (SUCCEED);

out:
    return (FAIL);
} 

static void *
udata_copy(void *_udata)
{
    H5LT_file_image_ud_t *udata = (H5LT_file_image_ud_t *)_udata;

    
    if (!(udata->flags & H5LT_FILE_IMAGE_DONT_COPY))
        goto out;
    if (udata->ref_count == 0)
        goto out;

    udata->ref_count++;

    return (udata);

out:
    return NULL;
} 

static herr_t
udata_free(void *_udata)
{
    H5LT_file_image_ud_t *udata = (H5LT_file_image_ud_t *)_udata;

    
    if (!(udata->flags & H5LT_FILE_IMAGE_DONT_COPY))
        goto out;
    if (udata->ref_count == 0)
        goto out;

    udata->ref_count--;

    if (udata->ref_count == 0) {
        
        assert(udata->fapl_ref_count == 0);
        assert(udata->vfd_ref_count == 0);

        free(udata);
    }

    return (SUCCEED);

out:
    return (FAIL);
} 

static herr_t H5LT_get_attribute_mem(hid_t loc_id, const char *obj_name, const char *attr_name,
                                     hid_t mem_type_id, void *data);

static herr_t
H5LT_make_dataset_numerical(hid_t loc_id, const char *dset_name, int rank, const hsize_t *dims, hid_t tid,
                            const void *data)
{
    hid_t did = -1, sid = -1;

    
    if (dset_name == NULL)
        return -1;

    
    if ((sid = H5Screate_simple(rank, dims, NULL)) < 0)
        return -1;

    
    if ((did = H5Dcreate2(loc_id, dset_name, tid, sid, H5P_DEFAULT, H5P_DEFAULT, H5P_DEFAULT)) < 0)
        goto out;

    
    if (data)
        if (H5Dwrite(did, tid, H5S_ALL, H5S_ALL, H5P_DEFAULT, data) < 0)
            goto out;

    
    if (H5Dclose(did) < 0)
        return -1;

    
    if (H5Sclose(sid) < 0)
        return -1;

    return 0;

out:
    H5E_BEGIN_TRY
    {
        H5Dclose(did);
        H5Sclose(sid);
    }
    H5E_END_TRY
    return -1;
}

herr_t
H5LTmake_dataset(hid_t loc_id, const char *dset_name, int rank, const hsize_t *dims, hid_t tid,
                 const void *data)
{
    return (H5LT_make_dataset_numerical(loc_id, dset_name, rank, dims, tid, data));
}

herr_t
H5LTmake_dataset_char(hid_t loc_id, const char *dset_name, int rank, const hsize_t *dims, const char *data)
{
    return (H5LT_make_dataset_numerical(loc_id, dset_name, rank, dims, H5T_NATIVE_CHAR, data));
}

herr_t
H5LTmake_dataset_short(hid_t loc_id, const char *dset_name, int rank, const hsize_t *dims, const short *data)
{
    return (H5LT_make_dataset_numerical(loc_id, dset_name, rank, dims, H5T_NATIVE_SHORT, data));
}

herr_t
H5LTmake_dataset_int(hid_t loc_id, const char *dset_name, int rank, const hsize_t *dims, const int *data)
{
    return (H5LT_make_dataset_numerical(loc_id, dset_name, rank, dims, H5T_NATIVE_INT, data));
}

herr_t
H5LTmake_dataset_long(hid_t loc_id, const char *dset_name, int rank, const hsize_t *dims, const long *data)
{
    return (H5LT_make_dataset_numerical(loc_id, dset_name, rank, dims, H5T_NATIVE_LONG, data));
}

herr_t
H5LTmake_dataset_float(hid_t loc_id, const char *dset_name, int rank, const hsize_t *dims, const float *data)
{
    return (H5LT_make_dataset_numerical(loc_id, dset_name, rank, dims, H5T_NATIVE_FLOAT, data));
}

herr_t
H5LTmake_dataset_double(hid_t loc_id, const char *dset_name, int rank, const hsize_t *dims,
                        const double *data)
{
    return (H5LT_make_dataset_numerical(loc_id, dset_name, rank, dims, H5T_NATIVE_DOUBLE, data));
}

herr_t
H5LTmake_dataset_string(hid_t loc_id, const char *dset_name, const char *buf)
{
    hid_t  did = -1;
    hid_t  sid = -1;
    hid_t  tid = -1;
    size_t size;

    
    if (dset_name == NULL)
        return -1;

    
    if ((tid = H5Tcopy(H5T_C_S1)) < 0)
        goto out;

    size = strlen(buf) + 1; 

    if (H5Tset_size(tid, size) < 0)
        goto out;

    if (H5Tset_strpad(tid, H5T_STR_NULLTERM) < 0)
        goto out;

    
    if ((sid = H5Screate(H5S_SCALAR)) < 0)
        goto out;

    
    if ((did = H5Dcreate2(loc_id, dset_name, tid, sid, H5P_DEFAULT, H5P_DEFAULT, H5P_DEFAULT)) < 0)
        goto out;

    
    if (buf)
        if (H5Dwrite(did, tid, H5S_ALL, H5S_ALL, H5P_DEFAULT, buf) < 0)
            goto out;

    
    if (H5Dclose(did) < 0)
        return -1;
    if (H5Sclose(sid) < 0)
        return -1;
    if (H5Tclose(tid) < 0)
        goto out;

    return 0;

out:
    H5E_BEGIN_TRY
    {
        H5Dclose(did);
        H5Tclose(tid);
        H5Sclose(sid);
    }
    H5E_END_TRY
    return -1;
}

hid_t
H5LTopen_file_image(void *buf_ptr, size_t buf_size, unsigned flags)
{
    H5LT_file_image_ud_t       *udata = NULL;            
    hid_t                       fapl = -1, file_id = -1; 
    unsigned                    file_open_flags;         
    char                        file_name[64];           
    size_t                      alloc_incr;              
    size_t                      min_incr  = 65536;       
    double                      buf_prcnt = 0.1;         
    static long                 file_name_counter;
    H5FD_file_image_callbacks_t callbacks = {&image_malloc, &image_memcpy, &image_realloc, &image_free,
                                             &udata_copy,   &udata_free,   (void *)NULL};

    
    if (buf_ptr == NULL)
        goto out;
    if (buf_size == 0)
        goto out;
    if (flags & (unsigned)~(H5LT_FILE_IMAGE_ALL))
        goto out;

    
    if ((fapl = H5Pcreate(H5P_FILE_ACCESS)) < 0)
        goto out;

    
    if ((size_t)(buf_prcnt * (double)buf_size) > min_incr)
        alloc_incr = (size_t)(buf_prcnt * (double)buf_size);
    else
        alloc_incr = min_incr;

    
    if (H5Pset_fapl_core(fapl, alloc_incr, false) < 0)
        goto out;

    
    if (flags & H5LT_FILE_IMAGE_DONT_COPY) {
        
        if (NULL == (udata = (H5LT_file_image_ud_t *)malloc(sizeof(H5LT_file_image_ud_t))))
            goto out;

        
        udata->app_image_ptr   = buf_ptr;
        udata->app_image_size  = buf_size;
        udata->fapl_image_ptr  = NULL;
        udata->fapl_image_size = 0;
        udata->fapl_ref_count  = 0;
        udata->vfd_image_ptr   = NULL;
        udata->vfd_image_size  = 0;
        udata->vfd_ref_count   = 0;
        udata->flags           = flags;

        
        udata->ref_count = 1;

        
        callbacks.udata = (void *)udata;

        
        if (H5Pset_file_image_callbacks(fapl, &callbacks) < 0) {
            udata_free(udata);
            goto out;
        }
    } 

    
    if (H5Pset_file_image(fapl, buf_ptr, buf_size) < 0)
        goto out;

    
    if (flags & H5LT_FILE_IMAGE_OPEN_RW)
        file_open_flags = H5F_ACC_RDWR;
    else
        file_open_flags = H5F_ACC_RDONLY;

    
    snprintf(file_name, (sizeof(file_name) - 1), "file_image_%ld", file_name_counter++);

    
    if ((file_id = H5Fopen(file_name, file_open_flags, fapl)) < 0)
        goto out;

    
    if (H5Pclose(fapl) < 0)
        goto out;

    
    return file_id;

out:
    H5E_BEGIN_TRY
    {
        H5Pclose(fapl);
        H5Fclose(file_id);
    }
    H5E_END_TRY

    return -1;
} 

static herr_t
H5LT_read_dataset_numerical(hid_t loc_id, const char *dset_name, hid_t tid, void *data)
{
    hid_t did;

    
    if (dset_name == NULL)
        return -1;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if (H5Dread(did, tid, H5S_ALL, H5S_ALL, H5P_DEFAULT, data) < 0)
        goto out;

    
    if (H5Dclose(did))
        return -1;

    return 0;

out:
    H5Dclose(did);
    return -1;
}

herr_t
H5LTread_dataset(hid_t loc_id, const char *dset_name, hid_t tid, void *data)
{
    return (H5LT_read_dataset_numerical(loc_id, dset_name, tid, data));
}

herr_t
H5LTread_dataset_char(hid_t loc_id, const char *dset_name, char *data)
{
    return (H5LT_read_dataset_numerical(loc_id, dset_name, H5T_NATIVE_CHAR, data));
}

herr_t
H5LTread_dataset_short(hid_t loc_id, const char *dset_name, short *data)
{
    return (H5LT_read_dataset_numerical(loc_id, dset_name, H5T_NATIVE_SHORT, data));
}

herr_t
H5LTread_dataset_int(hid_t loc_id, const char *dset_name, int *data)
{
    return (H5LT_read_dataset_numerical(loc_id, dset_name, H5T_NATIVE_INT, data));
}

herr_t
H5LTread_dataset_long(hid_t loc_id, const char *dset_name, long *data)
{
    return (H5LT_read_dataset_numerical(loc_id, dset_name, H5T_NATIVE_LONG, data));
}

herr_t
H5LTread_dataset_float(hid_t loc_id, const char *dset_name, float *data)
{
    return (H5LT_read_dataset_numerical(loc_id, dset_name, H5T_NATIVE_FLOAT, data));
}

herr_t
H5LTread_dataset_double(hid_t loc_id, const char *dset_name, double *data)
{
    return (H5LT_read_dataset_numerical(loc_id, dset_name, H5T_NATIVE_DOUBLE, data));
}

herr_t
H5LTread_dataset_string(hid_t loc_id, const char *dset_name, char *buf)
{
    hid_t did = -1;
    hid_t tid = -1;

    
    if (dset_name == NULL)
        return -1;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        return -1;

    if ((tid = H5Dget_type(did)) < 0)
        goto out;

    
    if (H5Dread(did, tid, H5S_ALL, H5S_ALL, H5P_DEFAULT, buf) < 0)
        goto out;

    
    if (H5Dclose(did))
        goto out;
    if (H5Tclose(tid))
        return -1;

    return 0;

out:
    H5E_BEGIN_TRY
    {
        H5Dclose(did);
        H5Tclose(tid);
    }
    H5E_END_TRY
    return -1;
}

herr_t
H5LTget_dataset_ndims(hid_t loc_id, const char *dset_name, int *rank)
{
    hid_t did = -1;
    hid_t sid = -1;

    
    if (dset_name == NULL)
        return -1;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if ((sid = H5Dget_space(did)) < 0)
        goto out;

    
    if ((*rank = H5Sget_simple_extent_ndims(sid)) < 0)
        goto out;

    
    if (H5Sclose(sid) < 0)
        goto out;

    
    if (H5Dclose(did))
        return -1;

    return 0;

out:
    H5E_BEGIN_TRY
    {
        H5Dclose(did);
        H5Sclose(sid);
    }
    H5E_END_TRY
    return -1;
}

herr_t
H5LTget_dataset_info(hid_t loc_id, const char *dset_name, hsize_t *dims, H5T_class_t *type_class,
                     size_t *type_size)
{
    hid_t did = -1;
    hid_t tid = -1;
    hid_t sid = -1;

    
    if (dset_name == NULL)
        return -1;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        return -1;

    
    tid = H5Dget_type(did);

    
    if (type_class != NULL)
        *type_class = H5Tget_class(tid);

    
    if (type_size != NULL)
        *type_size = H5Tget_size(tid);

    if (dims != NULL) {
        
        if ((sid = H5Dget_space(did)) < 0)
            goto out;

        
        if (H5Sget_simple_extent_dims(sid, dims, NULL) < 0)
            goto out;

        
        if (H5Sclose(sid) < 0)
            goto out;
    } 

    
    if (H5Tclose(tid))
        return -1;

    
    if (H5Dclose(did))
        return -1;

    return 0;

out:
    H5E_BEGIN_TRY
    {
        H5Tclose(tid);
        H5Sclose(sid);
        H5Dclose(did);
    }
    H5E_END_TRY
    return -1;
}

static herr_t
find_dataset(H5_ATTR_UNUSED hid_t loc_id, const char *name, H5_ATTR_UNUSED const H5L_info2_t *linfo,
             void *op_data)
{
    
    int ret = 0;

    
    if (name == NULL)
        return ret;

    
    (void)loc_id;
    (void)linfo;

    
    if (strcmp(name, (char *)op_data) == 0)
        ret = 1;

    return ret;
}

H5_WARN_CAST_AWAY_CONST_OFF
herr_t
H5LTfind_dataset(hid_t loc_id, const char *dset_name)
{
    return H5Literate2(loc_id, H5_INDEX_NAME, H5_ITER_INC, 0, find_dataset, (void *)dset_name);
}
H5_WARN_CAST_AWAY_CONST_ON

herr_t
H5LTset_attribute_string(hid_t loc_id, const char *obj_name, const char *attr_name, const char *attr_data)
{
    hid_t  attr_type;
    hid_t  attr_space_id;
    hid_t  attr_id;
    hid_t  obj_id;
    htri_t has_attr;
    size_t attr_size;

    
    if (obj_name == NULL)
        return -1;
    if (attr_name == NULL)
        return -1;
    if (attr_data == NULL)
        return -1;

    
    if ((obj_id = H5Oopen(loc_id, obj_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if ((attr_type = H5Tcopy(H5T_C_S1)) < 0)
        goto out;

    attr_size = strlen(attr_data) + 1; 

    if (H5Tset_size(attr_type, (size_t)attr_size) < 0)
        goto out;

    if (H5Tset_strpad(attr_type, H5T_STR_NULLTERM) < 0)
        goto out;

    if ((attr_space_id = H5Screate(H5S_SCALAR)) < 0)
        goto out;

    
    if ((has_attr = H5Aexists(obj_id, attr_name)) < 0)
        goto out;
    if (has_attr > 0)
        if (H5Adelete(obj_id, attr_name) < 0)
            goto out;

    

    if ((attr_id = H5Acreate2(obj_id, attr_name, attr_type, attr_space_id, H5P_DEFAULT, H5P_DEFAULT)) < 0)
        goto out;

    if (H5Awrite(attr_id, attr_type, attr_data) < 0)
        goto out;

    if (H5Aclose(attr_id) < 0)
        goto out;

    if (H5Sclose(attr_space_id) < 0)
        goto out;

    if (H5Tclose(attr_type) < 0)
        goto out;

    
    if (H5Oclose(obj_id) < 0)
        return -1;

    return 0;

out:

    H5Oclose(obj_id);
    return -1;
}

herr_t
H5LT_set_attribute_numerical(hid_t loc_id, const char *obj_name, const char *attr_name, size_t size,
                             hid_t tid, const void *data)
{

    hid_t   obj_id, sid, attr_id;
    hsize_t dim_size = size;
    htri_t  has_attr;

    
    if (obj_name == NULL)
        return -1;
    if (attr_name == NULL)
        return -1;

    
    if ((obj_id = H5Oopen(loc_id, obj_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if ((sid = H5Screate_simple(1, &dim_size, NULL)) < 0)
        goto out;

    
    if ((has_attr = H5Aexists(obj_id, attr_name)) < 0)
        goto out;
    if (has_attr > 0)
        if (H5Adelete(obj_id, attr_name) < 0)
            goto out;

    
    if ((attr_id = H5Acreate2(obj_id, attr_name, tid, sid, H5P_DEFAULT, H5P_DEFAULT)) < 0)
        goto out;

    
    if (H5Awrite(attr_id, tid, data) < 0)
        goto out;

    
    if (H5Aclose(attr_id) < 0)
        goto out;

    
    if (H5Sclose(sid) < 0)
        goto out;

    
    if (H5Oclose(obj_id) < 0)
        return -1;

    return 0;

out:
    H5Oclose(obj_id);
    return -1;
}

herr_t
H5LTset_attribute_char(hid_t loc_id, const char *obj_name, const char *attr_name, const char *data,
                       size_t size)
{

    if (H5LT_set_attribute_numerical(loc_id, obj_name, attr_name, size, H5T_NATIVE_CHAR, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTset_attribute_uchar(hid_t loc_id, const char *obj_name, const char *attr_name, const unsigned char *data,
                        size_t size)
{

    if (H5LT_set_attribute_numerical(loc_id, obj_name, attr_name, size, H5T_NATIVE_UCHAR, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTset_attribute_short(hid_t loc_id, const char *obj_name, const char *attr_name, const short *data,
                        size_t size)
{

    if (H5LT_set_attribute_numerical(loc_id, obj_name, attr_name, size, H5T_NATIVE_SHORT, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTset_attribute_ushort(hid_t loc_id, const char *obj_name, const char *attr_name,
                         const unsigned short *data, size_t size)
{

    if (H5LT_set_attribute_numerical(loc_id, obj_name, attr_name, size, H5T_NATIVE_USHORT, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTset_attribute_int(hid_t loc_id, const char *obj_name, const char *attr_name, const int *data, size_t size)
{

    if (H5LT_set_attribute_numerical(loc_id, obj_name, attr_name, size, H5T_NATIVE_INT, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTset_attribute_uint(hid_t loc_id, const char *obj_name, const char *attr_name, const unsigned int *data,
                       size_t size)
{

    if (H5LT_set_attribute_numerical(loc_id, obj_name, attr_name, size, H5T_NATIVE_UINT, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTset_attribute_long(hid_t loc_id, const char *obj_name, const char *attr_name, const long *data,
                       size_t size)
{

    if (H5LT_set_attribute_numerical(loc_id, obj_name, attr_name, size, H5T_NATIVE_LONG, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTset_attribute_long_long(hid_t loc_id, const char *obj_name, const char *attr_name, const long long *data,
                            size_t size)
{

    if (H5LT_set_attribute_numerical(loc_id, obj_name, attr_name, size, H5T_NATIVE_LLONG, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTset_attribute_ulong(hid_t loc_id, const char *obj_name, const char *attr_name, const unsigned long *data,
                        size_t size)
{

    if (H5LT_set_attribute_numerical(loc_id, obj_name, attr_name, size, H5T_NATIVE_ULONG, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTset_attribute_ullong(hid_t loc_id, const char *obj_name, const char *attr_name,
                         const unsigned long long *data, size_t size)
{

    if (H5LT_set_attribute_numerical(loc_id, obj_name, attr_name, size, H5T_NATIVE_ULLONG, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTset_attribute_float(hid_t loc_id, const char *obj_name, const char *attr_name, const float *data,
                        size_t size)
{

    if (H5LT_set_attribute_numerical(loc_id, obj_name, attr_name, size, H5T_NATIVE_FLOAT, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTset_attribute_double(hid_t loc_id, const char *obj_name, const char *attr_name, const double *data,
                         size_t size)
{

    if (H5LT_set_attribute_numerical(loc_id, obj_name, attr_name, size, H5T_NATIVE_DOUBLE, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTfind_attribute(hid_t loc_id, const char *attr_name)
{
    return (herr_t)H5Aexists(loc_id, attr_name);
}

herr_t
H5LTget_attribute_ndims(hid_t loc_id, const char *obj_name, const char *attr_name, int *rank)
{
    hid_t attr_id;
    hid_t sid;
    hid_t obj_id;

    
    if (obj_name == NULL)
        return -1;
    if (attr_name == NULL)
        return -1;

    
    if ((obj_id = H5Oopen(loc_id, obj_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if ((attr_id = H5Aopen(obj_id, attr_name, H5P_DEFAULT)) < 0) {
        H5Oclose(obj_id);
        return -1;
    }

    
    if ((sid = H5Aget_space(attr_id)) < 0)
        goto out;

    
    if ((*rank = H5Sget_simple_extent_ndims(sid)) < 0)
        goto out;

    
    if (H5Sclose(sid) < 0)
        goto out;

    
    if (H5Aclose(attr_id))
        goto out;

    
    if (H5Oclose(obj_id) < 0)
        return -1;

    return 0;

out:
    H5Aclose(attr_id);
    H5Oclose(obj_id);
    return -1;
}

herr_t
H5LTget_attribute_info(hid_t loc_id, const char *obj_name, const char *attr_name, hsize_t *dims,
                       H5T_class_t *type_class, size_t *type_size)
{
    hid_t attr_id;
    hid_t tid;
    hid_t sid;
    hid_t obj_id;

    
    if (obj_name == NULL)
        return -1;
    if (attr_name == NULL)
        return -1;

    
    if ((obj_id = H5Oopen(loc_id, obj_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if ((attr_id = H5Aopen(obj_id, attr_name, H5P_DEFAULT)) < 0) {
        H5Oclose(obj_id);
        return -1;
    }

    
    tid = H5Aget_type(attr_id);

    
    *type_class = H5Tget_class(tid);

    
    *type_size = H5Tget_size(tid);

    
    if ((sid = H5Aget_space(attr_id)) < 0)
        goto out;

    
    if (H5Sget_simple_extent_dims(sid, dims, NULL) < 0)
        goto out;

    
    if (H5Sclose(sid) < 0)
        goto out;

    
    if (H5Tclose(tid))
        goto out;

    
    if (H5Aclose(attr_id))
        goto out;

    
    if (H5Oclose(obj_id) < 0)
        return -1;

    return 0;

out:
    H5Tclose(tid);
    H5Aclose(attr_id);
    H5Oclose(obj_id);
    return -1;
}

hid_t
H5LTtext_to_dtype(const char *text, H5LT_lang_t lang_type)
{
    hid_t type_id;

    
    if (text == NULL)
        return -1;

    if (lang_type <= H5LT_LANG_ERR || lang_type >= H5LT_NO_LANG)
        goto out;

    if (lang_type != H5LT_DDL) {
        Rfprintf(Rstderr, "only DDL is supported for now.\n");
        goto out;
    }

    input_len = strlen(text);
    myinput   = strdup(text);

    if ((type_id = H5LTyyparse()) < 0) {
        free(myinput);
        goto out;
    }

    free(myinput);
    input_len = 0;

    return type_id;

out:
    return -1;
}

static char *
realloc_and_append(bool _no_user_buf, size_t *len, char *buf, const char *str_to_add)
{
    size_t size_str_to_add, size_str;

    if (_no_user_buf) {
        char *tmp_realloc;

        if (!buf)
            goto out;

        
        if (str_to_add && ((ssize_t)(*len - (strlen(buf) + strlen(str_to_add) + 1)) < LIMIT)) {
            *len += ((strlen(buf) + strlen(str_to_add) + 1) / INCREMENT + 1) * INCREMENT;
        }
        else if (!str_to_add && ((ssize_t)(*len - strlen(buf) - 1) < LIMIT)) {
            *len += INCREMENT;
        }

        tmp_realloc = (char *)realloc(buf, *len);
        if (tmp_realloc == NULL) {
            free(buf);
            buf = NULL;
            goto out;
        }
        else
            buf = tmp_realloc;
    }

    if (str_to_add) {
        
        size_str_to_add = strlen(str_to_add);
        
        size_str = strlen(buf);

        
        if (size_str < *len - 1) {
            if (size_str + size_str_to_add < *len - 1) {
                strcat(buf, str_to_add);
            }
            else {
                strncat(buf, str_to_add, (*len - 1) - size_str);
            }
        }
        else {
            buf[*len - 1] = '\0'; 
        }
    }

    return buf;

out:
    return NULL;
}

static char *
indentation(size_t x, char *str, bool no_u_buf, size_t *s_len)
{
    char tmp_str[TMP_LEN];

    if (x < 80) {
        memset(tmp_str, ' ', x);
        tmp_str[x] = '\0';
    }
    else
        snprintf(tmp_str, TMP_LEN, "error: the indentation exceeds the number of cols.");

    if (!(str = realloc_and_append(no_u_buf, s_len, str, tmp_str)))
        goto out;

    return str;

out:
    return NULL;
}

static char *
print_enum(hid_t type, char *str, size_t *str_len, bool no_ubuf, size_t indt)
{
    char         **name  = NULL; 
    unsigned char *value = NULL; 
    int            nmembs;       
    char           tmp_str[TMP_LEN];
    int            nchars;      
    hid_t          super  = -1; 
    hid_t          native = -1; 
    size_t         super_size;  
    size_t         dst_size;    
    int            i;

    if ((nmembs = H5Tget_nmembers(type)) <= 0)
        goto out;

    if ((super = H5Tget_super(type)) < 0)
        goto out;

    
    if (H5T_SGN_NONE == H5Tget_sign(super)) {
        native = H5T_NATIVE_UINT;
    }
    else {
        native = H5T_NATIVE_INT;
    }

    super_size = H5Tget_size(super);
    dst_size   = H5Tget_size(native);

    
    name  = (char **)calloc((size_t)nmembs, sizeof(char *));
    value = (unsigned char *)calloc((size_t)nmembs, MAX(dst_size, super_size));

    for (i = 0; i < nmembs; i++) {
        if ((name[i] = H5Tget_member_name(type, (unsigned)i)) == NULL)
            goto out;
        if (H5Tget_member_value(type, (unsigned)i, value + (size_t)i * super_size) < 0)
            goto out;
    }

    
    if (native > 0) {
        if (H5Tconvert(super, native, (size_t)nmembs, value, NULL, H5P_DEFAULT) < 0)
            goto out;
    }

    

    
    for (i = 0; i < nmembs; i++) {
        if (!(str = indentation(indt + COL, str, no_ubuf, str_len)))
            goto out;
        nchars = snprintf(tmp_str, TMP_LEN, "\"%s\"", name[i]);
        if (!(str = realloc_and_append(no_ubuf, str_len, str, tmp_str)))
            goto out;
        memset(tmp_str, ' ', (size_t)MAX(3, 19 - nchars) + 1);
        tmp_str[MAX(3, 19 - nchars)] = '\0';
        if (!(str = realloc_and_append(no_ubuf, str_len, str, tmp_str)))
            goto out;

        if (H5T_SGN_NONE == H5Tget_sign(native))
            snprintf(tmp_str, TMP_LEN, "%u", *((unsigned int *)((void *)(value + (size_t)i * dst_size))));
        else
            snprintf(tmp_str, TMP_LEN, "%d", *((int *)((void *)(value + (size_t)i * dst_size))));
        if (!(str = realloc_and_append(no_ubuf, str_len, str, tmp_str)))
            goto out;

        snprintf(tmp_str, TMP_LEN, ";\n");
        if (!(str = realloc_and_append(no_ubuf, str_len, str, tmp_str)))
            goto out;
    }

    
    for (i = 0; i < nmembs; i++)
        H5free_memory(name[i]);

    free(name);
    free(value);
    H5Tclose(super);

    return str;

out:

    if (0 == nmembs) {
        str = realloc_and_append(no_ubuf, str_len, str, "\n");
        assert((indt + 4) < TMP_LEN);
        memset(tmp_str, ' ', (indt + 4) + 1);
        tmp_str[(indt + 4)] = '\0';
        str                 = realloc_and_append(no_ubuf, str_len, str, tmp_str);
        str                 = realloc_and_append(no_ubuf, str_len, str, " <empty>");
    } 

    
    if (name) {
        for (i = 0; i < nmembs; i++)
            if (name[i])
                free(name[i]);
        free(name);
    } 

    if (value)
        free(value);

    if (super >= 0)
        H5Tclose(super);

    return NULL;
}

herr_t
H5LTdtype_to_text(hid_t dtype, char *str, H5LT_lang_t lang_type, size_t *len)
{
    size_t str_len  = INCREMENT;
    char  *text_str = NULL;
    herr_t ret      = SUCCEED;

    if (lang_type <= H5LT_LANG_ERR || lang_type >= H5LT_NO_LANG)
        goto out;

    if (len && !str) {
        text_str    = (char *)calloc(str_len, sizeof(char));
        text_str[0] = '\0';
        if (!(text_str = H5LT_dtype_to_text(dtype, text_str, lang_type, &str_len, 1)))
            goto out;
        *len = strlen(text_str) + 1;
        if (text_str)
            free(text_str);
        text_str = NULL;
    }
    else if (len && str) {
        if (!(H5LT_dtype_to_text(dtype, str, lang_type, len, 0)))
            goto out;
        str[*len - 1] = '\0';
    }

    return ret;

out:
    free(text_str);

    return FAIL;
}

char *
H5LT_dtype_to_text(hid_t dtype, char *dt_str, H5LT_lang_t lang, size_t *slen, bool no_user_buf)
{
    H5T_class_t tcls;
    char        tmp_str[TMP_LEN];
    int         i;

    if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, NULL)))
        goto out;

    if (lang != H5LT_DDL) {
        snprintf(dt_str, *slen, "only DDL is supported for now");
        goto out;
    }

    if ((tcls = H5Tget_class(dtype)) < 0)
        goto out;

    switch (tcls) {
        case H5T_INTEGER:
        case H5T_BITFIELD:
            if (H5Tequal(dtype, H5T_STD_I8BE)) {
                snprintf(dt_str, *slen, "H5T_STD_I8BE");
            }
            else if (H5Tequal(dtype, H5T_STD_I8LE)) {
                snprintf(dt_str, *slen, "H5T_STD_I8LE");
            }
            else if (H5Tequal(dtype, H5T_STD_I16BE)) {
                snprintf(dt_str, *slen, "H5T_STD_I16BE");
            }
            else if (H5Tequal(dtype, H5T_STD_I16LE)) {
                snprintf(dt_str, *slen, "H5T_STD_I16LE");
            }
            else if (H5Tequal(dtype, H5T_STD_I32BE)) {
                snprintf(dt_str, *slen, "H5T_STD_I32BE");
            }
            else if (H5Tequal(dtype, H5T_STD_I32LE)) {
                snprintf(dt_str, *slen, "H5T_STD_I32LE");
            }
            else if (H5Tequal(dtype, H5T_STD_I64BE)) {
                snprintf(dt_str, *slen, "H5T_STD_I64BE");
            }
            else if (H5Tequal(dtype, H5T_STD_I64LE)) {
                snprintf(dt_str, *slen, "H5T_STD_I64LE");
            }
            else if (H5Tequal(dtype, H5T_STD_U8BE)) {
                snprintf(dt_str, *slen, "H5T_STD_U8BE");
            }
            else if (H5Tequal(dtype, H5T_STD_U8LE)) {
                snprintf(dt_str, *slen, "H5T_STD_U8LE");
            }
            else if (H5Tequal(dtype, H5T_STD_U16BE)) {
                snprintf(dt_str, *slen, "H5T_STD_U16BE");
            }
            else if (H5Tequal(dtype, H5T_STD_U16LE)) {
                snprintf(dt_str, *slen, "H5T_STD_U16LE");
            }
            else if (H5Tequal(dtype, H5T_STD_U32BE)) {
                snprintf(dt_str, *slen, "H5T_STD_U32BE");
            }
            else if (H5Tequal(dtype, H5T_STD_U32LE)) {
                snprintf(dt_str, *slen, "H5T_STD_U32LE");
            }
            else if (H5Tequal(dtype, H5T_STD_U64BE)) {
                snprintf(dt_str, *slen, "H5T_STD_U64BE");
            }
            else if (H5Tequal(dtype, H5T_STD_U64LE)) {
                snprintf(dt_str, *slen, "H5T_STD_U64LE");
            }
            else if (H5Tequal(dtype, H5T_NATIVE_SCHAR)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_SCHAR");
            }
            else if (H5Tequal(dtype, H5T_NATIVE_UCHAR)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_UCHAR");
            }
            else if (H5Tequal(dtype, H5T_NATIVE_SHORT)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_SHORT");
            }
            else if (H5Tequal(dtype, H5T_NATIVE_USHORT)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_USHORT");
            }
            else if (H5Tequal(dtype, H5T_NATIVE_INT)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_INT");
            }
            else if (H5Tequal(dtype, H5T_NATIVE_UINT)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_UINT");
            }
            else if (H5Tequal(dtype, H5T_NATIVE_LONG)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_LONG");
            }
            else if (H5Tequal(dtype, H5T_NATIVE_ULONG)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_ULONG");
            }
            else if (H5Tequal(dtype, H5T_NATIVE_LLONG)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_LLONG");
            }
            else if (H5Tequal(dtype, H5T_NATIVE_ULLONG)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_ULLONG");
            }
            else {
                snprintf(dt_str, *slen, "undefined integer");
            }

            break;
        case H5T_FLOAT:
            if (H5Tequal(dtype, H5T_IEEE_F16BE)) {
                snprintf(dt_str, *slen, "H5T_IEEE_F16BE");
            }
            else if (H5Tequal(dtype, H5T_IEEE_F16LE)) {
                snprintf(dt_str, *slen, "H5T_IEEE_F16LE");
            }
            else if (H5Tequal(dtype, H5T_IEEE_F32BE)) {
                snprintf(dt_str, *slen, "H5T_IEEE_F32BE");
            }
            else if (H5Tequal(dtype, H5T_IEEE_F32LE)) {
                snprintf(dt_str, *slen, "H5T_IEEE_F32LE");
            }
            else if (H5Tequal(dtype, H5T_IEEE_F64BE)) {
                snprintf(dt_str, *slen, "H5T_IEEE_F64BE");
            }
            else if (H5Tequal(dtype, H5T_IEEE_F64LE)) {
                snprintf(dt_str, *slen, "H5T_IEEE_F64LE");
            }
            else if (H5Tequal(dtype, H5T_FLOAT_BFLOAT16BE)) {
                snprintf(dt_str, *slen, "H5T_FLOAT_BFLOAT16BE");
            }
            else if (H5Tequal(dtype, H5T_FLOAT_BFLOAT16LE)) {
                snprintf(dt_str, *slen, "H5T_FLOAT_BFLOAT16LE");
            }
            else if (H5Tequal(dtype, H5T_FLOAT_F8E4M3)) {
                snprintf(dt_str, *slen, "H5T_FLOAT_F8E4M3");
            }
            else if (H5Tequal(dtype, H5T_FLOAT_F8E5M2)) {
                snprintf(dt_str, *slen, "H5T_FLOAT_F8E5M2");
            }
#ifdef H5_HAVE__FLOAT16
            else if (H5Tequal(dtype, H5T_NATIVE_FLOAT16)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_FLOAT16");
            }
#endif
            else if (H5Tequal(dtype, H5T_NATIVE_FLOAT)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_FLOAT");
            }
            else if (H5Tequal(dtype, H5T_NATIVE_DOUBLE)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_DOUBLE");
            }
            else if (H5Tequal(dtype, H5T_NATIVE_LDOUBLE)) {
                snprintf(dt_str, *slen, "H5T_NATIVE_LDOUBLE");
            }
            else {
                snprintf(dt_str, *slen, "undefined float");
            }

            break;
        case H5T_STRING: {
            
            hid_t       str_type;
            H5T_order_t order;
            hid_t       tmp_type;
            size_t      size;
            H5T_str_t   str_pad;
            H5T_cset_t  cset;
            htri_t      is_vlstr;

            if ((tmp_type = H5Tcopy(dtype)) < 0)
                goto out;
            if ((size = H5Tget_size(tmp_type)) == 0)
                goto out;
            if ((str_pad = H5Tget_strpad(tmp_type)) < 0)
                goto out;
            if ((cset = H5Tget_cset(tmp_type)) < 0)
                goto out;
            if ((is_vlstr = H5Tis_variable_str(tmp_type)) < 0)
                goto out;

            
            snprintf(dt_str, *slen, "H5T_STRING {\n");
            indent += COL;

            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;

            if (is_vlstr)
                snprintf(tmp_str, TMP_LEN, "STRSIZE H5T_VARIABLE;\n");
            else
                snprintf(tmp_str, TMP_LEN, "STRSIZE %d;\n", (int)size);

            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;

            if (str_pad == H5T_STR_NULLTERM)
                snprintf(tmp_str, TMP_LEN, "STRPAD H5T_STR_NULLTERM;\n");
            else if (str_pad == H5T_STR_NULLPAD)
                snprintf(tmp_str, TMP_LEN, "STRPAD H5T_STR_NULLPAD;\n");
            else if (str_pad == H5T_STR_SPACEPAD)
                snprintf(tmp_str, TMP_LEN, "STRPAD H5T_STR_SPACEPAD;\n");
            else
                snprintf(tmp_str, TMP_LEN, "STRPAD H5T_STR_ERROR;\n");

            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;

            if (cset == H5T_CSET_ASCII)
                snprintf(tmp_str, TMP_LEN, "CSET H5T_CSET_ASCII;\n");
            else if (cset == H5T_CSET_UTF8)
                snprintf(tmp_str, TMP_LEN, "CSET H5T_CSET_UTF8;\n");
            else
                snprintf(tmp_str, TMP_LEN, "CSET unknown;\n");

            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            
            if ((str_type = H5Tcopy(H5T_C_S1)) < 0)
                goto out;
            if (is_vlstr) {
                if (H5Tset_size(str_type, H5T_VARIABLE) < 0)
                    goto out;
            }
            else {
                if (H5Tset_size(str_type, size) < 0)
                    goto out;
            }
            if (H5Tset_cset(str_type, cset) < 0)
                goto out;
            if (H5Tset_strpad(str_type, str_pad) < 0)
                goto out;

            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;

            
            if (H5Tequal(tmp_type, str_type)) {
                snprintf(tmp_str, TMP_LEN, "CTYPE H5T_C_S1;\n");
                if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                    goto out;
                goto next;
            }

            
            if ((order = H5Tget_order(tmp_type)) < 0)
                goto out;
            if (order == H5T_ORDER_LE) {
                if (H5Tset_order(str_type, H5T_ORDER_LE) < 0)
                    goto out;
            }
            else if (order == H5T_ORDER_BE) {
                if (H5Tset_order(str_type, H5T_ORDER_BE) < 0)
                    goto out;
            }

            if (H5Tequal(tmp_type, str_type)) {
                snprintf(tmp_str, TMP_LEN, "CTYPE H5T_C_S1;\n");
                if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                    goto out;
                goto next;
            }

            
            if (H5Tclose(str_type) < 0)
                goto out;
            if ((str_type = H5Tcopy(H5T_FORTRAN_S1)) < 0)
                goto out;
            if (H5Tset_cset(str_type, cset) < 0)
                goto out;
            if (H5Tset_size(str_type, size) < 0)
                goto out;
            if (H5Tset_strpad(str_type, str_pad) < 0)
                goto out;

            
            if (H5Tequal(tmp_type, str_type)) {
                snprintf(tmp_str, TMP_LEN, "CTYPE H5T_FORTRAN_S1;\n");
                if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                    goto out;
                goto next;
            }

            
            if ((order = H5Tget_order(tmp_type)) < 0)
                goto out;
            if (order == H5T_ORDER_LE) {
                if (H5Tset_order(str_type, H5T_ORDER_LE) < 0)
                    goto out;
            }
            else if (order == H5T_ORDER_BE) {
                if (H5Tset_order(str_type, H5T_ORDER_BE) < 0)
                    goto out;
            }

            
            if (H5Tequal(tmp_type, str_type)) {
                snprintf(tmp_str, TMP_LEN, "CTYPE H5T_FORTRAN_S1;\n");
                if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                    goto out;
                goto next;
            }

            
            snprintf(tmp_str, TMP_LEN, "CTYPE unknown_one_character_type;\n");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

next:
            H5Tclose(str_type);
            H5Tclose(tmp_type);

            
            indent -= COL;
            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;
            snprintf(tmp_str, TMP_LEN, "}");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            break;
        }
        case H5T_OPAQUE: {
            char *tag = NULL;

            
            snprintf(dt_str, *slen, "H5T_OPAQUE {\n");
            indent += COL;

            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;
            snprintf(tmp_str, TMP_LEN, "OPQ_SIZE %lu;\n", (unsigned long)H5Tget_size(dtype));
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;
            tag = H5Tget_tag(dtype);
            if (tag) {
                snprintf(tmp_str, TMP_LEN, "OPQ_TAG \"%s\";\n", tag);
                if (tag)
                    H5free_memory(tag);
                tag = NULL;
            }
            else
                snprintf(tmp_str, TMP_LEN, "OPQ_TAG \"\";\n");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            
            indent -= COL;
            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;
            snprintf(tmp_str, TMP_LEN, "}");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            break;
        }
        case H5T_ENUM: {
            hid_t  super;
            size_t super_len;
            char  *stmp = NULL;

            
            snprintf(dt_str, *slen, "H5T_ENUM {\n");
            indent += COL;
            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;

            if ((super = H5Tget_super(dtype)) < 0)
                goto out;
            if (H5LTdtype_to_text(super, NULL, lang, &super_len) < 0)
                goto out;
            stmp = (char *)calloc(super_len, sizeof(char));
            if (H5LTdtype_to_text(super, stmp, lang, &super_len) < 0) {
                free(stmp);
                goto out;
            }
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, stmp))) {
                free(stmp);
                goto out;
            }

            if (stmp)
                free(stmp);
            stmp = NULL;

            snprintf(tmp_str, TMP_LEN, ";\n");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;
            H5Tclose(super);

            if (!(dt_str = print_enum(dtype, dt_str, slen, no_user_buf, indent)))
                goto out;

            
            indent -= COL;
            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;
            snprintf(tmp_str, TMP_LEN, "}");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            break;
        }
        case H5T_VLEN: {
            hid_t  super;
            size_t super_len;
            char  *stmp = NULL;

            
            snprintf(dt_str, *slen, "H5T_VLEN {\n");
            indent += COL;
            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;

            if ((super = H5Tget_super(dtype)) < 0)
                goto out;
            if (H5LTdtype_to_text(super, NULL, lang, &super_len) < 0)
                goto out;
            stmp = (char *)calloc(super_len, sizeof(char));
            if (H5LTdtype_to_text(super, stmp, lang, &super_len) < 0) {
                free(stmp);
                goto out;
            }
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, stmp))) {
                free(stmp);
                goto out;
            }

            if (stmp)
                free(stmp);
            stmp = NULL;
            snprintf(tmp_str, TMP_LEN, "\n");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;
            H5Tclose(super);

            
            indent -= COL;
            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;
            snprintf(tmp_str, TMP_LEN, "}");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            break;
        }
        case H5T_ARRAY: {
            hid_t   super;
            size_t  super_len;
            char   *stmp = NULL;
            hsize_t dims[H5S_MAX_RANK];
            int     ndims;

            
            snprintf(dt_str, *slen, "H5T_ARRAY {\n");
            indent += COL;
            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;

            
            if ((ndims = H5Tget_array_ndims(dtype)) < 0)
                goto out;
            if (H5Tget_array_dims2(dtype, dims) < 0)
                goto out;

            
            for (i = 0; i < ndims; i++) {
                snprintf(tmp_str, TMP_LEN, "[%d]", (int)dims[i]);
                if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                    goto out;
            }
            snprintf(tmp_str, TMP_LEN, " ");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            if ((super = H5Tget_super(dtype)) < 0)
                goto out;
            if (H5LTdtype_to_text(super, NULL, lang, &super_len) < 0)
                goto out;
            stmp = (char *)calloc(super_len, sizeof(char));
            if (H5LTdtype_to_text(super, stmp, lang, &super_len) < 0) {
                free(stmp);
                goto out;
            }
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, stmp))) {
                free(stmp);
                goto out;
            }
            if (stmp)
                free(stmp);
            stmp = NULL;
            snprintf(tmp_str, TMP_LEN, "\n");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;
            H5Tclose(super);

            
            indent -= COL;
            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;
            snprintf(tmp_str, TMP_LEN, "}");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            break;
        }
        case H5T_COMPOUND: {
            char       *mname = NULL;
            hid_t       mtype;
            size_t      moffset;
            H5T_class_t mclass;
            size_t      mlen;
            char       *mtmp = NULL;
            int         nmembs;

            if ((nmembs = H5Tget_nmembers(dtype)) < 0)
                goto out;

            snprintf(dt_str, *slen, "H5T_COMPOUND {\n");
            indent += COL;

            for (i = 0; i < nmembs; i++) {
                if ((mname = H5Tget_member_name(dtype, (unsigned)i)) == NULL)
                    goto out;
                if ((mtype = H5Tget_member_type(dtype, (unsigned)i)) < 0)
                    goto out;
                moffset = H5Tget_member_offset(dtype, (unsigned)i);
                if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                    goto out;

                if ((mclass = H5Tget_class(mtype)) < 0)
                    goto out;
                if (H5T_COMPOUND == mclass)
                    indent += COL;

                if (H5LTdtype_to_text(mtype, NULL, lang, &mlen) < 0)
                    goto out;
                mtmp = (char *)calloc(mlen, sizeof(char));
                if (H5LTdtype_to_text(mtype, mtmp, lang, &mlen) < 0) {
                    free(mtmp);
                    goto out;
                }
                if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, mtmp))) {
                    free(mtmp);
                    goto out;
                }
                if (mtmp)
                    free(mtmp);
                mtmp = NULL;

                if (H5T_COMPOUND == mclass)
                    indent -= COL;

                snprintf(tmp_str, TMP_LEN, " \"%s\"", mname);
                if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                    goto out;
                if (mname)
                    H5free_memory(mname);
                mname = NULL;

                snprintf(tmp_str, TMP_LEN, " : %lu;\n", (unsigned long)moffset);
                if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                    goto out;
            }

            
            indent -= COL;
            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;
            snprintf(tmp_str, TMP_LEN, "}");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            break;
        }
        case H5T_COMPLEX: {
            hid_t  super;
            size_t super_len;
            char  *stmp = NULL;

            
            snprintf(dt_str, *slen, "H5T_COMPLEX {\n");
            indent += COL;
            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;

            if ((super = H5Tget_super(dtype)) < 0)
                goto out;
            if (H5LTdtype_to_text(super, NULL, lang, &super_len) < 0)
                goto out;
            stmp = (char *)calloc(super_len, sizeof(char));
            if (H5LTdtype_to_text(super, stmp, lang, &super_len) < 0) {
                free(stmp);
                goto out;
            }
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, stmp))) {
                free(stmp);
                goto out;
            }

            if (stmp)
                free(stmp);
            stmp = NULL;
            snprintf(tmp_str, TMP_LEN, "\n");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;
            H5Tclose(super);

            
            indent -= COL;
            if (!(dt_str = indentation(indent + COL, dt_str, no_user_buf, slen)))
                goto out;
            snprintf(tmp_str, TMP_LEN, "}");
            if (!(dt_str = realloc_and_append(no_user_buf, slen, dt_str, tmp_str)))
                goto out;

            break;
        }
        case H5T_TIME:
            snprintf(dt_str, *slen, "H5T_TIME: not yet implemented");
            break;
        case H5T_NO_CLASS:
            snprintf(dt_str, *slen, "H5T_NO_CLASS");
            break;
        case H5T_REFERENCE:
            if (H5Tequal(dtype, H5T_STD_REF_DSETREG) == true) {
                snprintf(dt_str, *slen, " H5T_REFERENCE { H5T_STD_REF_DSETREG }");
            }
            else {
                snprintf(dt_str, *slen, " H5T_REFERENCE { H5T_STD_REF_OBJECT }");
            }
            break;
        case H5T_NCLASSES:
            break;
        default:
            snprintf(dt_str, *slen, "unknown data type");
    }

    return dt_str;

out:
    return NULL;
}

herr_t
H5LTget_attribute_string(hid_t loc_id, const char *obj_name, const char *attr_name, char *data)
{
    
    hid_t obj_id;

    
    if (obj_name == NULL)
        return -1;
    if (attr_name == NULL)
        return -1;

    
    if ((obj_id = H5Oopen(loc_id, obj_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if (H5LT_get_attribute_disk(obj_id, attr_name, data) < 0) {
        H5Oclose(obj_id);
        return -1;
    }

    
    if (H5Oclose(obj_id) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute_char(hid_t loc_id, const char *obj_name, const char *attr_name, char *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, H5T_NATIVE_CHAR, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute_uchar(hid_t loc_id, const char *obj_name, const char *attr_name, unsigned char *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, H5T_NATIVE_UCHAR, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute_short(hid_t loc_id, const char *obj_name, const char *attr_name, short *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, H5T_NATIVE_SHORT, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute_ushort(hid_t loc_id, const char *obj_name, const char *attr_name, unsigned short *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, H5T_NATIVE_USHORT, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute_int(hid_t loc_id, const char *obj_name, const char *attr_name, int *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, H5T_NATIVE_INT, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute_uint(hid_t loc_id, const char *obj_name, const char *attr_name, unsigned int *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, H5T_NATIVE_UINT, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute_long(hid_t loc_id, const char *obj_name, const char *attr_name, long *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, H5T_NATIVE_LONG, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute_long_long(hid_t loc_id, const char *obj_name, const char *attr_name, long long *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, H5T_NATIVE_LLONG, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute_ulong(hid_t loc_id, const char *obj_name, const char *attr_name, unsigned long *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, H5T_NATIVE_ULONG, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute_ullong(hid_t loc_id, const char *obj_name, const char *attr_name, unsigned long long *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, H5T_NATIVE_ULLONG, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute_float(hid_t loc_id, const char *obj_name, const char *attr_name, float *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, H5T_NATIVE_FLOAT, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute_double(hid_t loc_id, const char *obj_name, const char *attr_name, double *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, H5T_NATIVE_DOUBLE, data) < 0)
        return -1;

    return 0;
}

herr_t
H5LTget_attribute(hid_t loc_id, const char *obj_name, const char *attr_name, hid_t mem_type_id, void *data)
{
    
    if (H5LT_get_attribute_mem(loc_id, obj_name, attr_name, mem_type_id, data) < 0)
        return -1;

    return 0;
}

static herr_t
H5LT_get_attribute_mem(hid_t loc_id, const char *obj_name, const char *attr_name, hid_t mem_type_id,
                       void *data)
{
    
    hid_t obj_id  = -1;
    hid_t attr_id = -1;

    
    if (obj_name == NULL)
        return -1;
    if (attr_name == NULL)
        return -1;

    
    if ((obj_id = H5Oopen(loc_id, obj_name, H5P_DEFAULT)) < 0)
        goto out;

    if ((attr_id = H5Aopen(obj_id, attr_name, H5P_DEFAULT)) < 0)
        goto out;

    if (H5Aread(attr_id, mem_type_id, data) < 0)
        goto out;

    if (H5Aclose(attr_id) < 0)
        goto out;
    attr_id = -1;

    
    if (H5Oclose(obj_id) < 0)
        goto out;
    obj_id = -1;

    return 0;

out:
    if (obj_id > 0)
        H5Oclose(obj_id);
    if (attr_id > 0)
        H5Aclose(attr_id);
    return -1;
}

herr_t
H5LT_get_attribute_disk(hid_t loc_id, const char *attr_name, void *attr_out)
{
    
    hid_t attr_id;
    hid_t attr_type;

    if ((attr_id = H5Aopen(loc_id, attr_name, H5P_DEFAULT)) < 0)
        return -1;

    if ((attr_type = H5Aget_type(attr_id)) < 0)
        goto out;

    if (H5Aread(attr_id, attr_type, attr_out) < 0)
        goto out;

    if (H5Tclose(attr_type) < 0)
        goto out;

    if (H5Aclose(attr_id) < 0)
        return -1;

    return 0;

out:
    H5Tclose(attr_type);
    H5Aclose(attr_id);
    return -1;
}

herr_t
H5LT_set_attribute_string(hid_t dset_id, const char *name, const char *buf)
{
    hid_t  tid;
    hid_t  sid = -1;
    hid_t  aid = -1;
    htri_t has_attr;
    size_t size;

    
    if ((has_attr = H5Aexists(dset_id, name)) < 0)
        return FAIL;
    if (has_attr > 0)
        if (H5Adelete(dset_id, name) < 0)
            return FAIL;

    
    if ((tid = H5Tcopy(H5T_C_S1)) < 0)
        return FAIL;

    size = strlen(buf) + 1; 

    if (H5Tset_size(tid, (size_t)size) < 0)
        goto out;

    if (H5Tset_strpad(tid, H5T_STR_NULLTERM) < 0)
        goto out;

    if ((sid = H5Screate(H5S_SCALAR)) < 0)
        goto out;

    
    if ((aid = H5Acreate2(dset_id, name, tid, sid, H5P_DEFAULT, H5P_DEFAULT)) < 0)
        goto out;

    if (H5Awrite(aid, tid, buf) < 0)
        goto out;

    if (H5Aclose(aid) < 0)
        goto out;

    if (H5Sclose(sid) < 0)
        goto out;

    if (H5Tclose(tid) < 0)
        goto out;

    return SUCCEED;

    
out:
    H5E_BEGIN_TRY
    {
        H5Aclose(aid);
        H5Tclose(tid);
        H5Sclose(sid);
    }
    H5E_END_TRY
    return FAIL;
}

htri_t
H5LTpath_valid(hid_t loc_id, const char *path, bool check_object_valid)
{
    char      *tmp_path = NULL; 
    char      *curr_name;       
    char      *delimit;         
    H5I_type_t obj_type;
    htri_t     link_exists, obj_exists;
    size_t     path_length;
    htri_t     ret_value;

    
    ret_value = false;

    
    if (path == NULL) {
        ret_value = FAIL;
        goto done;
    }

    
    if ((obj_type = H5Iget_type(loc_id)) == H5I_BADID) {
        ret_value = FAIL;
        goto done;
    }

    
    path_length = strlen(path);

    
    if (strncmp(path, ".", path_length) == 0) {
        if (check_object_valid) {
            obj_exists = H5Oexists_by_name(loc_id, path, H5P_DEFAULT);
            ret_value  = obj_exists;
            goto done;
        }
        else {
            ret_value = true; 
            goto done;
        }
    }

    
    if (NULL == (tmp_path = strdup(path))) {
        ret_value = FAIL;
        goto done;
    }

    curr_name = tmp_path;

    
    if (strncmp(path, "/", 1) == 0)
        curr_name++;

    
    if (strncmp(path, "./", 2) == 0)
        curr_name += 2;

    while ((delimit = strchr(curr_name, '/')) != NULL) {
        
        *delimit = '\0';

        obj_exists = false;
        if ((link_exists = H5Lexists(loc_id, tmp_path, H5P_DEFAULT)) < 0) {
            ret_value = FAIL;
            goto done;
        }

        
        if (link_exists != true) {
            ret_value = false;
            goto done;
        }

        
        if ((obj_exists = H5Oexists_by_name(loc_id, tmp_path, H5P_DEFAULT)) < 0) {
            ret_value = FAIL;
            goto done;
        }

        if (obj_exists != true)
            break;

        
        *delimit = '/';

        
        curr_name = delimit + 1;

    } 

    

    
    if ((link_exists = H5Lexists(loc_id, tmp_path, H5P_DEFAULT)) < 0) {
        ret_value = FAIL;
    }
    else {
        ret_value = link_exists;
        
        if (check_object_valid == true && link_exists == true) {
            if ((obj_exists = H5Oexists_by_name(loc_id, tmp_path, H5P_DEFAULT)) < 0) {
                ret_value = FAIL;
            }
            else {
                ret_value = obj_exists;
            }
        }
    }

done:
    if (tmp_path != NULL)
        free(tmp_path);

    return ret_value;
}
