% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blueprint-formula-default.R, R/mold.R
\name{default_formula_blueprint}
\alias{default_formula_blueprint}
\alias{mold.formula}
\title{Default formula blueprint}
\usage{
default_formula_blueprint(
  intercept = FALSE,
  allow_novel_levels = FALSE,
  indicators = TRUE
)

\method{mold}{formula}(formula, data, ..., blueprint = NULL)
}
\arguments{
\item{intercept}{A logical. Should an intercept be included in the
processed data? This information is used by the \code{process} function
in the \code{mold} and \code{forge} function list.}

\item{allow_novel_levels}{A logical. Should novel factor levels be allowed at
prediction time? This information is used by the \code{clean} function in the
\code{forge} function list, and is passed on to \code{\link[=scream]{scream()}}.}

\item{indicators}{A logical. Should factors be expanded into dummy variables?}

\item{formula}{A formula specifying the predictors and the outcomes.}

\item{data}{A data frame or matrix containing the outcomes and predictors.}

\item{...}{Not used.}

\item{blueprint}{A preprocessing \code{blueprint}. If left as \code{NULL}, then a
\code{\link[=default_formula_blueprint]{default_formula_blueprint()}} is used.}
}
\value{
For \code{default_formula_blueprint()}, a formula blueprint.
}
\description{
This pages holds the details for the formula preprocessing blueprint. This
is the blueprint used by default from \code{mold()} if \code{x} is a formula.
}
\details{
While not different from base R, the behavior of expanding factors into
dummy variables when an intercept is \emph{not} present should be documented.
\itemize{
\item When an intercept is present, factors are expanded into \code{K-1} new columns,
where \code{K} is the number of levels in the factor.
\item When an intercept is \emph{not} present, factors are expanded into all \code{K}
columns (one-hot encoding).
}

Offsets can be included in the formula method through the use of the inline
function \code{\link[stats:offset]{stats::offset()}}. These are returned as a tibble with 1 column
named \code{".offset"} in the \verb{$extras$offset} slot of the return value.
}
\section{Mold}{


When \code{mold()} is used with the default formula blueprint:
\itemize{
\item Predictors
\itemize{
\item The RHS of the \code{formula} is isolated, and converted to its own
1 sided formula: \code{~ RHS}.
\item Runs \code{\link[stats:model.frame]{stats::model.frame()}} on the RHS formula and uses \code{data}.
\item If \code{indicators = TRUE}, it then runs \code{\link[stats:model.matrix]{stats::model.matrix()}} on the
result.
\item If \code{indicators = FALSE}, factors are removed before \code{model.matrix()}
is run, and then added back afterwards. No interactions or inline
functions involving factors are allowed.
\item If any offsets are present from using \code{offset()}, then they are
extracted with \code{\link[=model_offset]{model_offset()}}.
\item If \code{intercept = TRUE}, adds an intercept column.
\item Coerces the result of the above steps to a tibble.
}
\item Outcomes
\itemize{
\item The LHS of the \code{formula} is isolated, and converted to its own
1 sided formula: \code{~ LHS}.
\item Runs \code{\link[stats:model.frame]{stats::model.frame()}} on the LHS formula and uses \code{data}.
\item Coerces the result of the above steps to a tibble.
}
}
}

\section{Forge}{


When \code{forge()} is used with the default formula blueprint:
\itemize{
\item It calls \code{\link[=shrink]{shrink()}} to trim \code{new_data} to only the required columns and
coerce \code{new_data} to a tibble.
\item It calls \code{\link[=scream]{scream()}} to perform validation on the structure of the columns
of \code{new_data}.
\item Predictors
\itemize{
\item It runs \code{\link[stats:model.frame]{stats::model.frame()}} on \code{new_data} using the stored terms
object corresponding to the \emph{predictors}.
\item If, in the original \code{\link[=mold]{mold()}} call, \code{indicators = TRUE} was set, it
then runs \code{\link[stats:model.matrix]{stats::model.matrix()}} on the result.
\item If, in the original \code{\link[=mold]{mold()}} call, \code{indicators = FALSE} was set, it
runs \code{\link[stats:model.matrix]{stats::model.matrix()}} on the result without the factor columns,
and then adds them on afterwards.
\item If any offsets are present from using \code{offset()} in the original call
to \code{\link[=mold]{mold()}}, then they are extracted with \code{\link[=model_offset]{model_offset()}}.
\item If \code{intercept = TRUE} in the original call to \code{\link[=mold]{mold()}}, then an
intercept column is added.
\item It coerces the result of the above steps to a tibble.
}
\item Outcomes
\itemize{
\item It runs \code{\link[stats:model.frame]{stats::model.frame()}} on \code{new_data} using the
stored terms object corresponding to the \emph{outcomes}.
\item Coerces the result to a tibble.
}
}
}

\section{Differences From Base R}{


There are a number of differences from base R regarding how formulas are
processed by \code{mold()} that require some explanation.

Multivariate outcomes can be specified on the LHS using syntax that is
similar to the RHS (i.e. \code{outcome_1 + outcome_2 ~ predictors}).
If any complex calculations are done on the LHS and they return matrices
(like \code{\link[stats:poly]{stats::poly()}}), then those matrices are flattened into multiple
columns of the tibble after the call to \code{model.frame()}. While this is
possible, it is not recommended, and if a large amount of preprocessing is
required on the outcomes, then you are better off
using a \code{\link[recipes:recipe]{recipes::recipe()}}.

Global variables are \emph{not} allowed in the formula. An error will be thrown
if they are included. All terms in the formula should come from \code{data}.

By default, intercepts are \emph{not} included in the predictor output from the
formula. To include an intercept, set
\code{blueprint = default_formula_blueprint(intercept = TRUE)}. The rationale
for this is that many packages either always require or never allow an
intercept (for example, the \code{earth} package), and they do a large amount of
extra work to keep the user from supplying one or removing it. This
interface standardizes all of that flexibility in one place.
}

\examples{
# ---------------------------------------------------------------------------
# Setup

train <- iris[1:100,]
test <- iris[101:150,]

# ---------------------------------------------------------------------------
# Formula Example

# Call mold() with the training data
processed <- mold(
  log(Sepal.Width) ~ Sepal.Length + Species,
  train,
  blueprint = default_formula_blueprint(intercept = TRUE)
)

# Then, call forge() with the blueprint and the test data
# to have it preprocess the test data in the same way
forge(test, processed$blueprint)

# Use `outcomes = TRUE` to also extract the preprocessed outcome
forge(test, processed$blueprint, outcomes = TRUE)

# ---------------------------------------------------------------------------
# Factors without an intercept

# No intercept is added by default
processed <- mold(Sepal.Width ~ Species, train)

# So factor columns are completely expanded
# into all `K` columns (the number of levels)
processed$predictors

# ---------------------------------------------------------------------------
# Global variables

y <- rep(1, times = nrow(train))

# In base R, global variables are allowed in a model formula
frame <- model.frame(Species ~ y + Sepal.Length, train)
head(frame)

# mold() does not allow them, and throws an error
tryCatch(
  expr = mold(Species ~ y + Sepal.Length, train),
  error = function(e) print(e$message)
)

# ---------------------------------------------------------------------------
# Dummy variables and interactions

# By default, factor columns are expanded
# and interactions are created, both by
# calling model.matrix(). Some models (like
# tree based models) can take factors directly
# but still might want to use the formula method.
# In those cases, set `indicators = FALSE` to not
# run model.matrix() on factor columns. Interactions
# are still allowed and are run on numeric columns.

blueprint_no_indicators <- default_formula_blueprint(indicators = FALSE)

processed <- mold(
  ~ Species + Sepal.Width:Sepal.Length,
  train,
  blueprint = blueprint_no_indicators
)

processed$predictors

# An informative error is thrown when `indicators = FALSE` and
# factors are present in interaction terms or in inline functions
try(mold(Sepal.Width ~ Sepal.Length:Species, train, blueprint = blueprint_no_indicators))
try(mold(Sepal.Width ~ paste0(Species), train, blueprint = blueprint_no_indicators))

# ---------------------------------------------------------------------------
# Multivariate outcomes

# Multivariate formulas can be specified easily
processed <- mold(Sepal.Width + log(Sepal.Length) ~ Species, train)
processed$outcomes

# Inline functions on the LHS are run, but any matrix
# output is flattened (like what happens in `model.matrix()`)
# (essentially this means you don't wind up with columns
# in the tibble that are matrices)
processed <- mold(poly(Sepal.Length, degree = 2) ~ Species, train)
processed$outcomes

# TRUE
ncol(processed$outcomes) == 2

# Multivariate formulas specified in mold()
# carry over into forge()
forge(test, processed$blueprint, outcomes = TRUE)

# ---------------------------------------------------------------------------
# Offsets

# Offsets are handled specially in base R, so they deserve special
# treatment here as well. You can add offsets using the inline function
# offset()
processed <- mold(Sepal.Width ~ offset(Sepal.Length) + Species, train)

processed$extras$offset

# Multiple offsets can be included, and they get added together
processed <- mold(
  Sepal.Width ~ offset(Sepal.Length) + offset(Petal.Width),
  train
)

identical(
  processed$extras$offset$.offset,
  train$Sepal.Length + train$Petal.Width
)

# Forging test data will also require
# and include the offset
forge(test, processed$blueprint)

# ---------------------------------------------------------------------------
# Intercept only

# Because `1` and `0` are intercept modifying terms, they are
# not allowed in the formula and are controlled by the
# `intercept` argument of the blueprint. To use an intercept
# only formula, you should supply `NULL` on the RHS of the formula.
mold(~ NULL, train, blueprint = default_formula_blueprint(intercept = TRUE))

}
