
#' Calculate and summarize distribution of susceptibilities by gene
#'
#' @description This function will calculate the distribution of
#' susceptibilities by gene.
#' @param x a `data.frame` containing the data.
#' @param cutoff value for percent susceptible cutoff. Numeric.
#' @param control value used to denote the susceptible control in the `gene`
#'  column. Character.
#' @param sample column providing the unique identification for each sample
#'  being tested. Character.
#' @param gene column providing the gene(s) being tested. Character.
#' @param perc_susc column providing the percent susceptible reactions.
#'  Character.
#'
#' @examples
#' # Using the built-in data set, P_sojae_survey
#' data(P_sojae_survey)
#'
#' P_sojae_survey
#'
#' # calculate susceptibilities with a 60 % cutoff value
#' susc <- summarize_gene(x = P_sojae_survey,
#'                        cutoff = 60,
#'                        control = "susceptible",
#'                        sample = "Isolate",
#'                        gene = "Rps",
#'                        perc_susc = "perc.susc")
#' susc
#'
#' @return  returns an object of [class] `hagis.gene.summary`
#' An object of class `hagis.gene.summary` is a\cr [data.table::data.table()]
#'  containing the following components columns
#'   \describe{
#'     \item{gene}{the gene}
#'     \item{N_virulent_isolates}{the total number virulent isolates for a given
#'     gene in the `gene` column}
#'     \item{percent_pathogenic}{the frequency with which a gene is pathogenic}
#'   }
#' @export summarize_gene

summarize_gene <- function(x,
                           cutoff,
                           control,
                           sample,
                           gene,
                           perc_susc) {
  # check inputs and rename columns to work with this package
  x <- .check_inputs(
    .x = x,
    .cutoff = cutoff,
    .control = control,
    .sample = sample,
    .gene = gene,
    .perc_susc = perc_susc
  )

  # CRAN NOTE avoidance
  susceptible.1 <- percent_pathogenic <- N_virulent_isolates <- NULL

  # summarise the reactions, create susceptible.1 column, see
  # internal_functions.R
  x <- .binary_cutoff(.x = x, .cutoff = cutoff)

  # create new data.table with percentages
  y <-
    x[, list(N_virulent_isolates = sum(susceptible.1)), by = list(gene)]
  y[, percent_pathogenic :=
      (N_virulent_isolates) / max(N_virulent_isolates) * 100]

  # Set new class
  class(y) <- union("hagis.gene.summary", class(y))
  return(y)
}

#' @importFrom ggplot2 autoplot
#' @export
ggplot2::autoplot

#' Plot \pkg{hagis} summary objects
#'
#' @description Creates a \pkg{ggplot2} object of the gene summaries calculated
#'  by [summarize_gene()]
#' @param object a `hagis.gene.summary` object generated by [summarize_gene()].
#'  Character.
#' @param type a vector of values for which the bar plot is desired. Specify
#'  whether to return a graph of the percent pathogenic isolates, `percentage`,
#'  or as the count, "`count`". Character.
#' @param color a named or hexadecimal color value to use for the bar color
#' @param order sort the x-axis of the bar chart by ascending or descending
#' order of `N_virulent_isolates` or `percent_pathogenic`. Accepts `ascending`
#' or `descending` input values. Defaults to `gene` name. Character.
#' @param ... passed to the chosen `geom(s)`
#' @return A \link[ggplot2]{ggplot2} plot
#' @method autoplot hagis.gene.summary
#' @examples
#' # Using the built-in data set, P_sojae_survey
#' data(P_sojae_survey)
#'
#' # calculate susceptibilities with a 60 % cutoff value
#' susc <- summarize_gene(x = P_sojae_survey,
#'                        cutoff = 60,
#'                        control = "susceptible",
#'                        sample = "Isolate",
#'                        gene = "Rps",
#'                        perc_susc = "perc.susc")
#'
#' # Visualize the summary of genes
#' autoplot(susc, type = "percentage")
#' @export

autoplot.hagis.gene.summary <-
  function(object,
           type,
           color = NULL,
           order = NULL,
           ...) {
    # CRAN NOTE avoidance
    gene <- percent_pathogenic <- N_virulent_isolates <- NULL

    # order cols based on user input
    if (!is.null(order)) {
      if (order == "ascending") {
        setorder(object,
                             cols = N_virulent_isolates)
        object$order <- seq_len(nrow(object))
      } else if (order == "descending") {
        setorder(x = object,
                             cols = -N_virulent_isolates)
        object$order <- seq_len(nrow(object))
      }
    } else
      # if no order is specified
      setorder(object, cols = gene)
    object$order <- seq_len(nrow(object))

    plot_percentage <- function(.data, .color) {
      perc_plot <- ggplot2::ggplot(data = .data,
                                   ggplot2::aes(x = stats::reorder(gene, order),
                                                y = percent_pathogenic)) +
        ggplot2::labs(y = "Percent of samples",
                      x = "Gene") +
        ggplot2::ggtitle(expression("Percentage of samples pathogenic"))

      if (!is.null(.color)) {
        perc_plot +
          ggplot2::geom_col(fill = .color,
                            colour = .color)
      } else {
        perc_plot +
          ggplot2::geom_col()
      }
    }

    plot_count <- function(.data, .color) {
      N_virulent_isolates <- NULL
      num_plot <- ggplot2::ggplot(data = .data,
                                  ggplot2::aes(x = stats::reorder(gene, order),
                                               y = N_virulent_isolates)) +
        ggplot2::labs(y = "Number of samples",
                      x = "Gene") +
        ggplot2::ggtitle(expression("Number of samples pathogenic"))

      if (!is.null(.color)) {
        num_plot +
          ggplot2::geom_col(fill = .color,
                            colour = .color)
      } else {
        num_plot +
          ggplot2::geom_col()
      }
    }

    if (type == "percentage") {
      plot_percentage(.data = object, .color = color)
    } else if (type == "count") {
      plot_count(.data = object, .color = color)
    } else
      stop(.call = FALSE,
           "You have entered an invalid `type`.")
  }
