#' Sample correlator matrix
#'
#' @description
#' Sample data for a correlation function for a 24 cube times 48 lattice QCD
#' simulation representing a pion propagation. It is stored in form of an
#' object of class \code{cf}, which is derived from \code{list}.
#' 
#' @name correlatormatrix
#' 
#' @docType data
#' @format list of 7 elements: "nrObs" "Time" "nrStypes" "symmetrised" "cf"
#' "icf" "cf0"
#' @keywords datasets
#' @examples
#' 
#' data("correlatormatrix")
#' 
NULL

#' Sample loop data
#'
#' @description
#' Sample data for fermion loops for a 24 cube times 48 lattice QCD
#' simulation. It is stored in form of a list.
#' 
#' @name loopdata
#'
#' @docType data
#' @format list of 10 elements: "nrObs" "Time" "nrStypes" "symmetrised" "cf" 
#' "icf" "scf" "iscf" "nrSamples" "obs"
#' @keywords datasets
#' @examples
#'
#' data("loopdata")
#'
NULL

#' Internal Hadron Functions
#' 
#' Internal hadron functions
#' 
#' These are not to be called by the user (or in some cases are just waiting
#' for proper documentation to be written :).
#' 
#' @aliases arrangeCor.vector arrangeCor.pion arrangeCor.b1 arrangeCor.a0
#' getNxNmatrix ChiSqr.1mass ChiSqr.2mass ChiSqr.3mass fitmasses.vector
#' fitmasses.vector.boot fitmasses.pion fitmasses.pion.boot fitmasses.b1
#' fitmasses.b1.boot fitmasses.a0 fitmasses.a0.boot fitf.pion mean.index
#' fitmpcac.pion effectivemass effmass effmass2 read_nissa_textcf_kernel
#' compute.qtildesq compute.qtildesq.contdisp
#' @keywords internal
NULL

#' The Hadron Package
#' 
#' An R implementation of fitting routines used in lattice QCD. It provides
#' useful functions for extraction hadronic quantities and such like.
#'
#' Toolkit to perform statistical analyses of correlation
#' functions generated from Lattice Monte Carlo simulations. In
#' particular, a class `cf` for correlation functions and
#' methods to analyse those are defined. This includes (blocked)
#' bootstrap and jackknife, but also an automatic determination of
#' integrated autocorrelation times. `hadron` also provides a very
#' general function `bootstrap.nlsfit` to bootstrap a non-linear least squares fit.
#' More specific functions are provided to extract hadronic quantities
#' from Lattice Quantum Chromodynamics simulations, a particular Monte
#' Carlo simulation,(see e.g. European Twisted Mass Collaboration, P. Boucaud et
#' al. (2008) \doi{10.1016/j.cpc.2008.06.013}). Here, to determine
#' energy eigenvalues of hadronic states, specific fitting routines
#' and in particular the generalised eigenvalue method (see
#' e.g. B. Blossier et al. (2009) \doi{10.1088/1126-6708/2009/04/094}
#' and M. Fischer et al. (2020)
#' <https://inspirehep.net/literature/1792113>) are implemented.
#' In addition, input/output and plotting routines are available.
#' 
#' @name hadron
#' @docType package
#' @author Carsten Urbach, \email{urbach@hiskp.uni-bonn.de}
#'
#' @importFrom grDevices col2rgb dev.cur dev.new dev.off rainbow rgb
#' @importFrom graphics abline arrows grconvertX grconvertY hist legend
#' lines par plot points polygon rect segments title
#' @importFrom stats acf approx complete.cases cor cov
#'                na.omit optim pchisq pgamma qchisq qqplot
#'                quantile rchisq rgeom rnorm sd spec.pgram
#'                spline uniroot var weighted.mean median density
#' @importFrom utils combn count.fields head read.table
#'                setTxtProgressBar txtProgressBar tail
#' @keywords package
NULL

#' jackknife-after-bootstrap analysis
#' 
#' jackknife-after-bootstrap (JAB) analysis for errors of errors of correlation
#' functions of class \code{cf}.
#' 
#' We apply the jackknife-after-bootstrap method as proposed by Efron (1992)
#' for iid data and extended by Lahiri (2002) for dependent data. Blocks of
#' bootstrap samples are deleted for a jackknife analysis. The jackknife
#' replicates are computed from the bootstrap samples in which the
#' corresponding block of blocks is missing.
#' 
#' We use here the moving blocked bootstrap (MBB) which uses overlapping
#' blocks. The estimate of standard error of the bootstrap error is computed
#' using formula (2.3) from Lahiri, 2002: \deqn{\mathrm{var}_\mathrm{jab} =
#' (m(N-m)^{-1})M^{-1} \sum_{i=1}^M (\tilde t_n^{(i)}-\hat t_n)^2}{% var_jap =
#' (m(N-m)^-1)M^-1 \sum_{i=1}^M (ttilde_n^(i)-that_n)^2} with \deqn{\tilde
#' t_n^{(i)} = m^{-1}(N\hat t_n - (N-m)\hat t_n^{(i)}).}{% ttilde_n^(i) = m^-1
#' (N that_n - (N-m) that_n^(i)).} Here, \eqn{\hat t_n}{that_n} is the MBB
#' estimate (in our case of standard deviation) and \eqn{\hat
#' t_n^{(i)}}{that_n^(i)} is the i-th jackknife replication of it.
#' 
#' @aliases jackknifeafterboot jab.cf jab.cf.derived jab.effectivemass
#' jab.effectivemassfit jab.matrixfit
#' @param cf An object of class \code{cf} generated by
#' \code{\link{bootstrap.cf}} with \code{sim="fixed"}.
#' @param m \code{m} denotes the number of (overlapping) blocks to delete for
#' the JAB analysis.
#' @return Returns an object of class \code{cf} with an element
#' \code{jack.boot.se}, which is the JAB estimate of standard error of the
#' standard error.
#' @author Carsten Urbach \email{curbach@@gmx.de}
#' @seealso \code{\link{bootstrap.cf}}, \code{\link{cf}},
#' \code{\link{jackknife.cf}}
#' @references S.N. Lahiri, "On the jackknife-after-bootstrap method for
#' dependent data and its consistency properties", Econometric Theory, 2002,
#' Vol. 18, 79-98
#' @keywords bootstrap jackknife boostrap-after-jackknife error-of-error
NULL

#' Sample plaquette time series
#' 
#' A time series of so-called plaquette values generated by a Markov Chain MC
#' process using the Hybrid Monte-Carlo algorithm. Plaquettes are the smallest
#' possible closed loops which can be build in lattice QCD in discretised
#' Euclidean space time.
#' 
#' 
#' @name plaq.sample
#' @docType data
#' @format The format is: num 0.583 0.582 0.582 0.582 0.582 ...
#' @keywords datasets
#' @examples
#' 
#' data(plaq.sample)
#' plot(x=c(1:length(plaq.sample)), y=plaq.sample, type="l", xlab="t", ylab="<P>")
#' 
NULL

#' Sample pseudoscalar correlator
#' 
#' Sample data for a pseudoscalar correlator for time extent Time=48.
#' 
#' 
#' @name pscor.sample
#' @docType data
#' @format list of 2 elements: "t" "ps"
#' @keywords datasets
#' @examples
#' 
#' data("pscor.sample")
#' 
NULL

#' Sample cf data
#' 
#' Sample data for a correlation function for a 24 cube times 48 lattice QCD
#' simulation representing a pion propagation. It is stored in form of an
#' object of class \code{cf}, which is derived from \code{list}.
#' 
#' 
#' @name samplecf
#' @docType data
#' @format The format is: List of 15 $ cf : num  521 533 532 531
#' 561 ...  $ icf : num  521 533 532 531 561 ...  $ Time : num 48
#' $ nrStypes : num 1 $ nrObs : num 1 $ boot.samples : logi TRUE $
#' jackknife.samples: logi FALSE $ symmetrised : logi TRUE $ boot.R : num 1500
#' $ boot.l : num 2 $ seed : num 1442556 $ sim : chr "geom" $ cf0 : num 
#' 519 375 274 221 185 ...  $ cf.tsboot :List of 11 ..$ t0 : num  519 375
#' 274 221 185 ...  ..$ t : num  521 518 520 519 519 ...  ..$ R :
#' num 1500 ..$ data : num  521 533 532 531 561 ...  ..$ seed :
#' int  403 624 -867935848 1692432057 -1535150298 -1438296209 912697060
#' 1838233749 1438572626 999279531 ...  ..$ statistic:function (x) ..$ sim :
#' chr "geom" ..$ n.sim : int 1018 ..$ call : language tsboot(tseries = cf$cf,
#' statistic = function(x) return(apply(x, MARGIN = 2L, FUN = mean)) ...)  ..$
#' l : num 2 ..$ endcorr : logi TRUE ..- attr(*, "class")= chr "boot" ..-
#' attr(*, "boot_type")= chr "tsboot" $ tsboot.se : num  1.001 0.615
#' 0.572 0.537 0.499 ...  - attr(*, "class")= chr "cf" "list"
#' @keywords datasets
#' @examples
#' 
#' data(samplecf)
#' bootstrapped <- bootstrap.cf(samplecf)
#' plot(bootstrapped)
#' 
NULL

#' A three pion correlator with significant thermal states.
"cA2.09.48_3pi_I3_0_A1u_1_pc"
