% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_data.R
\name{fmt_fraction}
\alias{fmt_fraction}
\title{Format values as mixed fractions}
\usage{
fmt_fraction(
  data,
  columns = everything(),
  rows = everything(),
  accuracy = NULL,
  simplify = TRUE,
  layout = c("inline", "diagonal"),
  use_seps = TRUE,
  pattern = "{x}",
  sep_mark = ",",
  system = c("intl", "ind"),
  locale = NULL
)
}
\arguments{
\item{data}{A table object that is created using the \code{\link[=gt]{gt()}} function.}

\item{columns}{The columns to format. Can either be a series of column names
provided in \code{\link[=c]{c()}}, a vector of column indices, or a helper function
focused on selections. The select helper functions are: \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=one_of]{one_of()}}, \code{\link[=num_range]{num_range()}}, and
\code{\link[=everything]{everything()}}.}

\item{rows}{Optional rows to format. Providing \code{\link[=everything]{everything()}} (the
default) results in all rows in \code{columns} being formatted. Alternatively,
we can supply a vector of row captions within \code{\link[=c]{c()}}, a vector of row
indices, or a helper function focused on selections. The select helper
functions are: \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=one_of]{one_of()}}, \code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}. We can also use expressions
to filter down to the rows we need (e.g.,
\verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{accuracy}{The type of fractions to generate. This can either be one of
the keywords \code{"low"}, \code{"med"}, or \code{"high"} (to generate fractions with
denominators of up to 1, 2, or 3 digits, respectively) or an integer value
greater than zero to obtain fractions with a fixed denominator (\code{2} yields
halves, \code{3} is for thirds, \code{4} is quarters, etc.). For the latter option,
using \code{simplify = TRUE} will simplify fractions where possible (e.g., \code{2/4}
will be simplified as \code{1/2}). By default, the \code{"low"} option is used.}

\item{simplify}{If choosing to provide a numeric value for \code{accuracy}, the
option to simplify the fraction (where possible) can be taken with \code{TRUE}
(the default). With \code{FALSE}, denominators in fractions will be fixed to the
value provided in \code{accuracy}.}

\item{layout}{For HTML output, the \code{"inline"} layout is the default. This
layout places the numerals of the fraction on the baseline and uses a
standard slash character. The \code{"diagonal"} layout will generate fractions
that are typeset with raised/lowered numerals and a virgule.}

\item{use_seps}{An option to use digit group separators. The type of digit
group separator is set by \code{sep_mark} and overridden if a locale ID is
provided to \code{locale}. This setting is \code{TRUE} by default.}

\item{pattern}{A formatting pattern that allows for decoration of the
formatted value. The value itself is represented by \code{{x}} and all other
characters are taken to be string literals.}

\item{sep_mark}{The mark to use as a separator between groups of digits
(e.g., using \code{sep_mark = ","} with \code{1000} would result in a formatted value
of \verb{1,000}).}

\item{system}{The numbering system to use. By default, this is the
international numbering system (\code{"intl"}) whereby grouping separators
(i.e., \code{sep_mark}) are separated by three digits. The alternative system,
the Indian numbering system (\code{"ind"}) uses grouping separators that
correspond to thousand, lakh, crore, and higher quantities.}

\item{locale}{An optional locale identifier that can be used for formatting
the value according the locale's rules. Examples include \code{"en"} for English
(United States) and \code{"fr"} for French (France). The use of a locale ID will
override any locale-specific values provided. We can use the
\code{\link[=info_locales]{info_locales()}} function as a useful reference for all of the locales that
are supported.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
With numeric values in a \strong{gt} table, we can perform mixed-fraction-based
formatting. There are several options for setting the accuracy of the
fractions. Furthermore, there is an option for choosing a layout (i.e.,
typesetting style) for the mixed-fraction output.

The following options are available for controlling this type of formatting:
\itemize{
\item accuracy: how to express the fractional part of the mixed fractions; there
are three keyword options for this and an allowance for arbitrary denominator
settings
\item simplification: an option to simplify fractions whenever possible
\item layout: We can choose to output values with diagonal or inline fractions
\item digit grouping separators: options to enable/disable digit separators
and provide a choice of separator symbol for the whole number portion
\item pattern: option to use a text pattern for decoration of the formatted
mixed fractions
\item locale-based formatting: providing a locale ID will result in number
formatting specific to the chosen locale
}
}
\section{Compatibility of formatting function with data values}{


The \code{fmt_fraction()} formatting function is compatible with body cells that
are of the \code{"numeric"} or \code{"integer"} types. Any other types of body cells
are ignored during formatting. This is to say that cells of incompatible data
types may be targeted, but there will be no attempt to format them.
}

\section{Targeting cells with \code{columns} and \code{rows}}{


Targeting of values is done through \code{columns} and additionally by \code{rows} (if
nothing is provided for \code{rows} then entire columns are selected). The
\code{columns} argument allows us to target a subset of cells contained in the
resolved columns. We say resolved because aside from declaring column names
in \code{c()} (with bare column names or names in quotes) we can use
\strong{tidyselect}-style expressions. This can be as basic as supplying a select
helper like \code{starts_with()}, or, providing a more complex incantation like

\code{where(~ is.numeric(.x) && max(.x, na.rm = TRUE) > 1E6)}

which targets numeric columns that have a maximum value greater than
1,000,000 (excluding any \code{NA}s from consideration).

By default all columns and rows are selected (with the \code{everything()}
defaults). Cell values that are incompatible with a given formatting function
will be skipped over, like \code{character} values and numeric \verb{fmt_*()}
functions. So it's safe to select all columns with a particular formatting
function (only those values that can be formatted will be formatted), but,
you may not want that. One strategy is to format the bulk of cell values with
one formatting function and then constrain the columns for later passes with
other types of formatting (the last formatting done to a cell is what you get
in the final output).

Once the columns are targeted, we may also target the \code{rows} within those
columns. This can be done in a variety of ways. If a stub is present, then we
potentially have row identifiers. Those can be used much like column names in
the \code{columns}-targeting scenario. We can use simpler \strong{tidyselect}-style
expressions (the select helpers should work well here) and we can use quoted
row identifiers in \code{c()}. It's also possible to use row indices (e.g.,
\code{c(3, 5, 6)}) though these index values must correspond to the row numbers of
the input data (the indices won't necessarily match those of rearranged rows
if row groups are present). One more type of expression is possible, an
expression that takes column values (can involve any of the available columns
in the table) and returns a logical vector. This is nice if you want to base
formatting on values in the column or another column, or, you'd like to use a
more complex predicate expression.
}

\section{Adapting output to a specific \code{locale}}{


This formatting function can adapt outputs according to a provided \code{locale}
value. Examples include \code{"en"} for English (United States) and \code{"fr"} for
French (France). The use of a valid locale ID here means separator and
decimal marks will be correct for the given locale. Should any value be
provided in \code{sep_mark}, it will be overridden by the locale's preferred
values.

Note that a \code{locale} value provided here will override any global locale
setting performed in \code{\link[=gt]{gt()}}'s own \code{locale} argument (it is settable there as
a value received by all other functions that have a \code{locale} argument). As a
useful reference on which locales are supported, we can use the
\code{\link[=info_locales]{info_locales()}} function to view an info table.
}

\section{Examples}{


Use \code{\link{pizzaplace}} to create a \strong{gt} table. Format the \code{f_sold} and
\code{f_income} columns to display fractions.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pizzaplace |>
  dplyr::group_by(type, size) |>
  dplyr::summarize(
    sold = dplyr::n(),
    income = sum(price),
    .groups = "drop_last"
  ) |>
  dplyr::group_by(type) |>
  dplyr::mutate(
    f_sold = sold / sum(sold),
    f_income = income / sum(income),
  ) |>
  dplyr::arrange(type, dplyr::desc(income)) |>
  gt(rowname_col = "size") |>
  tab_header(
    title = "Pizzas Sold in 2015",
    subtitle = "Fraction of Sell Count and Revenue by Size per Type"
  ) |>
  fmt_integer(columns = sold) |>
  fmt_currency(columns = income) |>
  fmt_fraction(
    columns = starts_with("f_"),
    accuracy = 10,
    simplify = FALSE,
    layout = "diagonal"
  ) |>
  sub_missing(missing_text = "") |>
  tab_spanner(
    label = "Sold",
    columns = contains("sold")
  ) |>
  tab_spanner(
    label = "Revenue",
    columns = contains("income")
  ) |>
  text_transform(
    locations = cells_body(),
    fn = function(x) \{
      dplyr::case_when(
        x == 0 ~ "<em>nil</em>",
        x != 0 ~ x
      )
    \}
  ) |>
  cols_label(
    sold = "Amount",
    income = "Amount",
    f_sold = md("_f_"),
    f_income = md("_f_")
  ) |>
  cols_align(align = "center", columns = starts_with("f")) |>
  tab_options(
    table.width = px(400),
    row_group.as_column = TRUE
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_fraction_1.png" alt="This image of a table was generated from the first code example in the `fmt_fraction()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-7
}

\section{Function Introduced}{

\code{v0.4.0} (February 15, 2022)
}

\seealso{
The vector-formatting version of this function:
\code{\link[=vec_fmt_fraction]{vec_fmt_fraction()}}.

Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bins}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_date}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_spelled_num}()},
\code{\link{fmt_time}()},
\code{\link{fmt_url}()},
\code{\link{fmt}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
