% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_data.R
\name{fmt_bins}
\alias{fmt_bins}
\title{Format column data containing bin/interval information}
\usage{
fmt_bins(
  data,
  columns = everything(),
  rows = everything(),
  sep = "--",
  fmt = NULL
)
}
\arguments{
\item{data}{A table object that is created using the \code{\link[=gt]{gt()}} function.}

\item{columns}{The columns to format. Can either be a series of column names
provided in \code{\link[=c]{c()}}, a vector of column indices, or a helper function
focused on selections. The select helper functions are: \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=one_of]{one_of()}}, \code{\link[=num_range]{num_range()}}, and
\code{\link[=everything]{everything()}}.}

\item{rows}{Optional rows to format. Providing \code{\link[=everything]{everything()}} (the
default) results in all rows in \code{columns} being formatted. Alternatively,
we can supply a vector of row captions within \code{\link[=c]{c()}}, a vector of row
indices, or a helper function focused on selections. The select helper
functions are: \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=one_of]{one_of()}}, \code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}. We can also use expressions
to filter down to the rows we need (e.g.,
\verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{sep}{The separator text that indicates the values are ranged. The
default value of \code{"--"} indicates that an en dash will be used for the
range separator. Using \code{"---"} will be taken to mean that an em dash should
be used. Should you want these special symbols to be taken literally, they
can be supplied within the base \code{\link[=I]{I()}} function.}

\item{fmt}{Formatting expressions in formula form. The RHS of \code{~} should
contain a formatting call (e.g.,
\verb{~ fmt_number(., decimals = 3, use_seps = FALSE}).}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
When using the \code{cut()} function (or other functions that use it in some way)
you get bins that can look like this: \code{"(0,10]"}, \code{"(10,15]"}, \code{"(15,20]"},
\code{"(20,40]"}. This interval notation expresses the lower and upper limits of
each range. The square or round brackets define whether each of the endpoints
are included in the range (\code{[}/\verb{]} for inclusion, \code{(}/\verb{)} for exclusion).
Should bins of this sort be present in a table, the \code{fmt_bins()} function can
be used to format that syntax to a form that presents better in a display
table. It's possible to format the values of the intervals with the \code{fmt}
argument, and, the separator can be modified with the \code{sep} argument.
}
\section{Compatibility of formatting function with data values}{


The \code{fmt_bins()} formatting function is compatible with body cells that are
of the \code{"character"} or \code{"factor"} types. Any other types of body cells are
ignored during formatting. This is to say that cells of incompatible data
types may be targeted, but there will be no attempt to format them.
}

\section{Targeting cells with \code{columns} and \code{rows}}{


Targeting of values is done through \code{columns} and additionally by \code{rows} (if
nothing is provided for \code{rows} then entire columns are selected). The
\code{columns} argument allows us to target a subset of cells contained in the
resolved columns. We say resolved because aside from declaring column names
in \code{c()} (with bare column names or names in quotes) we can use
\strong{tidyselect}-style expressions. This can be as basic as supplying a select
helper like \code{starts_with()}, or, providing a more complex incantation like

\code{where(~ is.numeric(.x) && max(.x, na.rm = TRUE) > 1E6)}

which targets numeric columns that have a maximum value greater than
1,000,000 (excluding any \code{NA}s from consideration).

By default all columns and rows are selected (with the \code{everything()}
defaults). Cell values that are incompatible with a given formatting function
will be skipped over, like \code{character} values and numeric \verb{fmt_*()}
functions. So it's safe to select all columns with a particular formatting
function (only those values that can be formatted will be formatted), but,
you may not want that. One strategy is to format the bulk of cell values with
one formatting function and then constrain the columns for later passes with
other types of formatting (the last formatting done to a cell is what you get
in the final output).

Once the columns are targeted, we may also target the \code{rows} within those
columns. This can be done in a variety of ways. If a stub is present, then we
potentially have row identifiers. Those can be used much like column names in
the \code{columns}-targeting scenario. We can use simpler \strong{tidyselect}-style
expressions (the select helpers should work well here) and we can use quoted
row identifiers in \code{c()}. It's also possible to use row indices (e.g.,
\code{c(3, 5, 6)}) though these index values must correspond to the row numbers of
the input data (the indices won't necessarily match those of rearranged rows
if row groups are present). One more type of expression is possible, an
expression that takes column values (can involve any of the available columns
in the table) and returns a logical vector. This is nice if you want to base
formatting on values in the column or another column, or, you'd like to use a
more complex predicate expression.
}

\section{Formatting expressions for \code{fmt}}{


We can supply a one-sided (RHS only) expression to \code{fmt}, and, several can be
provided in a list. The expression uses a formatting function (e.g.,
\code{\link[=fmt_number]{fmt_number()}}, \code{\link[=fmt_currency]{fmt_currency()}}, etc.) and it must contain an initial \code{.}
that stands for the data object. If performing numeric formatting it might
look something like this:

\code{fmt = ~ fmt_number(., decimals = 1, use_seps = FALSE)}
}

\section{Examples}{


Use the \code{\link{countrypops}} dataset to create a \strong{gt} table. Before even getting
to the \code{\link[=gt]{gt()}} call, we use the \code{cut()} function in conjunction with the
\code{\link[scales:breaks_log]{scales::breaks_log()}} function to create some highly customized bins.
Consequently each country's population in the 2021 year is assigned to a bin.
These bins have a characteristic type of formatting that can be used as input
to \code{fmt_bins()}, and using that formatting function allows us to customize
the presentation of those ranges. For instance, here we are formatting the
left and right values of the ranges with the \code{\link[=fmt_integer]{fmt_integer()}} function (using
formula syntax).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{countrypops |>
  dplyr::filter(year == 2021) |>
  dplyr::select(country_code_2, population) |>
  dplyr::mutate(population_class = cut(
    population,
    breaks = scales::breaks_log(n = 20)(population)
    )
  ) |>
  dplyr::group_by(population_class) |>
  dplyr::summarize(
    count = dplyr::n(),
    countries = paste0(country_code_2, collapse = ",")
  ) |>
  dplyr::arrange(desc(population_class)) |>
  gt() |>
  fmt_flag(columns = countries) |>
  fmt_bins(
    columns = population_class,
    fmt = ~ fmt_integer(., suffixing = TRUE)
  ) |>
  cols_label(
    population_class = "Population Range",
    count = "",
    countries = "Countries"
  ) |>
  cols_width(
    population_class ~ px(150),
    count ~ px(50)
  ) |>
  tab_style(
    style = cell_text(style = "italic"),
    locations = cells_body(columns = count)
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_bins_1.png" alt="This image of a table was generated from the first code example in the `fmt_bins()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-17
}

\section{Function Introduced}{

\emph{In Development}
}

\seealso{
Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_date}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_fraction}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_spelled_num}()},
\code{\link{fmt_time}()},
\code{\link{fmt_url}()},
\code{\link{fmt}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
