% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_vec.R
\name{vec_fmt_currency}
\alias{vec_fmt_currency}
\title{Format a vector as currency values}
\usage{
vec_fmt_currency(
  x,
  currency = "USD",
  use_subunits = TRUE,
  decimals = NULL,
  drop_trailing_dec_mark = TRUE,
  use_seps = TRUE,
  accounting = FALSE,
  scale_by = 1,
  suffixing = FALSE,
  pattern = "{x}",
  sep_mark = ",",
  dec_mark = ".",
  force_sign = FALSE,
  placement = "left",
  incl_space = FALSE,
  locale = NULL,
  output = c("auto", "plain", "html", "latex", "rtf", "word")
)
}
\arguments{
\item{x}{A numeric vector.}

\item{currency}{The currency to use for the numeric value. This input can be
supplied as a 3-letter currency code (e.g., \code{"USD"} for U.S. Dollars,
\code{"EUR"} for the Euro currency). Use \code{\link[=info_currencies]{info_currencies()}} to get an
information table with all of the valid currency codes and examples of
each. Alternatively, we can provide a common currency name (e.g.,
\code{"dollar"}, \code{"pound"}, \code{"yen"}, etc.) to simplify the process. Use
\code{\link[=info_currencies]{info_currencies()}} with the \code{type == "symbol"} option to view an
information table with all of the supported currency symbol names along
with examples.

We can also use the \code{\link[=currency]{currency()}} helper function to specify a custom
currency, where the string could vary across output contexts. For example,
using \code{currency(html = "&fnof;", default = "f")} would give us a suitable
glyph for the Dutch guilder in an HTML output table, and it would simply be
the letter "f" in all other output contexts). Please note that \code{decimals}
will default to \code{2} when using the \code{\link[=currency]{currency()}} helper function.

If nothing is provided to \code{currency} then \code{"USD"} (U.S. dollars) will be
used.}

\item{use_subunits}{An option for whether the subunits portion of a currency
value should be displayed. By default, this is \code{TRUE}.}

\item{decimals}{An option to specify the exact number of decimal places to
use. The default number of decimal places is \code{2}.}

\item{drop_trailing_dec_mark}{A logical value that determines whether decimal
marks should always appear even if there are no decimal digits to display
after formatting (e.g, \code{23} becomes \code{23.}). The default for this is \code{TRUE},
which means that trailing decimal marks are not shown.}

\item{use_seps}{An option to use digit group separators. The type of digit
group separator is set by \code{sep_mark} and overridden if a locale ID is
provided to \code{locale}. This setting is \code{TRUE} by default.}

\item{accounting}{An option to use accounting style for values. With \code{FALSE}
(the default), negative values will be shown with a minus sign. Using
\code{accounting = TRUE} will put negative values in parentheses.}

\item{scale_by}{A value to scale the input. The default is \code{1.0}. All numeric
values will be multiplied by this value first before undergoing formatting.
This value will be ignored if using any of the \code{suffixing} options (i.e.,
where \code{suffixing} is not set to \code{FALSE}).}

\item{suffixing}{An option to scale and apply suffixes to larger numbers
(e.g., \code{1924000} can be transformed to \verb{1.92M}). This option can accept a
logical value, where \code{FALSE} (the default) will not perform this
transformation and \code{TRUE} will apply thousands (\code{K}), millions (\code{M}),
billions (\code{B}), and trillions (\code{T}) suffixes after automatic value scaling.
We can also specify which symbols to use for each of the value ranges by
using a character vector of the preferred symbols to replace the defaults
(e.g., \code{c("k", "Ml", "Bn", "Tr")}).

Including \code{NA} values in the vector will ensure that the particular range
will either not be included in the transformation (e.g, \code{c(NA, "M", "B", "T")} won't modify numbers in the thousands range) or the range will
inherit a previous suffix (e.g., with \code{c("K", "M", NA, "T")}, all numbers
in the range of millions and billions will be in terms of millions).

Any use of \code{suffixing} (where it is not set expressly as \code{FALSE}) means
that any value provided to \code{scale_by} will be ignored.}

\item{pattern}{A formatting pattern that allows for decoration of the
formatted value. The value itself is represented by \code{{x}} and all other
characters are taken to be string literals.}

\item{sep_mark}{The mark to use as a separator between groups of digits
(e.g., using \code{sep_mark = ","} with \code{1000} would result in a formatted value
of \verb{1,000}).}

\item{dec_mark}{The character to use as a decimal mark (e.g., using
\code{dec_mark = ","} with \code{0.152} would result in a formatted value of
\verb{0,152}).}

\item{force_sign}{Should the positive sign be shown for positive values
(effectively showing a sign for all values except zero)? If so, use \code{TRUE}
for this option. The default is \code{FALSE}, where only negative numbers will
display a minus sign. This option is disregarded when using accounting
notation with \code{accounting = TRUE}.}

\item{placement}{The placement of the currency symbol. This can be either be
\code{left} (the default) or \code{right}.}

\item{incl_space}{An option for whether to include a space between the value
and the currency symbol. The default is to not introduce a space character.}

\item{locale}{An optional locale ID that can be used for formatting the value
according the locale's rules. Examples include \code{"en_US"} for English
(United States) and \code{"fr_FR"} for French (France). The use of a valid
locale ID will override any values provided in \code{sep_mark} and \code{dec_mark}.
We can use the \code{\link[=info_locales]{info_locales()}} function as a useful reference for all of
the locales that are supported.}

\item{output}{The output style of the resulting character vector. This can
either be \code{"auto"} (the default), \code{"plain"}, \code{"html"}, \code{"latex"}, \code{"rtf"},
or \code{"word"}. In \strong{knitr} rendering (i.e., Quarto or R Markdown), the
\code{"auto"} option will choose the correct \code{output} value}
}
\value{
A character vector.
}
\description{
With numeric values in a vector, we can perform currency-based
formatting. This function supports both automatic formatting with a
three-letter or numeric currency code. We can also specify a custom currency
that is formatted according to the output context with the \code{\link[=currency]{currency()}}
helper function. Numeric formatting facilitated through the use of a locale
ID. We have fine control over the conversion from numeric values to currency
values, where we could take advantage of the following options:
\itemize{
\item the currency: providing a currency code or common currency name will
procure the correct currency symbol and number of currency subunits; we could
also use the \code{\link[=currency]{currency()}} helper function to specify a custom currency
\item currency symbol placement: the currency symbol can be placed before
or after the values
\item decimals/subunits: choice of the number of decimal places, and a
choice of the decimal symbol, and an option on whether to include or exclude
the currency subunits (decimal portion)
\item negative values: choice of a negative sign or parentheses for values
less than zero
\item digit grouping separators: options to enable/disable digit separators
and provide a choice of separator symbol
\item scaling: we can choose to scale targeted values by a multiplier value
\item large-number suffixing: larger figures (thousands, millions, etc.) can
be autoscaled and decorated with the appropriate suffixes
\item pattern: option to use a text pattern for decoration of the formatted
currency values
\item locale-based formatting: providing a locale ID will result in
currency formatting specific to the chosen locale
}

We can use the \code{\link[=info_currencies]{info_currencies()}} function for a useful reference on all of
the possible inputs to the \code{currency} argument.
}
\section{Examples}{


Let's create a numeric vector for the next few examples:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{num_vals <- c(5.2, 8.65, 0, -5.3, NA)
}\if{html}{\out{</div>}}

Using \code{vec_fmt_currency()} with the default options will create a character
vector where the numeric values have been transformed to U.S. Dollars
(\code{"USD"}). Furthermore, the rendering context will be autodetected unless
specified in the \code{output} argument (here, it is of the \code{"plain"} output
type).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_currency(num_vals)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "$5.20" "$8.65" "$0.00" "-$5.30" "NA"
}\if{html}{\out{</div>}}

We can supply a currency code to the \code{currency} argument. Let's use British
Pounds by using \code{currency = "GBP"}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_currency(num_vals, currency = "GBP")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "GBP5.20" "GBP8.65" "GBP0.00" "-GBP5.30" "NA"
}\if{html}{\out{</div>}}

If we are formatting for a different locale, we could supply the locale ID
and let \strong{gt} handle qll locale-specific formatting options:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_currency(num_vals, currency = "EUR", locale = "fr")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "EUR5,20" "EUR8,65" "EUR0,00" "-EUR5,30" "NA"
}\if{html}{\out{</div>}}

There are many options for formatting values. Perhaps you need to have
explicit positive and negative signs? Use \code{force_sign = TRUE} for that.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_currency(num_vals, force_sign = TRUE)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "+$5.20" "+$8.65" "$0.00" "-$5.30" "NA"
}\if{html}{\out{</div>}}

As a last example, one can wrap the values in a pattern with the \code{pattern}
argument. Note here that \code{NA} values won't have the pattern applied.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_currency(num_vals, pattern = "`\{x\}`")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "`$5.20`" "`$8.65`" "`$0.00`" "`-$5.30`" "NA"
}\if{html}{\out{</div>}}
}

\section{Function ID}{

14-8
}

\seealso{
Other vector formatting functions: 
\code{\link{vec_fmt_bytes}()},
\code{\link{vec_fmt_datetime}()},
\code{\link{vec_fmt_date}()},
\code{\link{vec_fmt_duration}()},
\code{\link{vec_fmt_engineering}()},
\code{\link{vec_fmt_fraction}()},
\code{\link{vec_fmt_integer}()},
\code{\link{vec_fmt_markdown}()},
\code{\link{vec_fmt_number}()},
\code{\link{vec_fmt_partsper}()},
\code{\link{vec_fmt_percent}()},
\code{\link{vec_fmt_scientific}()},
\code{\link{vec_fmt_time}()}
}
\concept{vector formatting functions}
