% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/helpers.R
\name{unit_conversion}
\alias{unit_conversion}
\title{Get a conversion factor across two measurement units of a given class}
\usage{
unit_conversion(from, to)
}
\arguments{
\item{from}{\emph{Units for the input value}

\verb{scalar<character>} // \strong{required}

The keyword representing the units for the value that requires unit
conversion. In the case where the value has units of miles, the necessary
input is \code{"length.mile"}.}

\item{to}{\emph{Desired units for the value}

\verb{scalar<character>} // \strong{required}

The keyword representing the target units for the value with units defined
in \code{from}. In the case where input value has units of miles and we would
rather want the value to be expressed as kilometers, the \code{to} value should
be \code{"length.kilometer"}.}
}
\value{
A single numerical value.
}
\description{
The \code{unit_conversion()} helper function gives us a conversion factor for
transforming a value from one form of measurement units to a target form.
For example if you have a length value that is expressed in miles you could
transform that value to one in kilometers through multiplication of the value
by the conversion factor (in this case \code{1.60934}).

For \code{unit_conversion()} to understand the source and destination units, you
need to provide a keyword value for the \code{from} and \code{to} arguments. To aid as
a reference for this, call \code{\link[=info_unit_conversions]{info_unit_conversions()}} to display an
information table that contains all of the keywords for every conversion
type.
}
\section{Examples}{


Let's use a portion of the \code{\link{towny}} dataset and create a table showing
population, density, and land area for 10 municipalities. The \code{land_area_km2}
values are in units of square kilometers, however, we'd rather the values
were in square miles. We can convert the numeric values while formatting the
values with \code{\link[=fmt_number]{fmt_number()}} by using \code{unit_conversion()} in the \code{scale_by}
argument since the return value of that is a conversion factor (which is
applied to each value by multiplication). The same is done for converting the
'people per square kilometer' values in \code{density_2021} to 'people per square
mile', however, the units to convert are in the denominator so the inverse
of the conversion factor must be used.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{towny |>
  dplyr::arrange(desc(density_2021)) |>
  dplyr::slice_head(n = 10) |>
  dplyr::select(name, population_2021, density_2021, land_area_km2) |>
  gt(rowname_col = "name") |>
  fmt_integer(columns = population_2021) |>
  fmt_number(
    columns = land_area_km2,
    decimals = 1,
    scale_by = unit_conversion(
      from = "area.square-kilometer",
      to = "area.square-mile"
    )
  ) |>
  fmt_number(
    columns = density_2021,
    decimals = 1,
    scale_by = 1 / unit_conversion(
      from = "area.square-kilometer",
      to = "area.square-mile"
    )
  ) |>
  cols_label(
    land_area_km2 = "Land Area,<br>sq. mi",
    population_2021 = "Population",
    density_2021 = "Density,<br>ppl / sq. mi",
    .fn = md
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_unit_conversion_1.png" alt="This image of a table was generated from the first code example in the `unit_conversion()` help file." style="width:100\%;">
}}

With a small slice of the \code{\link{gibraltar}} dataset, let's display the
temperature values in terms of degrees Celsius (present in the data) \emph{and} as
temperatures in degrees Fahrenheit (achievable via conversion). We can
duplicate the \code{temp} column through \code{\link[=cols_add]{cols_add()}} (naming the new column as
\code{temp_f}) and when formatting through \code{\link[=fmt_integer]{fmt_integer()}} we can call
\code{unit_conversion()} within the \code{scale_by} argument to perform this
transformation while formatting the values as integers.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{gibraltar |>
  dplyr::filter(
    date == "2023-05-15",
    time >= "06:00",
    time <= "12:00"
  ) |>
  dplyr::select(time, temp) |>
  gt() |>
  tab_header(
    title = "Air Temperature During Late Morning Hours at LXGB Stn.",
    subtitle = "May 15, 2023"
  ) |>
  cols_add(temp_f = temp) |>
  cols_move(columns = temp_f, after = temp) |>
  tab_spanner(
    label = "Temperature",
    columns = starts_with("temp")
  ) |>
  fmt_number(
    columns = temp,
    decimals = 1
  ) |>
  fmt_integer(
    columns = temp_f,
    scale_by = unit_conversion(
      from = "temperature.C",
      to = "temperature.F"
    )
  ) |>
  cols_label(
    time = "Time",
    temp = "\{\{degC\}\}",
    temp_f = "\{\{degF\}\}"
  ) |>
  cols_width(
    starts_with("temp") ~ px(80),
    time ~ px(100)
  ) |>
  opt_horizontal_padding(scale = 3) |>
  opt_vertical_padding(scale = 0.5) |>
  opt_align_table_header(align = "left") |>
  tab_options(heading.title.font.size = px(16))
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_unit_conversion_2.png" alt="This image of a table was generated from the second code example in the `unit_conversion()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

8-7
}

\section{Function Introduced}{

\code{v0.11.0} (July 9, 2024)
}

\seealso{
Other helper functions: 
\code{\link{adjust_luminance}()},
\code{\link{cell_borders}()},
\code{\link{cell_fill}()},
\code{\link{cell_text}()},
\code{\link{currency}()},
\code{\link{default_fonts}()},
\code{\link{escape_latex}()},
\code{\link{from_column}()},
\code{\link{google_font}()},
\code{\link{gt_latex_dependencies}()},
\code{\link{html}()},
\code{\link{md}()},
\code{\link{nanoplot_options}()},
\code{\link{pct}()},
\code{\link{px}()},
\code{\link{random_id}()},
\code{\link{row_group}()},
\code{\link{stub}()},
\code{\link{system_fonts}()}
}
\concept{helper functions}
