library(gsw)
# Test against with values provided on the TEOS-10 website, e.g.
#   http://www.teos-10.org/pubs/gsw/html/gsw_adiabatic_lapse_rate_from_t.html
# for the first function.

## gsw_adiabatic_lapse_rate_from_CT()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
lr <- gsw_adiabatic_lapse_rate_from_CT(SA, CT, p)
stopifnot(all.equal(lr, 1e-7*c(0.240199646230069, 0.238457486976761, 0.203635157319712,
                               0.119829566859790, 0.100052760967308, 0.087773070307283)))

## gsw_alpha()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
a <- gsw_alpha(SA,CT,p)
stopifnot(all.equal(a, 1e-3*c(0.324480399390879, 0.322623026070215, 0.281341259491397,
                              0.173481771204433, 0.146828379793380, 0.130191733440652)))

## gsw_alpha_on_beta()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
p <- c(      10,      50,     125,     250,     600,    1000)
ab <- gsw_alpha_on_beta(SA,CT,p)
stopifnot(all.equal(ab, c(0.452454540612631, 0.449585262707113, 0.387121114716631,
                          0.230686622275719, 0.193629961364881, 0.170827066345350)))

## gsw_alpha_wrt_t_exact
SA = c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t = c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p =  c(     10,      50,     125,     250,     600,    1000)
alpha_wrt_t_exact <- gsw_alpha_wrt_t_exact(SA, t, p)
stopifnot(all.equal(alpha_wrt_t_exact, 1e-3*c(0.325601747227247, 0.323448083851267,
                                              0.281413883319329, 0.172825692975230,
                                              0.145569941503599, 0.128362986933288)))

## gsw_beta()
SA = c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT = c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p =  c(     10,      50,     125,     250,     600,    1000)
beta <- gsw_beta(SA, CT, p)
stopifnot(all.equal(beta, 1e-3*c(0.717580024283098, 0.717712392082254, 0.726221642634641,
                                 0.750510976803418, 0.755001726838233, 0.756941554699334)))
## gsw_beta_const_t_exact
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
b <- gsw_beta_const_t_exact(SA, t, p)
stopifnot(all.equal(b, 1e-3*c(0.731120837010429, 0.731071779078011, 0.736019128913071,
                              0.753810501711847, 0.757259405338257, 0.758649268096996)))

## gsw_C_from_SP()
SP <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
C <- gsw_C_from_SP(SP, t, p)
stopifnot(all.equal(C, c(56.412599581571186, 56.316185602699953, 50.670369333973944,
                         38.134518936104350, 35.056577637635257, 32.986550607990118)))

## gsw_cabbeling()
SA = c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT = c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
p =  c(     10,      50,     125,     250,     600,    1000)
cabbeling <- gsw_cabbeling(SA,CT,p)
stopifnot(all.equal(cabbeling, 1e-4*c(0.086714418920643, 0.086896846102756,
                                      0.092576793980783, 0.108971394389223,
                                      0.112995099355420, 0.115427535647523)))

## gsw_CT_freezing()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
p <- c(     10,      50,     125,     250,     600,    1000)
saturation_fraction <- 1
CT_freezing <- gsw_CT_freezing(SA, p, saturation_fraction)
warning("gsw_CT_freezing() test values are from gsw_CT_freezing_poly()\n")
stopifnot(all.equal(CT_freezing, c(-1.899657519404743, -1.940781208575193,
                                   -2.006256923078589, -2.092413889940425,
                                   -2.359438414203326, -2.677347349358548)))

## gsw_CT_from_pt()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
pt <- c(28.7832, 28.4209, 22.7850, 10.2305,  6.8292,  4.3245)
CT <- gsw_CT_from_pt(SA, pt)
stopifnot(all.equal(CT, c(28.809923015982083, 28.439144260767169,
                          22.786246608464264, 10.226165605435785,
                           6.827183417643142,  4.323565182322069)))

## gsw_cp_t_exact()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
CP <- gsw_cp_t_exact(SA, t, p)
stopifnot(all.equal(CP, c(4002.888003958537, 4000.980283927373, 3995.546468894633,
                          3985.076769021370, 3973.593843482723, 3960.184084786622 ))) 

## gsw_CT_from_t()
CT <- gsw_CT_from_t(34.7118, 28.7856, 10)
stopifnot(all.equal(CT, 28.809919826700281))

## gsw_deltaSA_from_SP()
SP =   c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
p =    c(     10,      50,     125,     250,     600,    1000)
lat =  c(      4,       4,       4,       4,       4,       4)
long = c(    188,     188,     188,     188,     188,     188)
deltaSA = gsw_deltaSA_from_SP(SP,p,long,lat)
stopifnot(all.equal(deltaSA, c(0.000167203365230, 0.000268836122231,
                               0.000665803155705, 0.002706154619403,
                               0.005652977406832,  0.009444734661606)))

## gsw_dynamic_enthalpy()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <-c( 28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
p <- c(      10,      50,     125,     250,     600,    1000)
de <- gsw_dynamic_enthalpy(SA, CT, p)
stopifnot(all.equal(de, 1e3 * c(0.097864649180491, 0.489161224318287,
                                1.220511487957649, 2.433730243703573,
                                5.833879407549211, 9.711445229571982)))

## gsw_enthalpy()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <-c( 28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
p <- c(      10,      50,     125,     250,     600,    1000)
e <- gsw_enthalpy(SA, CT, p)
stopifnot(all.equal(e, 1e5 * c(1.151031813070013, 1.140146924304349, 0.921800131324770,
                               0.432553703468003, 0.330871603243963, 0.269706855289744)))

## gsw_enthalpy_t_exact()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
e <- gsw_enthalpy_t_exact(SA, t, p)
stopifnot(all.equal(e, 1e5 * c(1.151032604783763, 1.140148036012021, 0.921799209310966,
                               0.432553283808897, 0.330872159700175, 0.269705880448018)))

## gsw_entropy_from_t()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
e <- gsw_entropy_from_t(SA, t, p)
stopifnot(all.equal(e, 100 * c(4.003894252787245, 3.954381784340642, 3.198664981986740,
                               1.467908815899072, 0.986473408657975, 0.627915087346090)))

## gsw_grav()
g <- gsw_grav(c(-90, -60, -30, 0), 0)
stopifnot(all.equal(g, c(9.832186205884799, 9.819178859991149,
                         9.793249257048750, 9.780327000000000)))
## gsw_internal_energy()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <-  c(     10,      50,     125,     250,     600,    1000)
e <- gsw_internal_energy(SA, CT, p)
stopifnot(all.equal.numeric(e, 1e5*c(1.148091577452400, 1.134013146037679, 0.909571141862217,
                                     0.408593071164669, 0.273985268520158, 0.175019387936133)))

## gsw_IPV_vs_fNsquared_ratio
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
p <-  c(     10,      50,     125,     250,     600,    1000)
p_ref <- 0
r <- gsw_IPV_vs_fNsquared_ratio(SA, CT, p, p_ref)
stopifnot(all.equal.numeric(r$IPV_vs_fNsquared_ratio,
                            c(0.999745283730840, 0.996950635279959, 0.986153962640181,
                              0.931618955820649, 0.861271753240207)))
stopifnot(all.equal.numeric(r$p_mid, c(30, 87.5, 187.5, 425, 800)))

## gsw_kappa()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <-c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
kappa <- gsw_kappa(SA, CT, p)
stopifnot(all.equal(kappa, 1e-9*c(0.411346577902628, 0.411103645001243, 0.416540984994117,
                                  0.435546004842899, 0.438741805834381, 0.439808252322948)))

## gsw_kappa_t_exact()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t <-c(  28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
kappa <- gsw_kappa_t_exact(SA, t, p)
stopifnot(all.equal(kappa, 1e-9*c(0.411245799180373, 0.411029072229334, 0.416539558054756,
                                  0.435668337689072, 0.438923693006423, 0.440037575765429)))

## gsw_latentheat_evap_CT
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
lh  <- gsw_latentheat_evap_CT(SA, CT)
stopifnot(all.equal(lh, 1e6*c(2.429947107462561, 2.430774073049213, 2.444220372158452,
                              2.474127109232524, 2.482151446148560, 2.488052297193594)))

## gsw_latentheat_evap_t
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
lh = gsw_latentheat_evap_t(SA, t)
stopifnot(all.equal(lh, 1e6*c(2.429882982734836, 2.430730236218543, 2.444217294049004,
                              2.474137411322517, 2.482156276375029, 2.488054617630297)))
## gsw_latentheat_evap_melting
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
p <- c(      10,      50,     125,     250,     600,    1000)
lh = gsw_latentheat_melting(SA, p)
stopifnot(all.equal(lh, 1e5*c(3.299495187300804, 3.298611954422526, 3.297124383647952,
                              3.294972884747496, 3.288480015369891, 3.280715953443947)))

## gsw_Nsquared()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
p <- c(      10,      50,     125,     250,     600,    1000)
latitude <- 4
r <- gsw_Nsquared(SA, CT, p, latitude)
N2 <- r$N2
p_mid <- r$p_mid
stopifnot(all.equal(N2, 1e-3*c(0.060847042791371, 0.235730438897447, 0.216590355253073,
                               0.012935081031687, 0.008430222212653)))
stopifnot(all.equal(p_mid, c(30, 87.5, 187.5, 425, 800)))

## gsw_p_from_z
z <- c(10, 50, 125, 250, 600, 1000)
latitude <- 4
p <- gsw_p_from_z(z, latitude)
pexpected <- -1e3*c(0.010055235976847, 0.050271267635044, 0.125655128514602,
                    0.251233371685054, 0.602442093622451, 1.003079508920876)
print(data.frame(p, pexpected))
stopifnot(all.equal.numeric(p, pexpected))

## gsw_pot_rho_t_exact
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
p_ref  <- 0
prho <- gsw_pot_rho_t_exact(SA,t,p,p_ref)
stopifnot(all.equal(prho, c(1021.798145811089, 1022.052484416980, 1023.893583651958,
                            1026.667621124443, 1027.107230868492, 1027.409631264134)))

## gsw_pt0_from_t
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
pt0 <- gsw_pt0_from_t(SA, t, p)
stopifnot(all.equal(pt0, c(28.783196819670632, 28.420983342398962, 22.784930399117108,
                           10.230523661095731, 6.829230224409661, 4.324510571845719)))

## gsw_pt_from_CT
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
pt <- gsw_pt_from_CT(SA, CT)
stopifnot(all.equal(pt, c(28.783177048624573, 28.420955597191984, 22.784953468087107,
                          10.230534394434429, 6.829216587061605, 4.324534835990236)))

## gsw_pt_from_t
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
p_ref <- 0
pt <- gsw_pt_from_t(SA, t, p, p_ref)
stopifnot(all.equal(pt, c(28.783196819670632, 28.420983342398962, 22.784930399117108,
                          10.230523661095731, 6.829230224409661, 4.324510571845719)))

## gsw_rho()
## FIXME: add other values to gsw_rho() test
rho <- gsw_rho(34.7118, 28.8099, 10)
stopifnot(all.equal(rho, 1021.8404465661))

## gsw_rho_first_derivatives()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
p <- c(      10,      50,     125,     250,     600,    1000)
r <- gsw_rho_first_derivatives(SA, CT, p)
stopifnot(all.equal(r$drho_dSA, c(0.733213518869108, 0.733680726153097, 0.744004504721854,
                                  0.771450239175361, 0.777682888539079, 0.781381240968611)))
stopifnot(all.equal(r$drho_dCT, c(-0.331745785850892, -0.329852042010685, -0.288019853222119,
                                  -0.177963249929159, -0.150582707661951, -0.133481065091957)))
stopifnot(all.equal(r$drho_dp, 1e-6*c(0.420305747137598, 0.420249518076609, 0.426747374547013,
                                      0.447719879751583, 0.451969476434040, 0.454082698501517)))

## gsw_rho_t_exact()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t <-  c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <-  c(     10,      50,     125,     250,     600,    1000)
rho <- gsw_rho_t_exact(SA, t, p)
stopifnot(all.equal(rho, c(1021.840173185531, 1022.262689926782, 1024.427715941676,
                           1027.790201811623, 1029.837714725961, 1032.002404116447)))

## gsw_SA_from_rho()
rho <- c(1021.8482, 1022.2647, 1024.4207, 1027.7841, 1029.8287, 1031.9916)
CT <-c(    28.7856,   28.4329,   22.8103,   10.2600,    6.8863,    4.4036)
p <- c(         10,        50,       125,       250,       600,      1000)
SA <- gsw_SA_from_rho(rho, CT, p)
stopifnot(all.equal(SA, c(34.711382887931144, 34.891005107845949, 35.025429483046373,
                          34.846708947660034, 34.736425587999484, 34.732694768360886)))

## gsw_SA_from_SP()
SP <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
p <- c(      10,      50,     125,     250,     600,    1000)
lat <- c(     4,       4,       4,       4,       4,       4)
long <- c(  188,     188,     188,     188,     188,     188)
SA <- gsw_SA_from_SP(SP, p, long, lat)
stopifnot(all.equal(SA, c(34.711778344814114, 34.891522618230098, 35.025544862476920,
                          34.847229026189588, 34.736628474576051, 34.732363065590846)))

## gsw_SA_from_Sstar()
SP <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
p <- c(      10,      50,     125,     250,     600,    1000)
lat <- c(     4,       4,       4,       4,       4,       4)
long <- c(  188,     188,     188,     188,     188,     188)
SA <- gsw_SA_from_SP(SP, p, long, lat)
stopifnot(all.equal(SA, c(34.711778344814114, 34.891522618230098, 35.025544862476920,
                          34.847229026189588, 34.736628474576051, 34.732363065590846)))

## gsw_SR_from_SP()
SP <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
SR <- gsw_SR_from_SP(SP)
stopifnot(all.equal(SR, c(34.711611927085727, 34.891255045714303, 35.024882197714305,
                          34.844535778285724, 34.731002934857159, 34.722965211428587)))

## gsw_sigma0()
sigma0 <- gsw_sigma0(34.7118, 28.8099)
stopifnot(all.equal(sigma0, 21.798411276610750))

## gsw_sigma1()
sigma1 <- gsw_sigma1(34.7118, 28.8099)
stopifnot(all.equal(sigma1, 25.955891533636986))

## gsw_sigma2()
sigma2 <- gsw_sigma2(34.7118, 28.8099)
stopifnot(all.equal(sigma2, 30.02279641606605))

## gsw_sigma3()
sigma3 <- gsw_sigma3(34.7118, 28.8099)
stopifnot(all.equal(sigma3, 34.002600253012133))

## gsw_sigma4()
sigma4 <- gsw_sigma4(34.7118, 28.8099)
stopifnot(all.equal(sigma4, 37.898467323406976))

## gsw_specvol()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
p <- c(      10,      50,     125,     250,     600,    1000)
specvol <- gsw_specvol(SA, CT, p)
stopifnot(all.equal(specvol, 1e-3 * c(0.978626363206202, 0.978221861206179, 0.976154716658432,
                                      0.972960929615236, 0.971026743358570, 0.968990291993136)))

## gsw_specvol_anom()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
p <- c(      10,      50,     125,     250,     600,    1000)
a <- gsw_specvol_anom(SA, CT, p)
stopifnot(all.equal(a, 1e-5 * c(0.601005694856401, 0.578559312084076, 0.405540722918385,
                                0.142144553233903, 0.104304693754049, 0.076366488048843)))

## gsw_specvol_t_exact()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
v <- gsw_specvol_t_exact(SA, t, p)
stopifnot(all.equal(v, 1e-3 * c(0.978626625025472, 0.978222143734527, 0.976154768597586,
                                0.972961211575438, 0.971026779948624, 0.968989990731808)))

## gsw_SP_from_C()
SP <- gsw_SP_from_C(34.5487, 28.7856, 10)
stopifnot(all.equal(SP, 20.009869599086951))

## gsw_SP_from_SA()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
p <- c(      10,      50,     125,     250,     600,    1000)
latitude <- 4
longitude <- 188
SP <- gsw_SP_from_SA(SA, p, longitude, latitude)
stopifnot(all.equal(SP, c(34.548721553448317, 34.727477488096639, 34.860554877708005,
                          34.680971112271791, 34.567971663653388, 34.560036751118204)))

## gsw_SP_from_SK()
SK <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
SP <- gsw_SP_from_SK(SK)
stopifnot(all.equal(SP, c(34.548342096952908, 34.727295637119113, 34.860409847645435,
                          34.680755706371187, 34.567658670360110, 34.559651800554022)))

## gsw_SP_from_SR()
SR <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
SP <- gsw_SP_from_SR(SR)
stopifnot(all.equal(SP, c(34.386552667080714, 34.564513505458834, 34.696889296869848,
                          34.518231743800094, 34.405762086435850, 34.397799632817147)))

## gsw_SP_from_Sstar()
Sstar <- c(34.7115, 34.8912, 35.0247, 34.8436, 34.7291, 34.7197)
p <- c(         10,      50,     125,     250,     600,    1000)
longitude <- 188
latitude <- 4
SP <- gsw_SP_from_Sstar(Sstar, p, longitude, latitude)
stopifnot(all.equal(SP, c(34.548646570969929, 34.727538423586189, 34.860549501859502,
                          34.681006826476434, 34.568065697992346, 34.560023926979518))) 

## gsw_sound_speed()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
ss <- gsw_sound_speed(SA, CT, p)
stopifnot(all.equal(ss, c(1542.420534932182, 1542.561816671942, 1530.847504375638,
                          1494.624012692069, 1487.691791461714, 1484.329368325987))) 

## gsw_sound_speed_t_exact()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
t <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
p <- c(      10,      50,     125,     250,     600,    1000)
ss <- gsw_sound_speed_t_exact(SA, t, p)
stopifnot(all.equal(ss, c(1542.615803587414, 1542.703534065789, 1530.844979136360,
                          1494.409996920661, 1487.377102518027, 1483.934609078705)))
## gsw_Sstar_from_SA()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
p <- c(      10,      50,     125,     250,     600,    1000)
longitude <- 188
latitude <- 4
Sstar <- gsw_Sstar_from_SA(SA, p, longitude, latitude)
stopifnot(all.equal.numeric(Sstar, c(34.711575335926490, 34.891138777337822, 35.024705401162166,
                                     34.843564118358302, 34.729005527604883, 34.719712883389462)))

## gsw_Sstar_from_SP()
SP <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
p <- c(      10,      50,     125,     250,     600,    1000)
longitude <- 188
latitude <- 4
Sstar <- gsw_Sstar_from_SP(SP, p, longitude, latitude)
stopifnot(all.equal.numeric(Sstar, c(34.711553680880769, 34.891161395333754, 35.024650265047370,
                                     34.843593141519356, 34.729033995955525, 34.719675962471783))) 

## gsw_t_freezing()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
p <- c(      10,      50,     125,     250,     600,    1000)
saturation_fraction <- 1
tf <- gsw_t_freezing(SA, p, saturation_fraction)
warning("gsw_t_freezing() test values are from gsw_t_freezing_poly()\n")
stopifnot(all.equal(tf, c(-1.902704434299200, -1.942897945475226, -2.006877364578649,
                          -2.091041391538033, -2.351431560178946, -2.660685237633709)))

## gsw_t_from_CT()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
p <-  c(     10,      50,     125,     250,     600,    1000)
t <- gsw_t_from_CT(SA, CT, p)
stopifnot(all.equal(t, c(28.785580227725703, 28.432872246163946, 22.810323087627076,
                         10.260010752788906, 6.886286301029376, 4.403624452383043)))

## gsw_thermobarbaric()
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
p <-  c(     10,      50,     125,     250,     600,    1000)
tb <- gsw_thermobaric(SA, CT, p)
stopifnot(all.equal(tb, 1e-11 * c(0.140572143831373, 0.141670674489603, 0.163021073914596,
                                  0.225896125166222, 0.246017471421262, 0.261310958174722)))

## gsw_Turner_Rsubrho
SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
p <-  c(     10,      50,     125,     250,     600,    1000)
r <- gsw_Turner_Rsubrho(SA, CT, p)
stopifnot(all.equal(r$Tu, c(-2.064830032393999, 41.758285540907067, 47.607293931126598,
                            53.714708172363480, 45.527415163693554)))
stopifnot(all.equal(r$Rsubrho, 100*c(-0.009304018848608, -0.176556664605209, 0.219600192807469,
                                     0.065238283043563, 1.086319945640579)))
stopifnot(all.equal(r$p_mid, 100*c(0.300000000000000, 0.875000000000000, 1.875000000000000,
                                   4.250000000000000, 8.000000000000000)))

## gsw_z_from_p()
z <- gsw_z_from_p(10, 4)
stopifnot(all.equal(z, -9.9445831334188))
