% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/inference.R
\name{inference}
\alias{inference}
\alias{stashed_samples}
\alias{mcmc}
\alias{opt}
\title{statistical inference on greta models}
\usage{
stashed_samples()

mcmc(model, method = c("hmc"), n_samples = 1000, thin = 1, warmup = 100,
  verbose = TRUE, pb_update = 10, control = list(),
  initial_values = NULL)

opt(model, method = c("adagrad"), max_iterations = 100, tolerance = 1e-06,
  control = list(), initial_values = NULL)
}
\arguments{
\item{model}{greta_model object}

\item{method}{the method used to sample or optimise values. Currently only
one method is available for each procedure: \code{hmc} and \code{adagrad}}

\item{n_samples}{the number of MCMC samples to draw (after any warm-up, but
before thinning)}

\item{thin}{the MCMC thinning rate; every \code{thin} samples is retained,
the rest are discarded}

\item{warmup}{the number of samples to spend warming up the mcmc sampler.
During this phase the sampler moves toward the highest density area and
tunes sampler hyperparameters.}

\item{verbose}{whether to print progress information to the console}

\item{pb_update}{how regularly to update the progress bar (in iterations)}

\item{control}{an optional named list of hyperparameters and options to
control behaviour of the sampler or optimiser. See Details.}

\item{initial_values}{an optional named vector of initial values for the free
parameters in the model. These will be used as the starting point for
sampling/optimisation}

\item{max_iterations}{the maximum number of iterations before giving up}

\item{tolerance}{the numerical tolerance for the solution, the optimiser stops when the (absolute) difference in the joint density between successive iterations drops below this level}
}
\value{
\code{mcmc} & \code{stashed_samples} - an \code{mcmc.list} object that can be analysed using
  functions from the coda package. This will contain mcmc samples of the
  greta arrays used to create \code{model}.

\code{opt} - a list containing the following named elements:
  \itemize{
    \item{par}{the best set of parameters found}
    \item{value}{the log joint density of the model at the parameters par}
    \item{iterations}{the number of iterations taken by the optimiser}
    \item{convergence}{an integer code, 0 indicates successful completion, 1
    indicates the iteration limit max_iterations had been reached}
  }
}
\description{
Carry out statistical inference on greta models by
  MCMC or likelihood/posterior optimisation.
}
\details{
If the sampler is aborted before finishing, the samples collected so
  far can be retrieved with \code{stashed_samples()}. Only samples from the
  sampling phase will be returned.

For \code{mcmc()} if \code{verbose = TRUE}, the progress bar shows
  the number of iterations so far and the expected time to complete the phase
  of model fitting (warmup or sampling). Updating the progress bar regularly
  slows down sampling, by as much as 9 seconds per 1000 updates. So if you
  want the sampler to run faster, you can change \code{pb_update} to increase
  the number of iterations between updates of the progress bar, or turn the
  progress bar off altogether by setting \code{verbose = FALSE}.

  Occasionally, a proposed set of parameters can cause numerical instability
  (I.e. the log density or its gradient is \code{NA}, \code{Inf} or
  \code{-Inf}); normally because the log joint density is so low that it
  can't be represented as a floating point number. When this happens, the
  progress bar will also display the proportion of samples so far that were
  'bad' (numerically unstable) and therefore rejected.
  If you're getting a lot of numerical instability, you might want to
  manually define starting values to move the sampler into a more reasonable
  part of the parameter space. Alternatively, you could redefine the model
  (via \code{model}) to have double precision, though this will slow down
  sampling.

  Currently, the only implemented MCMC procedure is static Hamiltonian
  Monte Carlo (\code{method = "hmc"}). During the warmup iterations, the
  leapfrog stepsize hyperparameter \code{epsilon} is tuned to maximise the
  sampler efficiency. The \code{control} argument can be used to specify the
  initial value for epsilon, along with two other hyperparameters: \code{Lmin}
  and \code{Lmax}; positive integers (with \code{Lmax > Lmin}) giving the
  upper and lower limits to the number of leapfrog steps per iteration (from
  which the number is selected uniformly at random).

  The default control options for HMC are:
  \code{control = list(Lmin = 10, Lmax = 20, epsilon = 0.005)}

Currently, the only implemented optimisation algorithm is Adagrad
  (\code{method = "adagrad"}). The \code{control} argument can be used to
  specify the optimiser hyperparameters: \code{learning_rate} (default 0.8),
  \code{initial_accumulator_value} (default 0.1) and \code{use_locking}
  (default \code{TRUE}). The are passed directly to TensorFlow's optimisers,
  see
  \href{https://www.tensorflow.org/api_docs/python/tf/train/AdagradOptimizer}{the
  TensorFlow docs} for more information
}
\examples{
\dontrun{
# define a simple model
mu = variable()
sigma = lognormal(1, 0.1)
x = rnorm(10)
distribution(x) = normal(mu, sigma)
m <- model(mu, sigma)

# carry out mcmc on the model
draws <- mcmc(m,
              n_samples = 100,
              warmup = 10)
}
\dontrun{
# find the MAP estimate
opt_res <- opt(m)
}
}
