% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derivatives.R
\name{partial_derivatives}
\alias{partial_derivatives}
\alias{partial_derivatives.default}
\alias{partial_derivatives.gamm}
\alias{partial_derivatives.gam}
\title{Partial derivatives of estimated multivariate smooths via finite
differences}
\usage{
partial_derivatives(object, ...)

\method{partial_derivatives}{default}(object, ...)

\method{partial_derivatives}{gamm}(object, ...)

\method{partial_derivatives}{gam}(
  object,
  select = NULL,
  term = deprecated(),
  focal = NULL,
  data = newdata,
  order = 1L,
  type = c("forward", "backward", "central"),
  n = 100,
  eps = 1e-07,
  interval = c("confidence", "simultaneous"),
  n_sim = 10000,
  level = 0.95,
  unconditional = FALSE,
  frequentist = FALSE,
  offset = NULL,
  ncores = 1,
  partial_match = FALSE,
  seed = NULL,
  ...,
  newdata = NULL
)
}
\arguments{
\item{object}{an R object to compute derivatives for.}

\item{...}{arguments passed to other methods.}

\item{select}{character; vector of one or more smooth terms for which
derivatives are required. If missing, derivatives for all smooth terms
will be returned. Can be a partial match to a smooth term; see argument
\code{partial_match} below.}

\item{term}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use \code{select} instead.}

\item{focal}{character; name of the focal variable. The partial derivative
of the estimated smooth with respect to this variable will be returned.
All other variables involved in the smooth will be held at constant. This
can be missing if supplying \code{data}, in which case, the focal variable will
be identified as the one variable that is not constant.}

\item{data}{a data frame containing the values of the model covariates
at which to evaluate the first derivatives of the smooths. If supplied,
all but one variable must be held at a constant value.}

\item{order}{numeric; the order of derivative.}

\item{type}{character; the type of finite difference used. One of
\code{"forward"}, \code{"backward"}, or \code{"central"}.}

\item{n}{numeric; the number of points to evaluate the derivative at.}

\item{eps}{numeric; the finite difference.}

\item{interval}{character; the type of interval to compute. One of
\code{"confidence"} for point-wise intervals, or \code{"simultaneous"} for
simultaneous intervals.}

\item{n_sim}{integer; the number of simulations used in computing the
simultaneous intervals.}

\item{level}{numeric; \verb{0 < level < 1}; the confidence level of the
point-wise or simultaneous interval. The default is \code{0.95} for a 95\%
interval.}

\item{unconditional}{logical; use smoothness selection-corrected Bayesian
covariance matrix?}

\item{frequentist}{logical; use the frequentist covariance matrix?}

\item{offset}{numeric; a value to use for any offset term}

\item{ncores}{number of cores for generating random variables from a
multivariate normal distribution. Passed to \code{\link[mvnfast:rmvn]{mvnfast::rmvn()}}.
Parallelization will take place only if OpenMP is supported (but appears
to work on Windows with current \code{R}).}

\item{partial_match}{logical; should smooths be selected by partial matches
with \code{term}? If \code{TRUE}, \code{term} can only be a single string to match
against.}

\item{seed}{numeric; RNG seed to use.}

\item{newdata}{Deprecated: use \code{data} instead.}
}
\value{
A tibble, currently with the following variables:
\itemize{
\item \code{.smooth}: the smooth each row refers to,
\item \code{.partial_deriv}: the estimated partial derivative,
\item \code{.se}: the standard error of the estimated partial derivative,
\item \code{.crit}: the critical value such that \code{derivative} ± \code{(crit * se)} gives
the upper and lower bounds of the requested confidence or simultaneous
interval (given \code{level}),
\item \code{.lower_ci}: the lower bound of the confidence or simultaneous interval,
\item \code{.upper_ci}: the upper bound of the confidence or simultaneous interval.
}
}
\description{
Partial derivatives of estimated multivariate smooths via finite
differences
}
\note{
\code{partial_derivatives()} will ignore any random effect smooths it
encounters in \code{object}.
}
\examples{

library("ggplot2")
library("patchwork")
load_mgcv()
\dontshow{
op <- options(pillar.sigfig = 3, cli.unicode = FALSE)
}
df <- data_sim("eg2", n = 2000, dist = "normal", scale = 0.5, seed = 42)

# fit the GAM (note: for execution time reasons, k is set articifially low)
m <- gam(y ~ te(x, z, k = c(5, 5)), data = df, method = "REML")

# data slice through te(x,z) holding z == 0.4
ds <- data_slice(m, x = evenly(x, n = 100), z = 0.4)

# evaluate te(x,z) at values of x & z
sm <- smooth_estimates(m, select = "te(x,z)", data = ds) |>
  add_confint()

# partial derivatives
pd_x <- partial_derivatives(m, data = ds, type = "central", focal = "x")

# draw te(x,z)
p1 <- draw(m, rug = FALSE) &
  geom_hline(yintercept = 0.4, linewidth = 1)
p1

# draw te(x,z) along slice
cap <- expression(z == 0.4)
p2 <- sm |>
  ggplot(aes(x = x, y = .estimate)) +
  geom_ribbon(aes(ymin = .lower_ci, ymax = .upper_ci), alpha = 0.2) +
  geom_line() +
  labs(
    x = "x", y = "Partial effect", title = "te(x,z)",
    caption = cap
  )
p2

# draw partial derivs
p3 <- pd_x |>
  draw() +
  labs(caption = cap)
p3

# draw all three panels
p1 + p2 + p3 + plot_layout(ncol = 3)
\dontshow{
options(op)
}
}
\author{
Gavin L. Simpson
}
