% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/buildsteps.R
\name{cr_buildstep}
\alias{cr_buildstep}
\title{Create a yaml build step}
\usage{
cr_buildstep(
  name,
  args = NULL,
  id = NULL,
  prefix = "gcr.io/cloud-builders/",
  entrypoint = NULL,
  dir = "",
  env = NULL,
  waitFor = NULL,
  volumes = NULL,
  secretEnv = NULL
)
}
\arguments{
\item{name}{name of docker image to call appended to \code{prefix}}

\item{args}{character vector of arguments}

\item{id}{Optional id for the step}

\item{prefix}{prefixed to name - set to "" to suppress.  Will be suppressed if \code{name} starts with gcr.io or \code{*-docker.pkg.dev}}

\item{entrypoint}{change the entrypoint for the docker container}

\item{dir}{The directory to use, relative to /workspace e.g. /workspace/deploy/}

\item{env}{Environment variables for this step.  A character vector for each assignment}

\item{waitFor}{Whether to wait for previous buildsteps to complete before running.  Default it will wait for previous step.}

\item{volumes}{volumes to connect and write to}

\item{secretEnv}{A list of secrets stored in Secret Manager referred to in args via a \code{$$var}}
}
\description{
Helper for creating build steps for upload to Cloud Build
}
\details{
This uses R to make building steps for cloudbuild.yml files harder to make mistakes with, and also means you can program creation of cloud build steps for use in R or other languages.  Various templates with common use cases of buildsteps are also available that wrap this function, refer to the "See Also" section.
}
\section{WaitFor}{


By default each buildstep waits for the previous, but if you pass \code{"-"} then it will start immediately, or if you pass in a list of ids it will wait for previous buildsteps to finish who have that id.  See \href{https://cloud.google.com/build/docs/configuring-builds/configure-build-step-order}{Configuring Build Step Order} for details.
}

\section{Build Macros}{

Fields can include the following variables, which will be expanded when the build is created:-

\itemize{
  \item $PROJECT_ID: the project ID of the build.
  \item $BUILD_ID: the autogenerated ID of the build.
  \item $REPO_NAME: the source repository name specified by RepoSource.
  \item $BRANCH_NAME: the branch name specified by RepoSource.
  \item $TAG_NAME: the tag name specified by RepoSource.
  \item $REVISION_ID or $COMMIT_SHA: the commit SHA specified by RepoSource or  resolved from the specified branch or tag.
  \item  $SHORT_SHA: first 7 characters of $REVISION_ID or $COMMIT_SHA.
}

Or you can add your own custom variables, set in the Build Trigger.  Custom variables always start with $_ e.g. $_MY_VAR
}

\section{secretEnv}{


You can pass secrets that are stored in Secret Manager directly instead of using a dedicated buildstep via \link{cr_buildstep_secret}

Within the code passed to \code{args} those secrets are referred to via \code{$$SECRET_NAME}.  If used then \link{cr_build_yaml} must also include the \code{availableSecrets} argument.
}

\examples{
cr_project_set("my-project")
cr_bucket_set("my-bucket")
# creating yaml for use in deploying cloud run
image <- "gcr.io/my-project/my-image:$BUILD_ID"
cr_build_yaml(
  steps = c(
    cr_buildstep("docker", c("build", "-t", image, ".")),
    cr_buildstep("docker", c("push", image)),
    cr_buildstep("gcloud", c(
      "beta", "run", "deploy", "test1",
      "--image", image
    ))
  ),
  images = image
)

# use premade docker buildstep - combine using c()
image <- "gcr.io/my-project/my-image"
cr_build_yaml(
  steps = c(
    cr_buildstep_docker(image),
    cr_buildstep("gcloud",
      args = c(
        "beta", "run", "deploy",
        "test1", "--image", image
      )
    )
  ),
  images = image
)

# list files with a new entrypoint for gcloud
cr_build_yaml(steps = cr_buildstep("gcloud", c("-c", "ls -la"),
  entrypoint = "bash"
))

# to call from images not using gcr.io/cloud-builders stem
cr_buildstep("alpine", c("-c", "ls -la"), entrypoint = "bash", prefix = "")

# to add environment arguments to the step
cr_buildstep("docker", "version", env = c("ENV1=env1", "ENV2=$PROJECT_ID"))

# to add volumes wrap in list()
cr_buildstep("test", "ls", volumes = list(list(name = "ssh", path = "/root/.ssh")))
}
\seealso{
\href{https://cloud.google.com/build/docs/configuring-builds/use-community-and-custom-builders}{Creating custom build steps how-to guide}

Other Cloud Buildsteps: 
\code{\link{cr_buildstep_bash}()},
\code{\link{cr_buildstep_decrypt}()},
\code{\link{cr_buildstep_df}()},
\code{\link{cr_buildstep_docker}()},
\code{\link{cr_buildstep_edit}()},
\code{\link{cr_buildstep_extract}()},
\code{\link{cr_buildstep_gcloud}()},
\code{\link{cr_buildstep_gitsetup}()},
\code{\link{cr_buildstep_mailgun}()},
\code{\link{cr_buildstep_nginx_setup}()},
\code{\link{cr_buildstep_packagetests}()},
\code{\link{cr_buildstep_pkgdown}()},
\code{\link{cr_buildstep_run}()},
\code{\link{cr_buildstep_r}()},
\code{\link{cr_buildstep_secret}()},
\code{\link{cr_buildstep_slack}()},
\code{\link{cr_buildstep_targets}()}
}
\concept{Cloud Buildsteps}
