% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data.R
\name{vdtest.gnm}
\alias{vdtest.gnm}
\title{Test for Varying Dispersion Parameter in Generalized Nonlinear Models}
\usage{
\method{vdtest}{gnm}(model, varformula, verbose = TRUE, ...)
}
\arguments{
\item{model}{an object of the class \emph{gnm} where the distribution of the response
variable is assumed to be \code{gaussian}, \code{Gamma} or \code{inverse.gaussian}.}

\item{varformula}{an (optional) \code{formula} expression of the form \code{~ z1 + z2 + ... + zq} describing only the potential explanatory variables for the dispersion. As default, the same explanatory variables are taken as in the model for the mean.}

\item{verbose}{an (optional) logical switch indicating if should the report of results be printed. As default, \code{verbose} is set to TRUE.}

\item{...}{further arguments passed to or from other methods.}
}
\value{
a list list with components including
\tabular{ll}{
\code{statistic} \tab value of the Rao's score test (\eqn{S}),\cr
\tab \cr
\code{df}        \tab number of degrees of freedom (\eqn{q}),\cr
\tab \cr
\code{p.value}   \tab \emph{p}-value of the test,\cr
\tab \cr
\code{vars}   \tab names of explanatory variables for the dispersion parameter,\cr
}
}
\description{
Performs Rao's score test for varying dispersion parameter in
weighted and unweighted generalized nonlinear models in which the response
distribution is assumed to be Gaussian, Gamma, or inverse Gaussian.
}
\details{
From the generalized nonlinear model with varying dispersion in which
\eqn{\log(\phi)=\gamma_0 + \gamma_1z_1 + \gamma_2z_2 + ... + \gamma_qz_q}, where
\eqn{\phi} is the dispersion parameter of the distribution used to describe the
response variable, the Rao's score test (denoted here as \eqn{S}) to assess the
hypothesis \eqn{H_0: \gamma=0} versus \eqn{H_1: \gamma\neq 0} is computed,
where \eqn{\gamma=(\gamma_1,\ldots,\gamma_q)}.  The corresponding \emph{p}-value is
computed from the chi-squared distribution with \eqn{q} degrees of freedom,
that is, \emph{p}-value = Prob\eqn{[\chi^2_{q} > S]}. If the object \code{model}
corresponds to an unweighted generalized linear model, this test assesses assumptions
of constant variance and constant coefficient of variation on models in which the
response distribution is assumed to be Gaussian and Gamma, respectively.
}
\examples{
###### Example 1: The effects of fertilizers on coastal Bermuda grass
data(Grass)
fit1 <- gnm(Yield ~ b0 + b1/(Nitrogen + a1) + b2/(Phosphorus + a2) + b3/(Potassium + a3),
            family=gaussian(inverse), start=c(b0=0.1,b1=13,b2=1,b3=1,a1=45,a2=15,a3=30), data=Grass)
vdtest(fit1)
vdtest(fit1,varformula = ~ Nitrogen + Phosphorus + Potassium)

###### Example 2: Developmental rate of Drosophila melanogaster
data(Drosophila)
fit2 <- gnm(Duration ~ b0 + b1*Temp + b2/(Temp-a), family=Gamma(log),
            start=c(b0=3,b1=-0.25,b2=-210,a=55), weights=Size, data=Drosophila)
vdtest(fit2)
vdtest(fit2,varformula = ~ Temp)
vdtest(fit2,varformula = ~ log(Temp))

###### Example 3: Radioimmunological Assay of Cortisol
data(Cortisol)
fit3 <- gnm(Y ~ b0 + (b1-b0)/(1 + exp(b2+ b3*lDose))^b4, family=Gamma(identity),
            start=c(b0=130,b1=2800,b2=3,b3=3,b4=0.5), data=Cortisol)
vdtest(fit3)
vdtest(fit3,varformula = ~ lDose)
vdtest(fit3,varformula = ~ exp(lDose))

###### Example 4: Age and Eye Lens Weight of Rabbits in Australia
data(rabbits)
fit4 <- gnm(wlens ~ b1 - b2/(age + b3), family=Gamma(log),
            start=c(b1=5.5,b2=130,b3=35), data=rabbits)
vdtest(fit4)
vdtest(fit4,varformula = ~ age)
vdtest(fit4,varformula = ~ log(age))
}
\references{
Wei B.-C., Shi, J.-Q., Fung W.-K., Hu Y.-Q. (1998) Testing for Varying Dispersion in Exponential Family Nonlinear Models. \emph{Annals of the Institute of Statistical Mathematics} 50, 277–294.
}
\seealso{
\link{vdtest.lm}, \link{vdtest.glm}
}
