% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gisco_get_airports.R
\name{gisco_get_airports}
\alias{gisco_get_airports}
\alias{gisco_get_ports}
\title{Get location of airports and ports from GISCO API}
\source{
\url{https://ec.europa.eu/eurostat/web/gisco/geodata/transport-networks}
}
\usage{
gisco_get_airports(
  year = "2013",
  country = NULL,
  cache_dir = NULL,
  update_cache = FALSE,
  verbose = FALSE
)

gisco_get_ports(
  year = "2013",
  country = NULL,
  cache_dir = NULL,
  update_cache = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{year}{Year of reference. Only year available right now is \code{"2013"}.}

\item{country}{Optional. A character vector of country codes. It could be
either a vector of country names, a vector of ISO3 country codes or a
vector of Eurostat country codes. Mixed types (as \code{c("Italy","ES","FRA")})
would not work. See also \code{\link[countrycode:countrycode]{countrycode::countrycode()}}.}

\item{cache_dir}{A path to a cache directory. See \strong{About caching}.}

\item{update_cache}{A logical whether to update cache. Default is \code{FALSE}.
When set to \code{TRUE} it would force a fresh download of the source
\code{.geojson} file.}

\item{verbose}{Logical, displays information. Useful for debugging,
default is \code{FALSE}.}
}
\value{
A \code{POINT} object on EPSG:4326.
}
\description{
Loads a \code{\link[sf:sf]{sf}} object from GISCO API or your local library.
}
\details{
\code{\link[=gisco_get_airports]{gisco_get_airports()}} refer to Europe. All shapefiles provided in
\href{https://epsg.io/4326}{EPSG:4326}.

\code{\link[=gisco_get_ports]{gisco_get_ports()}} adds a new field \code{CNTR_ISO2} to the original data
identifying the country of the port. Worldwide information available.
The port codes are aligned with
\href{https://unece.org/trade/uncefact/unlocode}{UN/LOCODE} standard.
}
\section{About caching}{
You can set your \code{cache_dir} with \code{\link[=gisco_set_cache_dir]{gisco_set_cache_dir()}}.

Sometimes cached files may be corrupt. On that case, try re-downloading
the data setting \code{update_cache = TRUE}.

If you experience any problem on download, try to download the
corresponding \code{.geojson} file by any other method and save it on your
\code{cache_dir}. Use the option \code{verbose = TRUE} for debugging the API query.

For a complete list of files available check \link{gisco_db}.
}

\examples{
\dontshow{if (gisco_check_access()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
library(sf)

greece <- gisco_get_countries(country = "EL", resolution = 3)
airp_gc <- gisco_get_airports(country = "EL")

library(ggplot2)

if (inherits(airp_gc, "sf")) {
  ggplot(greece) +
    geom_sf(fill = "grey80") +
    geom_sf(data = airp_gc, color = "blue") +
    labs(
      title = "Airports on Greece",
      shape = NULL,
      color = NULL,
      caption = gisco_attributions()
    )
}
##############################
#         Plot ports         #
##############################

ports <- gisco_get_ports()
coast <- giscoR::gisco_coastallines

# To Robinson projection :)

library(sf)
coast <- st_transform(coast, "ESRI:54030")
ports <- st_transform(ports, st_crs(coast))

if (inherits(ports, "sf")) {
  ggplot(coast) +
    geom_sf(fill = "#F6E1B9", color = "#0978AB") +
    geom_sf(data = ports, fill = "red", shape = 21) +
    theme_void() +
    theme(
      panel.background = element_rect(fill = "#C6ECFF"),
      panel.grid = element_blank(),
      plot.title = element_text(face = "bold", hjust = 0.5),
      plot.subtitle = element_text(face = "italic", hjust = 0.5)
    ) +
    labs(
      title = "Ports Worldwide", subtitle = "Year 2013",
      caption = "(c) European Union, 1995 - today"
    )
}
}
\dontshow{\}) # examplesIf}
}
\seealso{
Other infrastructure: 
\code{\link{gisco_get_education}()},
\code{\link{gisco_get_healthcare}()}
}
\concept{infrastructure}
