% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/contin_inhibit_sample.r
\name{contin.inhibit.sample}
\alias{contin.inhibit.sample}
\title{Spatially continuous sampling}
\usage{
contin.inhibit.sample(poly, size, delta, delta.fix = FALSE, k = 0,
  rho = NULL, ntries = 10000, plotit = TRUE)
}
\arguments{
\item{poly}{a \code{sf} or \code{sp} polygon in which to generate the design.}

\item{size}{a non-negative integer giving the total number of locations to be sampled.}

\item{delta}{minimum permissible distance between any two locations in preliminary sample. This can be allowed to vary with the number of \code{'close pairs'} if a \bold{simple inhibitory} design is compared to one of the \bold{inhibitory plus close pairs} design.}

\item{delta.fix}{'logical' specifies whether \code{delta} is fixed or allowed to vary with number of close pairs \eqn{k}. Default is \code{delta.fix = FALSE}.}

\item{k}{number of locations in preliminary sample to be replaced by near neighbours of other preliminary sample locations to form \code{close pairs} (integer between 0 and \code{size/2}). A \bold{simple inhibitory} deisgn is generated when \eqn{k = 0}.}

\item{rho}{maximum distance between the two locations in a \code{'close-pair'}.}

\item{ntries}{number of rejected proposals after which the algorithm will terminate.}

\item{plotit}{'logical' specifying if graphical output is required. Default is \code{plotit = TRUE}.}
}
\value{
a list with the following four components:

\code{size:} the total number of sampled locations.

\code{delta:} the value of \eqn{\delta} after taking into account the number of close pairs \eqn{k}. If \code{delta.fix = TRUE}, this will be \eqn{\delta} input by the user.

\eqn{k:} the number of close pairs included in the sample (for \bold{inhibitory plus close pairs} design).

\code{sample.locs:} a \code{sf} or \code{sp} object containing coordinates of dimension \code{n} by 2 containing the sampled locations.
}
\description{
Draws a spatially continous sample of locations within a polygonal sampling region according to an \bold{"inhibitory plus close pairs"} specification.
}
\details{
To draw a simple inhibitory (\bold{SI}) sample of size \code{n}  from a spatially continuous region \eqn{A}, with the property that the distance between any two sampled locations is at least \code{delta}, the following algorithm is used.
\itemize{
\item{Step 1.} Set \eqn{i  = 1} and generate a point \eqn{x_{1}}  uniformly distributed on \eqn{{\cal D}}.
\item{Step 2.} Generate a point \eqn{x}  uniformly distributed on \eqn{{\cal D}} and calculate the minimum, \eqn{d_{\min}}, of the distances from \eqn{x_{i}} to all \eqn{x_{j}: j \leq i }.
\item{Step 3.} If \eqn{d_{\min} \ge \delta}, increase \eqn{i}  by 1, set \eqn{x_{i} = x} and return to step 2 if \eqn{i \le n}, otherwise stop;
\item{Step 4.} If \eqn{d_{\min} < \delta}, return to step 2 without increasing \eqn{i}.
}

\bold{Sampling close pairs of points.}

For some purposes, it is desirable that a spatial sampling scheme include pairs of closely spaced points, resulting in an inhibitory plus close pairs (\bold{ICP}) design. In this case, the above algorithm requires the following additional steps to be taken.
Let \eqn{k}  be the required number of close pairs. Choose a value \code{rho}  such that a close pair  of points will be a pair of points separated by a distance of at most \code{rho}.
\itemize{
\item{Step 5.} Set \eqn{j  = 1} and draw a random sample of size 2 from integers \eqn{1, 2, \ldots, n}, say \eqn{(i_1, i_2)};
\item{Step 6.} Replace \eqn{x_{i_{1}}} by \eqn{x_{i_{2}} + u} , where \eqn{u}  is uniformly distributed on the disc with centre \eqn{x_{i_{2}}} and radius \code{rho}, increase \eqn{i} by 1 and return to step 5 if \eqn{i \le k}, otherwise stop.
}

When comparing a \bold{SI} design to one of the \bold{ICP} designs, the inhibitory components should have the same degree of spatial regularity.
This requires \eqn{\delta} to become a function of \eqn{k} namely \deqn{\delta_{k} = \delta_{0}\sqrt{n/(n - k)}} with \eqn{\delta_{0}} held fixed.
}
\note{
If \code{'delta'} is set to 0, a completely random sample is generated. In this case, \code{'close pairs'} are not permitted and \code{rho} is irrelevant.
}
\examples{
library("geoR")
library("sf")
data("parana")
poly <- parana$borders
poly <- matrix(c(poly[,1],poly[,2]),dim(poly)[1],2,byrow=FALSE)
#convert matrix to polygon
poly <- st_sf(st_sfc(st_polygon(list(as.matrix(poly)))))
#poly <- as(poly, "Spatial")
poly

# Generate spatially regular sample
set.seed(5871121)
xy.sample1 <- contin.inhibit.sample(poly=poly,size = 100, delta = 30, plotit = TRUE)


# Generate spatially regular sample with 10 close pairs
set.seed(5871122)
xy.sample2 <- contin.inhibit.sample(poly,size = 100, delta = 30,
                                    k = 5, rho = 15, plotit = TRUE)

# Generate spatially regular sample with 10 close pairs
set.seed(5871123)
xy.sample3 <- contin.inhibit.sample(poly,size = 100, delta = 30, delta.fix = TRUE,
                                    k = 10, rho = 15, plotit = TRUE)

}
\references{
Chipeta  M G, Terlouw D J, Phiri K S and Diggle P J. (2016b). Inhibitory geostatistical designs for spatial prediction taking account of uncertain covariance structure, \emph{Enviromentrics}, pp. 1-11.
}
\seealso{
\code{\link[geosample:random.sample]{random.sample}} and \code{\link[geosample:discrete.inhibit.sample]{discrete.inhibit.sample}}
}
\author{
Michael G. Chipeta \email{mchipeta@mlw.mw}

Peter J. Diggle \email{p.diggle@lancaster.ac.uk}
}
