test_that("intersectn can run on overlapping triangles", {
  ## Make star of David from isosceles triangles of length 3
  ps1 <- rbind(c(0,   sqrt(3)),
               c(3/2, -sqrt(3)/2),
               c(-3/2, -sqrt(3)/2))
  ps2 <- ps1
  ps2[,2] <- -ps2[,2]

  expect_equal(feasible.point(convhulln(ps1, output.options=TRUE),
                              convhulln(ps2, output.options=TRUE)),
               c(0, 0))
  is <-  intersectn(ps1, ps2)
  ## Intersecting area is same as 6 isosceles triangles of length 1, which have 
  ## area sqrt(3)/4
  ## 
  expect_equal(is$ch$vol, sqrt(3)/4*6)

  ## Another overlapping example
  ps2 <- ps1
  ps2[,2] <- ps2[,2]+2
  is <-  intersectn(ps1, ps2)

  ## Now make one element of feasible point negative
  ps3 <- ps1
  ps4 <- ps1
  ps4[,2] <- -ps4[,2]
  ps3[,2] <- ps3[,2] - 10
  ps4[,2] <- ps4[,2] - 10

  expect_equal(feasible.point(convhulln(ps3, output.options=TRUE),
                              convhulln(ps4, output.options=TRUE)),
               c(0, -10))
  expect_equal(intersectn(ps3, ps4)$ch$vol, sqrt(3)/4*6)
})

test_that("intersectn gives zero volume on non-overlapping triangles", {
  ps1 <- rbind(c(0,   sqrt(3)),
               c(3/2, -sqrt(3)/2),
               c(-3/2, -sqrt(3)/2))
  ps2 <- ps1
  ps2[,2] <- ps2[,2] + 3
  
  expect_equal(feasible.point(convhulln(ps1, "n"), convhulln(ps2, "n")), NA)
  is <-  intersectn(ps1, ps2)
  expect_equal(is$ch$vol, 0)
  
})


test_that("intersectn gives zero volume on non-overlapping triangles", {
  ps1 <- rbind(c(0,   sqrt(3)),
               c(3/2, -sqrt(3)/2),
               c(-3/2, -sqrt(3)/2))
  ps2 <- ps1
  ps2[,2] <- ps2[,2] + 3
  
  expect_equal(feasible.point(convhulln(ps1, "n"), convhulln(ps2, "n")), NA)
  is <-  intersectn(ps1, ps2)
  expect_equal(is$ch$vol, 0)
  
})

test_that("feasible.point works on a 3D example", {
  ## These tetrahedra do not overlap
  ps1 <- rbind(c( 0.5000000, -0.5000000,  0.5000000),
               c(-0.1018942,  0.1848312, -0.1260239),
               c( 0.5000000, -0.5000000, -0.5000000),
               c(-0.5000000, -0.5000000, -0.5000000))
  ps2 <- rbind(c( 0.7581575,  0.6352585,  0.32876),
               c( 1.0000000,  0.0000000,  1.00000),
               c( 0.0000000,  0.0000000,  1.00000),
               c( 1.0000000,  0.0000000,  0.00000))
  expect_equal(feasible.point(convhulln(ps1, "n"), convhulln(ps2, "n")), NA)
})

test_that("intersectn can run on overlapping tetrahedra", {
  ## Make star of David from isocelese triangles of length 3
  ps1 <- rbind(c(0,   sqrt(3), 0),
               c(3/2, -sqrt(3)/2, 0),
               c(-3/2, -sqrt(3)/2, 0),
               c(0, 0, 3*sqrt(2/3)))
  ch1 <- convhulln(ps1, "FA")
  expect_equal(ch1$vol, sqrt(2)/12*27)
  ps2 <- ps1

  ## By shifting tetrahedron up by half of its height, we should make
  ## something with 1/8 of the volume
  ps2[,3] <- ps2[,3] + 3/2*sqrt(2/3)
  is <-  intersectn(ps1, ps2)
  expect_equal(is$ch$vol, sqrt(2)/12*27/8)
  
})

test_that("intersectn can run on  tetrahedra with a common point", {
  ps1 <- rbind(c(-0.4015654, -0.1084358, -0.3727391),
               c( 0.2384763,  0.3896078, -0.4447473),
               c( 0.5000000, -0.5000000, -0.5000000),
               c(-0.5000000, -0.5000000, -0.5000000))
  ps2 <- rbind(c(-0.1392469,  0.03303547, -0.2436112),
               c( 0.3434195, -0.20338201, -0.4638141),
               c(-0.5000000,  0.50000000, -0.5000000),
               c(-0.5000000, -0.50000000, -0.5000000))
  is <-  intersectn(ps1, ps2)
})

test_that("intersectn can compute the volume of overlapping delaunay triangulations of boxes", {
  ## Volume of overlap should be 1/8
  ps1 <- rbox(2, B=0.5, C=0.5)
  ps2 <- rbox(2, B=0.5, C=0.5) + 0.5
  dt1 <- delaunayn(ps1)
  dt2 <- delaunayn(ps2)

  vol <- 0
  for (i in 1:nrow(dt1)) {
    for (j in 1:nrow(dt2)) {
      is <- intersectn(ps1[dt1[i,],], ps2[dt2[j,],])
      vol <- vol + is$ch$vol
    }
  }
  expect_equal(vol, 0.125, tol=0.0001)
})

test_that("intersectn can deal with some input that caused errors before fixing Issue #34", {
  ## Issue 34: https://github.com/davidcsterratt/geometry/issues/34
  ps1 <- rbind(
    c(500.9656357388012111187, 843268.9656357388012111, 5.5),
    c(658.9656357388012111187, 843109.9656357388012111, 10.0),
    c(576.9656357388012111187, 843174.9656357388012111,  2.0),
    c(795.9656357388012111187, 843235.9656357388012111, 20.0))
  ps2 <- rbind(
    c(707.9656400000000076034, 843153.9656399999512359, 12.000000000000000000000),
    c(645.6795799999999871943, 843166.4228499999735504, 10.200630000000000308091),
    c(631.6632399999999734064, 843182.9680800000205636,  8.772800000000000153477),
    c(707.9656400000000076034, 843153.9656399999512359, 12.000000000000000000000),
    c(608.9447900000000117871, 843172.7368899999419227,  7.772330000000000183036),
    c(607.9656400000000076034, 843173.9656399999512359,  7.669999999999999928946))
  ## Before Issue #34 was fixed this threw an error:
  ##   Received error code 2 from qhull. Qhull error:
  ##     qhull precision warning: 
  ##     The initial hull is narrow (cosine of min. angle is 1.0000000000000002).
  ## expect_error(intersectn(ps1, ps2, tol=1E-4, return.chs=FALSE, options="Tv"), ".*The initial hull is narrow.*")
  
  ## This threw an error in Rev aab45b7311b6
  out <- intersectn(ps1, ps2, tol=1E-4, return.chs=FALSE)
})

test_that("intersectn works in 4D", {
  seti <- structure(c(-1.20484799657496, 2.82200108251569, -0.899100156771137,
                      -0.761137469938235, 1.87272229281886, 3.15634835946662, 0.334530106394761,
                      2.09805617202296, 0.481201468838801, -0.164820595285586, -0.124206925786417,
                      0.0986071805998422, 0.817410269758491, 5.94509840746463, 1.43593006120388,
                      3.02107274205897, 1.68798332097861, -0.356863387711833, -1.07439981004341,
                      1.34298428431043, -0.628312706921134, 1.00162353656828, 1.07881967525775,
                      -0.4424483398044, -1.63299474001777, -0.569218072946004, 0.114900722107052,
                      -1.01085777445739, 0.138132168513739, 1.61746045606728, 1.2777976182106,
                      1.78294137701302, 1.44181831676921, 0.295143741496781, 0.510818512873005,
                      0.402601938714911, 1.16725171065398, 2.11831029185341, 0.0862509748448244,
                      0.361170603378489, 0.174814509665511, 0.273994224037487, 2.50182406934459,
                      2.09290298086104, 1.74698129392119, 1.47344263761709, 1.70657680301141,
                      1.92885763880671, -1.31530711548402, 1.0709426827893, -0.144265321086227,
                      3.08409477098756, 0.81912831760118, -1.10087292033274, -1.14991520357842,
                      0.330384999517859, 0.206357259857737, 3.84397299120238, 0.838986605624141,
                      2.2863133618924, 3.7706379495456, 1.2641868824271, -0.33077652792232,
                      -1.01747356001922, 0.141755058946007, 0.474608757978566, -0.658175656709011,
                      -1.59007655879819, -1.02651048612293, -0.988872363740542, -1.05369225499655,
                      -0.879098595410646, -1.0882175050281, -0.893457858700459, -0.117226455442837,
                      1.51939963593096, 1.72427445715258, 1.04737269768519, 1.17992535465126,
                      0.443948770797851, 2.24282696862518, 1.3132143264928, 2.26879851539103,
                      0.906947698327475, 1.09803081952012, 2.32329746826392, -0.628882742375884,
                      -1.21969543588583, -0.864943677174184, 1.32036011481997, 1.85626393647277,
                      1.59297704175946, 0.638819672720274, 1.67485095803488, 1.09121158005817,
                      -0.65226531994379, -0.24613817058787, 0.200482761874617, 10.1155818058019,
                      9.1934949678319, 4.23587353234132, -0.922323882604712, -0.661109698766882,
                      -0.114445095077024, -0.976296310931884, -1.43719510163004, -0.597217811282064,
                      1.18799294027513, -1.47368582779866, -0.449833487633611, -1.0110989869647,
                      -1.02491246754603, 2.14926856018086, 2.49543050239802, 1.24234059464143,
                      1.41932183694845, -3.52432543945342, 0.921086450400769, 0.92606560694932,
                      0.662598398116459, -0.676891022870777, 0.167984770894866, -1.53584072003236,
                      -3.12387300576501, -0.334873911056691, -1.58624587348191, -1.24801837868748,
                      -1.55297064680075, -1.00671415572242, -0.544841201023118, 1.4673190055672,
                      0.393082436327491, -0.819688053856946, -1.95495460866541, -0.760109405958924,
                      -1.92947570154211, -1.25427856120652, -0.96277638557598, -1.71744392148523,
                      0.773408286440073, 0.449245440177339, 0.297132163753282, 0.836322906375264,
                      0.131600643471952, 0.692589919245267, -0.285163480064811, -1.20824479942059,
                      -0.333099386674013, 0.90207499940522, -0.750043763527636, -1.71426292409226,
                      -1.20243525528554, 0.0394421869515282, 1.5422674767249, -0.259683271458097,
                      -0.872914174379128, -1.37941422470484, 2.50786072927067, 0.946803591769429,
                      -0.275714442591009, -0.0747497514019457, -0.67728532683017, 
                      1.73359280363577,
                      4.71707287368381, -0.0575696643566713, -1.99416101975695, -1.73343892292288,
                      -0.0819836583294515, -0.864944566136088, -2.53387248115194, -1.58938888273248,
                      -0.162858995905063, 0.0312568794352992, 
                      3.72904794788886, -0.639493226828736,
                      -0.968242685084208, 1.60482121231231, 1.88909344758905, -0.0112852948959791,
                      0.289112880955642, 0.642393450045674, -0.231179885744409, 1.45442960544931,
                      0.161543435927181, 0.802416026941346, 0.890386856879663, 0.269203436106351,
                      1.25121422637435, 0.81088336712333, 0.571730064920141, 0.764542604190663,
                      -1.9789913502926, -0.354389579720782, -1.00107404933639, -0.736680468180162,
                      -1.52864880135588, 0.313378939235502, -1.66970751786182, -0.643204425876921,
                      -0.874026229824324, -0.406882202845015, -0.447843890958249, -0.705836017333359,
                      -0.160090247261022, -0.324378094655366, -0.958843769323911, -1.52961543678282,
                      -0.244934320456502, 0.329786980968729, -0.282597642578179, 
                      0.103386624020509,
                      0.662213023652907, 0.0927401516657738, 1.56025202004909, 4.67878034567083,
                      4.66364376034552, 4.42847902381341, 0.670529447302663, 0.743204373971039,
                      0.892273763515352, 1.24826437723671, 0.227820001820949, 0.173495278596237,
                      0.575298841742683, 1.0913577788078, 0.930910104117228, 0.515890150790095,
                      3.38213484028453, -0.179469007625257, -0.630761001434975, -1.33910678001761,
                      -1.17148115817088, -8.05280794333236, 0.846418029251948, 0.943911410782081,
                      0.21289448843008, -0.44655576373837, -6.62386103393725, -0.778621877752637,
                      -7.68410651642059, -0.753571147744981, 0.566289104442078, 1.22050597546035,
                      0.166305510315177, 0.63680622358747, -3.63366056447483, 0.599357522692497,
                      0.340816406682826, 1.01003819852888, -1.15437782595199, 0.411064370991134,
                      0.916030962439134, -0.777978995893413, -0.806224137733504, -2.79762794812539,
                      0.976830454348644, 1.02404712766003, 0.964496872328393, 1.29418412576706,
                      0.052224765499911, 1.08740706747401, -0.0209120223358577, -1.57543205970367,
                      -0.84387369259261, 0.950235103576597, 0.710014611908787, 0.620617246389649,
                      0.479322129923222, 0.577718738970787, 0.882800047725343, 1.52030360579403,
                      0.93742629543439, 0.30362186945638, 1.1687872349504, 2.22341206560769,
                      0.92652964361512, 1.01957035010219, 1.30171311913142, -0.149274724285204,
                      0.00757023695818217, 0.454223822629539, 0.532856179376281, 1.21471261834645,
                      0.063956779355217, -1.9657453724823, -1.7773213083232, -1.3788958334607,
                      0.151114855330817, -1.26110212396932, -3.40939638684843, 0.574595651314148,
                      -0.0335719378212932, -6.25042560840886, 0.685787141233975, 1.58080791547358,
                      1.63573412425179, 0.681000179927472, 0.558609527583925, 0.288789930345576,
                      0.366721210641253, 0.432819570529014, 0.461688910118864, 0.778077845773192,
                      0.644063364714034, 0.614666747139133, 0.759414465051408, 0.927642809463859,
                      0.580803693964964, 1.17481786697446, 1.03890240786419, 1.21812804254473,
                      0.6283026051494, 1.98778475907356, 1.01575849356055, 1.4425929501343,
                      0.482861918570303, 1.13721666727394, 1.7201011394216, 0.901676201857483,
                      -0.968524501308496, -1.33012821267484, 1.22546680111774, 1.43773588072909,
                      0.487582615495005, 0.763755747075563, 0.416833492899437, 0.872485714800342,
                      -0.817889950002025, -0.738772875934499, 1.76926532524203, -6.07952878758855,
                      -5.64304520553749, -7.33061973209768, 0.478377757123796, 1.14783976429064,
                      0.376029614346347, 0.833263576811544, 1.22402651338212, 1.17475572505127,
                      0.463962202607282, 1.91594734290813, 0.712182415942064, 2.25076972498324,
                      0.103041487769464, 0.916281197800209, 1.05915750993665, 0.978288151407379,
                      1.53938242436147, 8.83190026265947, 0.555374339856875, 1.17007187562197,
                      0.372020046749031, -0.176158192351193, 0.0641288315786146, 2.16765382172273,
                      8.73986954200266, 1.1012423259372, -0.378589783226768, -0.517655657573868,
                      -0.379056783545116, -0.927398269106298, 2.13931350330768, 0.899523625484025,
                      -0.817988186484915, -0.535772521674405, 1.11042730311383, 4.23751771713984,
                      -0.768109467037727, -0.212849120187299, 1.38852857456175, 2.61803851187512,
                      1.06617699991509, 1.20197012059196, 0.929221459337807, 1.79672052316459,
                      0.62886086850437, 1.34605526390479, 0.0605577196886461, 3.11496486767917,
                      -0.0618114273730384, -0.756225179554681, 1.42829116600036, 
                      0.176008648222713,
                      0.206187497307935, 0.248604376418107, 1.18875881628006, -0.0242879602052556,
                      0.538978655541758, 
                      0.170571648966258, -0.00307960872430968, -0.913235203685699,
                      -1.22905548126324, -1.01134061971383, -0.0344089235375279, 1.32135132643056,
                      -0.0877601173864905, 
                      0.987164356141447, -0.88012255031617, -0.570360567290289,
                      -1.1381582734518, 3.21038072554745, 1.82191358860395, 2.63576579111517,
                      2.18285652654603, 0.671409717482476, -3.66101745864741, -1.20913045492514,
                      1.07176860448667, -1.63306996627716, 1.00360193512679, 1.88955017801555,
                      2.09705727953261, 1.01066573113643, 1.85287814346596, 0.675087453882825,
                      0.869602725964551, 0.842844127716402, 0.885999180970363, 0.36933415205967,
                      0.695295119313493, 0.493872030471009, 1.48955125724987, 1.94921244161194,
                      -1.04376882397965, -0.668521147250858, -0.301523709865078, -0.480600001013401,
                      -0.630087856603127, -0.514077354594145, -0.0770129380496566,
                      -0.705664073939674, -0.788852122528463, -0.550856731083188, -0.851270511793045,
                      0.863387538946008, -0.0749943775610221, 
                      0.635508161740975, -0.192148256395783,
                      -0.733669802373726, -0.381928943350642, -0.426311006874752, -0.297609526279491,
                      -0.186742652399271, -0.469004456597525, -0.00759270905788778,
                      1.06157098823808, 0.345417433771567, 0.766481474313542, -0.809922283242965,
                      0.792206914918604, 0.945318164384887, 0.191390484693526, 0.573745223878958,
                      0.272025834260555, 0.50377895546262, 0.0997421096514963, 1.05123680998101,
                      0.376198497619751, 0.445778425538035, 0.424805624421032, -1.07187301135746,
                      -1.01881980221873, -0.342583128150628, -0.132448752701073),
                    .Dim = c(116L, 4L))

                                        # the second one
  setj <- structure(c(1.16158321759836, 3.15634835946662, 1.3697271648662,
                      1.67364122725686, 1.88487441662601, 1.1798367224259, 1.45482858518368,
                      1.94380937427165, 0.361236068752765, 0.167984770894866, -0.587591496477515,
                      0.083832472444175, 0.236675728804549, -1.12807427716432, 0.413391222698197,
                      0.087212128837941, -0.321307166822276, -6.62386103393725, 0.86635452640865,
                      -1.27336208334175, -1.04668045096666, -2.92588599658233, -1.19746002110571,
                      -0.953037108123469, 0.308463816265702, 
                      0.0641288315786146, -0.488506401739884,
                      2.8481359326713, 0.35247765120558, 3.1943556818665, 2.50526529661154,
                      1.88671908720816), .Dim = c(8L, 4L))
  
  chi <- convhulln(seti, output.options=TRUE)
  chj <- convhulln(setj, output.options=TRUE)
  chij <- intersectn(seti, setj)
  chji <- intersectn(setj, seti)
  expect_equal(chij$ch$vol, chji$ch$vol)
  expect_true(chi$vol >= chij$ch$vol)
  expect_equal(chj$vol, chij$ch$vol)
})
