# %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
##### Cmeans algorithm function #####
# %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

#' @title C-means
#'
#' @description The classical c-mean algorithm
#'
#' @template data_fcm-arg
#' @template k_m-arg
#' @template shared_paramsfcm-arg
#' @template FCMres_return
#' @export
#' @examples
#' data(LyonIris)
#' AnalysisFields <-c("Lden","NO2","PM25","VegHautPrt","Pct0_14","Pct_65","Pct_Img",
#' "TxChom1564","Pct_brevet","NivVieMed")
#' dataset <- LyonIris@data[AnalysisFields]
#' result <- CMeans(dataset,k = 5, m = 1.5, standardize = TRUE)
CMeans <- function(data, k, m, maxiter = 500, tol = 0.01, standardize = TRUE, verbose = TRUE, init = "random", seed = NULL) {

    data_class <- class(data)

    if (verbose & standardize){
        print("Standardizing the data (set parameter to FALSE to avoid this step)")
    }

    if(data_class == "list"){ # if we have to deal with a raster dataset
        elements <- input_raster_data(dataset  = data,
                                      standardize = standardize)
        data <- elements$data
        missing <- elements$missing
        rst <- elements$rst

    }else if(data_class == "data.frame"){ # if we have to deal with a dataframe

        # standardize data if required
        if (standardize) {
            for (i in 1:ncol(data)) {
                data[, i] <- scale(data[, i])
            }
        }
        data <- as.matrix(data)
    }

    results <- main_worker("FCM", data = data, k = k, m = m,
                           maxiter = maxiter, tol = tol, standardize = standardize,
                           verbose = verbose, seed = seed, init = init
                           )

    # if we are working with rasters, we should set some additional values
    if(data_class == "list"){
        results <- output_raster_data(results, missing, rst)
    }


    return(results)
}


# %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
##### Spatial Cmeans algorithm function #####
# %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


#' @title SFCMeans
#'
#' @description spatial version of the c-mean algorithm (SFCMeans, FCM_S1)
#'
#' @details The implementation is based on the following article : \doi{10.1016/j.patcog.2006.07.011}.\cr
#'
#' the membership matrix (u) is calculated as follow \cr
#' \deqn{u_{ik} = \frac{(||x_{k} - v{_i}||^2 + \alpha||\bar{x_{k}} - v{_i}||^2)^{(-1/(m-1))}}{\sum_{j=1}^c(||x_{k} - v{_j}||^2 + \alpha||\bar{x_{k}} - v{_j}||^2)^{(-1/(m-1))}}}
#'
#' the centers of the groups are updated with the following formula
#' \deqn{v_{i} = \frac{\sum_{k=1}^N u_{ik}^m(x_{k} + \alpha\bar{x_{k}})}{(1 + \alpha)\sum_{k=1}^N u_{ik}^m}}
#'
#' with
#' \itemize{
#' \item vi the center of the group vi
#' \item xk the data point k
#' \item xk_bar the spatially lagged data point k
#' }
#'
#' @template data_fcm-arg
#' @template nblistw-arg
#' @template k_m-arg
#' @param alpha A float representing the weight of the space in the analysis (0
#'   is a typical fuzzy-c-mean algorithm, 1 is balanced between the two
#'   dimensions, 2 is twice the weight for space)
#' @template lag_method-arg
#' @template window-arg
#' @template shared_paramsfcm-arg
#' @template FCMres_return
#' @export
#' @examples
#' data(LyonIris)
#' AnalysisFields <-c("Lden","NO2","PM25","VegHautPrt","Pct0_14","Pct_65","Pct_Img",
#' "TxChom1564","Pct_brevet","NivVieMed")
#' dataset <- LyonIris@data[AnalysisFields]
#' queen <- spdep::poly2nb(LyonIris,queen=TRUE)
#' Wqueen <- spdep::nb2listw(queen,style="W")
#' result <- SFCMeans(dataset, Wqueen,k = 5, m = 1.5, alpha = 1.5, standardize = TRUE)
SFCMeans <- function(data, nblistw = NULL, k, m, alpha, lag_method="mean",
                     window = NULL,
                     maxiter = 500, tol = 0.01, standardize = TRUE, verbose = TRUE,
                     init = "random", seed = NULL) {

    data_class <- class(data)
    if(data_class != "list"){
        if(class(nblistw)[[1]] != "listw"){
            stop("the nblistw must be a listw object from spdep package")
        }

        if(lag_method %in% c("mean","median") == FALSE){
            stop("the parameter lag_method must be one of 'mean' or 'median'")
        }
    }

    if (verbose & standardize){
        print("Standardizing the data (set parameter to FALSE to avoid this step)")
    }


    if(data_class == "list"){ # if we have to deal with a raster dataset
        elements <- input_raster_data(data,
                                        w = window,
                                        fun = lag_method,
                                        standardize = standardize
                                        )
        data <- elements$data
        wdata <- elements$wdata
        missing <- elements$missing
        rst <- elements$rst

    }else if(data_class == "data.frame"){ # if we have to deal with a dataframe

        # standardize data if required
        if (standardize) {
            for (i in 1:ncol(data)) {
                data[, i] <- scale(data[, i])
            }
        }

        wdata <- calcLaggedData(data,nblistw,lag_method)
        data <- as.matrix(data)
        wdata <- as.matrix(wdata)
    }
    # calculating the final results
    results <- main_worker("SFCM", data = data, wdata = wdata,
                           nblistw = nblistw, k = k, m = m, alpha = alpha,
                           lag_method=lag_method, maxiter = maxiter, tol = tol,
                           standardize = standardize, verbose = verbose,
                           seed = seed, init = init)

    # if we are working with rasters, we should set some additional values
    if(data_class == "list"){
        results <- output_raster_data(results, missing, rst)
        results$window <- window
    }

    return(results)
}

