% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.geolmMan.R
\name{predict.geolmMan}
\alias{predict.geolmMan}
\title{Predict method for geostatistical models}
\usage{
\method{predict}{geolmMan}(object, newdata, nsim = 0, vop, vp, sp = TRUE, dmethod = "chol", ...)
}
\arguments{
\item{object}{An object produced by the \code{geolm} function.}

\item{newdata}{An optional data frame in which to look for the coordinates at which to predict. If omitted, the observed data locations are used.}

\item{nsim}{A non-negative integer indicating the number of realizations to sample at the specified coordinates using conditional simulation.}

\item{vop}{The cross-covariance matrix between the observed responses and the responses to predict.}

\item{vp}{The covariance matrix of the responses to predict.}

\item{sp}{A logical value indicating whether to object returned should be of class \code{\link[sp]{SpatialPointsDataFrame}} for easier plotting with the \code{sp} package.  Default is \code{TRUE}.}

\item{dmethod}{The method used to decompose the covariance matrix for conditional simulation.  Valid options are \code{"chol"}, \code{"eigen"}, and \code{"svd"}.  The default is \code{"chol"}.}

\item{...}{Currently unimplemented.}
}
\value{
If \code{sp = TRUE}, then a \code{SpatialPointDataFrame} from the \code{sp} package is returned, with components including the prediction coordinates, the predicted responses \code{pred}, the mean square prediction error (\code{mspe}), the root mean square prediction error (\code{rmspe}), and the conditional realizations, if application (\code{sim.1}, \code{sim.2}, ...).  If \code{sp = FALSE}, then a list of class \code{gearKrige} is returned, with components \code{pred}, \code{mspe}, \code{rmspe}, and \code{sim}, if relevant.
}
\description{
\code{predict} calculates the predicted values at specified locations.  The method can additionally provide the mean square prediction error (mspe) and perform conditional simulation.
}
\details{
The \code{newdata} data frame must include the relevant covariates for the prediction locations, where the covariates are specified on the right side of the \code{~} in \code{object$formula}.  \code{newdata} must also include the coordinates of the prediction locations, with these columns having the names provided in \code{object$coordnames}.
}
\examples{

# generate response
y = rnorm(10)
# generate coordinates
x1 = runif(10); x2 = runif(10)

# data frame for observed data
data = data.frame(y, x1, x2)
coords = cbind(x1, x2)
d = as.matrix(dist(coords))
psill = 2 # partial sill
r = 4 # range parameter
evar = .1 # error variance
fvar = .1 # add finescale variance
# one can't generally distinguish between evar and fvar, but
# this is done for illustration purposes

# manually specify a an expoential covariance model 
v = psill * exp(-d/r) + (evar + fvar) * diag(10)

cmod_man = cmod.man(v = v, evar = evar)

#' # geolm for universal kriging
gearmod_uk = geolm(y ~ x1 + x2, data = data,
                 coordnames = c("x1", "x2"),
                 cmod = cmod_man)

# newdata must have columns with prediction coordinates
# add 5 unsampled sites to sampled sites
newdata = data.frame(x1 = c(x1, runif(5)), x2 = c(x2, runif(5)))
newcoords = newdata[,c("x1", "x2")]
# create vop and vp using distances
dop = sp::spDists(as.matrix(coords), as.matrix(newcoords))
dp = as.matrix(dist(newcoords))

vop = psill * exp(-dop/r) + fvar * (dop == 0)
vp = psill * exp(-dp/r) + fvar * diag(nrow(newcoords))

# prediction for universal kriging, with conditional simulation, 
# using manual covariance matrices
pred_uk_man = predict(gearmod_uk, newdata, nsim = 2, vop = vop, vp = vp,
                      dmethod = "svd")

# do the same thing, but using the std covariance function

# prediction for universal kriging, with conditional simulation
cmod_std = cmod.std("exponential", psill = psill, r = r, evar = evar, fvar = fvar)
gearmod_uk2 = geolm(y ~ x1 + x2, data = data, coordnames = c("x1", "x2"), 
                    cmod = cmod_std)
pred_uk_std = predict(gearmod_uk2, newdata, nsim = 2, dmethod = "svd")

# compare results
range(pred_uk_man$pred - pred_uk_std$pred)
range(pred_uk_man$mspe - pred_uk_std$mspe)
}
\author{
Joshua French
}
