/*
   Copyright 2019 Marius Appel <marius.appel@uni-muenster.de>

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

#include "image_collection_ops.h"
#include <gdal_utils.h>
#include <thread>
#include <unordered_set>
#include "cube.h"

namespace gdalcubes {

void image_collection_ops::translate_cog(std::shared_ptr<gdalcubes::image_collection> in, std::string out_dir, uint16_t nthreads) {
    std::vector<image_collection::gdalrefs_row> gdalrefs = in->get_gdalrefs();

    if (!filesystem::exists(out_dir)) {
        filesystem::mkdir_recursive(out_dir);
    }

    if (!filesystem::is_directory(out_dir)) {
        throw std::string("ERROR in image_collection_ops::translate_cog(): output is not a directory.");
    }

    std::vector<std::thread> thrds;

    std::shared_ptr<progress> prg = config::instance()->get_default_progress_bar()->get();
    prg->set(0);  // explicitly set to zero to show progress bar immediately

    for (uint16_t it = 0; it < nthreads; ++it) {
        thrds.push_back(std::thread([it, nthreads, &out_dir, &gdalrefs, &prg]() {
            for (uint32_t i = it; i < gdalrefs.size(); i += nthreads) {
                prg->increment((double)1 / (double)gdalrefs.size());
                std::string descr = gdalrefs[i].descriptor;

                CPLStringList translate_args;
                translate_args.AddString("-of");
                translate_args.AddString("GTiff");

                translate_args.AddString("-co");
                translate_args.AddString("TILED=YES");

                translate_args.AddString("-co");
                translate_args.AddString("COPY_SRC_OVERVIEWS=YES");

                translate_args.AddString("-co");
                translate_args.AddString("COMPRESS=LZW");

                translate_args.AddString("-b");
                translate_args.AddString(std::to_string(gdalrefs[i].band_num).c_str());  // band_num is 1 based

                //                for (uint16_t i = 0; i < gdal_translate_args.size(); ++i) {
                //                    translate_args.AddString(gdal_translate_args[i].c_str());
                //                }

                GDALTranslateOptions* trans_options = GDALTranslateOptionsNew(translate_args.List(), NULL);
                if (trans_options == NULL) {
                    GCBS_WARN("Cannot create gdal_translate options.");
                    continue;
                }
                GDALDataset* dataset = (GDALDataset*)GDALOpen(descr.c_str(), GA_ReadOnly);
                if (!dataset) {
                    GCBS_WARN("Cannot open GDAL dataset '" + descr + "'.");
                    GDALTranslateOptionsFree(trans_options);
                    continue;
                }
                std::string outfile = filesystem::join(out_dir, std::to_string(gdalrefs[i].image_id) + "_" + std::to_string(gdalrefs[i].band_id) + ".tif");
                GDALDatasetH out = GDALTranslate(outfile.c_str(), (GDALDatasetH)dataset, trans_options, NULL);
                if (!out) {
                    GCBS_WARN("Cannot translate GDAL dataset '" + descr + "'.");
                    GDALClose((GDALDatasetH)dataset);
                    GDALTranslateOptionsFree(trans_options);
                }
                GDALClose((GDALDatasetH)dataset);
                GDALClose(out);
                GDALTranslateOptionsFree(trans_options);
            }
        }));
    }
    for (uint16_t it = 0; it < nthreads; ++it) {
        thrds[it].join();
    }
    prg->finalize();
}

void image_collection_ops::create_overviews(std::shared_ptr<image_collection> in, std::vector<int> levels, std::string resampling, uint16_t nthreads) {
    std::vector<image_collection::gdalrefs_row> gdalrefs = in->get_gdalrefs();

    std::unordered_set<std::string> done;
    std::mutex m;

    std::vector<std::thread> thrds;

    std::shared_ptr<progress> prg = config::instance()->get_default_progress_bar()->get();
    prg->set(0);  // explicitly set to zero to show progress bar immediately

    for (uint16_t it = 0; it < nthreads; ++it) {
        thrds.push_back(std::thread([it, nthreads, &done, &m, &gdalrefs, &resampling, &levels, &prg]() {
            for (uint32_t i = it; i < gdalrefs.size(); i += nthreads) {
                prg->increment((double)1 / (double)gdalrefs.size());
                std::string descr = gdalrefs[i].descriptor;
                m.lock();
                if (done.count(descr) > 0) {
                    m.unlock();
                    continue;
                }
                done.insert(descr);
                m.unlock();

                GDALDataset* dataset = (GDALDataset*)GDALOpen(descr.c_str(), GA_Update);
                if (!dataset) {
                    dataset = (GDALDataset*)GDALOpen(descr.c_str(), GA_ReadOnly);
                    if (!dataset) {
                        GCBS_WARN("Cannot open GDAL dataset '" + descr + "'.");
                        continue;
                    }
                }
                if (dataset->BuildOverviews(resampling.c_str(), levels.size(), levels.data(), 0, nullptr, NULL, nullptr) == CE_Failure) {
                    GCBS_WARN("Cannot build overviews for dataset '" + descr + "'.");
                }
                GDALClose((GDALDatasetH)dataset);
            }
        }));
    }
    for (uint16_t it = 0; it < nthreads; ++it) {
        thrds[it].join();
    }
    prg->finalize();
}

}  // namespace gdalcubes
