% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/basic_functions.R
\name{plot}
\alias{plot}
\alias{fsr_plot}
\alias{plot,pgeometry,missing-method}
\title{Graphically visualize \code{pgeometry} objects}
\usage{
fsr_plot(pgo, base_poly = NULL, add_base_poly = TRUE, low = "white", high = "black", 
         crs = NA, clip = FALSE, line_lwd = 1, region_lwd = 1, ...)

\S4method{plot}{pgeometry,missing}(x, y, ...)
}
\arguments{
\item{pgo}{A \code{pgeometry} object of any type.}

\item{base_poly}{An \code{sfg} object of the type \code{POLYGON} or \code{MULTIPOLYGON}. It can also be an \code{sfc} object with only one element of the type \code{POLYGON} or \code{MULTIPOLYGON}.}

\item{add_base_poly}{A Boolean value that indicates whether \code{base_poly} will added to the visualization.}

\item{low}{A character value that indicates the color for the lowest membership degree (i.e., 0). Default is \code{"white"}.}

\item{high}{A character value that indicates the color for the highest membership degree (i.e., 1). Default is \code{"black"}.}

\item{crs}{A numerical value that denotes the coordinate reference system (i.e., EPSG code) of the visualization. Default is \code{NA}.}

\item{clip}{A Boolean value that indicates whether the boundaries of the components must be clipped by the \code{sfg} object \code{base_poly} (if it is not \code{null}).}

\item{line_lwd}{A numeric value that specifies the line width of linear components.}

\item{region_lwd}{A numeric value that specifies the line width of the boundaries of polygonal components.}

\item{...}{<\code{\link[rlang:dyn-dots]{dynamic-dots}}> Optional parameters. They can be the same as the parameters of \code{geom_sf()} function from \code{ggplot2}.}

\item{x}{A \code{pgeometry} object of any type.}

\item{y}{Not applicable.}
}
\value{
A \code{ggplot} object.
}
\description{
The \code{fsr_plot()} function (and the S4 method \code{plot()}) plots a \code{pgeometry} object.
}
\details{
The \code{fsr_plot()} function uses a \code{ggplot2} package to built the resulting plot. It receives a \code{pgeometry} object as input (if it is empty, an empty graphics
in obtained).

The \code{low} and \code{high} parameters are the colors for the minimum and maximum limits of the membership degrees. The
default colors are \code{"white"} and \code{"black"}, respectively. Other colors can be given in the same way that colors are informed
to visualizations produced by the \code{ggplot2} package.

It is possible to clip the geometric format of the components by using the parameter \code{base_poly}. The boundaries of this object
can also be included in the visualization if the parameter \code{add_base_poly} is \code{TRUE}.

Since the returned value is a \code{ggplot} object, it can be further be customized (see examples below).
}
\examples{
library(sf)

pts <- rbind(c(0, 2), c(4, 2))
# Point components
pcp1 <- create_component(st_multipoint(pts), 0.3)
pcp2 <- create_component("MULTIPOINT((2 2), (2 4), (2 0))", 0.5)
pcp3 <- create_component("MULTIPOINT((1 1), (3 1), (1 3), (3 3))", 0.9)
# Line components
lcp1 <- create_component("LINESTRING(0 0, 1 1.5)", 0.2)
lcp2 <- create_component("LINESTRING(1 3, 1 2, 2 0.5)", 0.5)
lcp3 <- create_component("LINESTRING(2 1.2, 3 1.6, 4 4)", 0.7)
lcp4 <- create_component("LINESTRING(1 1.5, 2 1.2)", 1.0)
# Polygon components
rcp1 <- create_component("POLYGON((0 0, 1 4, 2 2, 0 0))", 0.4)
rcp2 <- create_component("POLYGON((2 0.5, 4 1, 4 0, 2 0.5))", 0.8)
# Creating spatial plateau objects
pp <- create_pgeometry(list(pcp1, pcp2, pcp3), "PLATEAUPOINT")
pl <- create_pgeometry(list(lcp1, lcp3, lcp4), "PLATEAULINE")
pr <- create_pgeometry(list(rcp1, rcp2), "PLATEAUREGION")
pcm <- create_pgeometry(list(pcp1, pcp2, lcp1, lcp2, lcp3, rcp2), "PLATEAUCOMPOSITION")
pcl <- create_pgeometry(list(pp, pr, pcm), "PLATEAUCOLLECTION")

# Displaying their textual representations
pp
pl
pr
pcm
pcl

# Plotting them
plot(pp)
plot(pl)
plot(pr)
plot(pcm)
plot(pcl)
\dontrun{
# Custom colors
fsr_plot(pr, low = "green", high = "blue")

# Changing the line width of line components
fsr_plot(pl, line_lwd = 2)

# Changing the line width of boundary lines of region components
fsr_plot(pr, region_lwd = 2)

# Changing the line width of boundary lines of region components and its color
fsr_plot(pr, region_lwd = 2, color = "blue")

# You can customize the whole visualization using ggplot
library(ggplot2)

fsr_plot(pp, size = 5) +   
  theme(legend.position = "none") +
  theme(text=element_text(size=20, family = "serif", color = "black"),
        axis.text=element_text(color="black")) +
  scale_x_continuous(breaks = c(0, 1, 2, 3, 4)) +
  scale_y_continuous(breaks = c(0, 1, 2, 3, 4))
}
}
\references{
\href{https://onlinelibrary.wiley.com/doi/10.1111/tgis.13044}{Carniel, A. C.; Venâncio, P. V. A. B; Schneider, M. fsr: An R package for fuzzy spatial data handling. Transactions in GIS, vol. 27, no. 3, pp. 900-927, 2023.}
}
