% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/basic_functions.R
\name{create_pgeometry}
\alias{create_pgeometry}
\title{Create a \code{pgeometry} object with components}
\usage{
create_pgeometry(x, type, is_valid = TRUE)
}
\arguments{
\item{x}{A list of \code{component} objects, a list of \code{pgeometry} objects or a \code{data.frame}/\code{tibble} object. For \code{PLATEAUPOINT}, \code{PLATEAULINE} and \code{PLATEAUREGION}, the type of each component must be the same for all components.}

\item{type}{A character value that indicates the type of the desired \code{pgeometry} object.
It should be either \code{"PLATEAUPOINT"}, \code{"PLATEAULINE"}, \code{"PLATEAUREGION"}, \code{"PLATEAUCOMPOSITION"}, or \code{"PLATEAUCOLLECTION"}.
It must be compatible with the components given in \code{x} parameter.}

\item{is_valid}{A Boolean value to check whether the user wants to validate the created spatial plateau object at the end. If \code{is_valid = TRUE}, it calls \code{validObject()} method.}
}
\value{
A \code{pgeometry} object.
}
\description{
\code{create_pgeometry()} creates a \code{pgeometry} object from a \code{data.frame} or \code{tibble} object, a list of components, or a list of spatial plateau objects.
}
\details{
\code{create_pgeometry()} is a flexible function that creates a \code{pgeometry} object by using the values given in \code{x}.
This object is built by using either a list of \code{component} objects, a list of \code{pgeometry} objects or a \code{data.frame} (or \code{tibble}) object.
If a \code{data.frame} or \code{tibble} object is given as input, its columns must have the following format: (i) first column is an \code{sfc} object, and
(ii) the second columns consists of the membership degree of each respective object of the \code{sfc} column.

By default, this function checks if the resulting spatial plateau object is valid.
That is, it checks whether all constraints defined by the Spatial Plateau Algebra are satisfied.
For instance, the components of a plateau point, plateau line, or plateau region must be adjacent or disjoint from each other and have to be unique membership degrees.

If you are sure that the component objects provided to this function satisfy all the constraints, then you can use \code{is_valid = FALSE} to improve the performance of this function.
}
\examples{
library(sf)

# Creating some components
pts <- rbind(c(0, 2), c(4, 2))
# Point components
pcp1 <- create_component(st_multipoint(pts), 0.3)
pcp2 <- create_component("MULTIPOINT((2 2), (2 4), (2 0))", 0.5)
pcp3 <- create_component("MULTIPOINT((1 1), (3 1), (1 3), (3 3))", 0.9)
# Line components
lcp1 <- create_component("LINESTRING(0 0, 1 1.5)", 0.2)
lcp2 <- create_component("LINESTRING(1 3, 1 2, 2 0.5)", 0.5)
lcp3 <- create_component("LINESTRING(2 1.2, 3 1.6, 4 4)", 0.7)
lcp4 <- create_component("LINESTRING(1 1.5, 2 1.2)", 1.0)
# Polygon components
rcp1 <- create_component("POLYGON((0 0, 1 4, 2 2, 0 0))", 0.4)
rcp2 <- create_component("POLYGON((2 0.5, 4 1, 4 0, 2 0.5))", 0.8)

# Creating spatial plateau objects from lists of components
pp <- create_pgeometry(list(pcp1, pcp2, pcp3), "PLATEAUPOINT")
pl <- create_pgeometry(list(lcp1, lcp3, lcp4), "PLATEAULINE")
pr <- create_pgeometry(list(rcp1, rcp2), "PLATEAUREGION")
pcm <- create_pgeometry(list(pcp1, pcp2, lcp1, lcp2, lcp3, rcp2), "PLATEAUCOMPOSITION")

# Creating a spatial plateau objects from a list of spatial plateau objects
pcl <- create_pgeometry(list(pp, pr, pcm), "PLATEAUCOLLECTION")

# Converting pp into a tibble
pp
tibble_pp <- as_tibble(pp)
tibble_pp

# Creating a spatial plateau point from the previous tibble
equivalent_pp <- create_pgeometry(tibble_pp, "PLATEAUPOINT")
equivalent_pp
}
\references{
\href{https://onlinelibrary.wiley.com/doi/10.1111/tgis.13044}{Carniel, A. C.; Venâncio, P. V. A. B; Schneider, M. fsr: An R package for fuzzy spatial data handling. Transactions in GIS, vol. 27, no. 3, pp. 900-927, 2023.}

Underlying concepts and formal definitions of spatial plateau data types are explained in detail in:
\itemize{
\item \href{https://ieeexplore.ieee.org/document/8491565}{Carniel, A. C.; Schneider, M. Spatial Plateau Algebra: An Executable Type System for Fuzzy Spatial Data Types. In Proceedings of the 2018 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2018), pp. 1-8, 2018.}
\item \href{https://ieeexplore.ieee.org/document/9177620}{Carniel, A. C.; Schneider, M. Spatial Data Types for Heterogeneously Structured Fuzzy Spatial Collections and Compositions. In Proceedings of the 2020 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2020), pp. 1-8, 2020.}
}
}
