#' Set/Get interpolation to SpotRateCurve
#'
#' Sets and gets interpolation method to the SpotRateCurve.
#'
#' @param x a SpotRateCurve object.
#' @param value a Interpolation object.
#' @param ... additional arguments. Currently unused.
#'
#' @return A Interpolatin object.
#' @aliases
#' interpolation,SpotRateCurve-method
#' interpolation<-,SpotRateCurve,Interpolation-method
#' interpolation<-,SpotRateCurve,NULL-method
#' @examples
#' terms <- c(1, 11, 26, 27, 28)
#' rates <- c(0.0719, 0.056, 0.0674, 0.0687, 0.07)
#' curve <- spotratecurve(rates, terms, "discrete", "actual/365", "actual")
#' interpolation(curve) <- interp_flatforward()
#' interpolation(curve)
#' @export
setGeneric(
  "interpolation",
  function(x, ...) {
    standardGeneric("interpolation")
  }
)

#' @rdname interpolation
#' @export
setGeneric(
  "interpolation<-",
  function(x, value) {
    standardGeneric("interpolation<-")
  }
)

#' Create the interpolation function
#'
#' Creates the interpolation function to a SpotRateCurve object.
#'
#' @param object a Interpolation object.
#' @param x a SpotRateCurve object.
#' @param ... additional arguments. Currently unused.
#' 
#' This method is used internally when the interpolation is set to a curve.
#' It uses the current state of the curve to build the interpolation function.
#' This is similar to call `approxfun` and `splinefun` to create functions that
#' perform interpolation of the given data points.
#' 
#' This method shouldn't be directly called, it is for internal use only.
#'
#' @return
#' A `Interpolation` object with the slot `func` properly defined.
#' This slot is set with a `function` (closure) that executes
#' the interpolation method.
#' 
#' @aliases
#' prepare_interpolation,FlatForward,SpotRateCurve-method
#' prepare_interpolation,HermiteSpline,SpotRateCurve-method
#' prepare_interpolation,Linear,SpotRateCurve-method
#' prepare_interpolation,LogLinear,SpotRateCurve-method
#' prepare_interpolation,MonotoneSpline,SpotRateCurve-method
#' prepare_interpolation,NaturalSpline,SpotRateCurve-method
#' prepare_interpolation,NelsonSiegel,SpotRateCurve-method
#' prepare_interpolation,NelsonSiegelSvensson,SpotRateCurve-method
#' @examples
#' terms <- c(1, 11, 26, 27, 28)
#' rates <- c(0.0719, 0.056, 0.0674, 0.0687, 0.07)
#' curve <- spotratecurve(rates, terms, "discrete", "actual/365", "actual")
#' prepare_interpolation(interp_flatforward(), curve)
#' @export
setGeneric(
  "prepare_interpolation",
  function(object, x, ...) {
    standardGeneric("prepare_interpolation")
  }
)

#' Fit parametric interpolation functions
#'
#' Fits parametric interpolation functions like [NelsonSiegel-class] or
#' [NelsonSiegelSvensson-class].
#'
#' @param object a Interpolation object with initial parameters set.
#' @param x a SpotRateCurve object.
#' @param ... additional arguments. Currently unused.
#' 
#' @return A `Interpolation` object.
#' @aliases
#' fit_interpolation,NelsonSiegel,SpotRateCurve-method
#' fit_interpolation,NelsonSiegelSvensson,SpotRateCurve-method
#' @examples
#' terms <- c(1, 11, 26, 27, 28)
#' rates <- c(0.0719, 0.056, 0.0674, 0.0687, 0.07)
#' curve <- spotratecurve(rates, terms, "discrete", "actual/365", "actual")
#' fit_interpolation(interp_nelsonsiegel(0.1, 0.01, 0.01, 0.01), curve)
#' @export
setGeneric(
  "fit_interpolation",
  function(object, x, ...) {
    standardGeneric("fit_interpolation")
  }
)

setMethod(
  "interpolation",
  signature(x = "SpotRateCurve"),
  function(x) {
    x@interpolation
  }
)

setReplaceMethod(
  "interpolation",
  signature(x = "SpotRateCurve", value = "Interpolation"),
  function(x, value) {
    x@interpolation <- prepare_interpolation(value, x)
    x
  }
)

setReplaceMethod(
  "interpolation",
  signature(x = "SpotRateCurve", value = "NULL"),
  function(x, value) {
    x@interpolation <- value
    x
  }
)

setMethod(
  "prepare_interpolation",
  signature(object = "FlatForward", x = "SpotRateCurve"),
  function(object, x, ...) {
    terms <- as.numeric(x@terms)
    prices <- compound(x)
    interp_coords <- xy.coords(terms, log(prices))
    interp_fun <- approxfun(interp_coords, method = "linear")
    dc <- x@daycount
    comp <- x@compounding
    object@func <- function(term) {
      log.price <- interp_fun(term)
      price <- exp(log.price)
      rates(comp, toyears(dc, term, "days"), price)
    }
    object
  }
)

setMethod(
  "prepare_interpolation",
  signature(object = "Linear", x = "SpotRateCurve"),
  function(object, x, ...) {
    interp_coords <- xy.coords(as.numeric(x@terms), as.numeric(x))
    interp_fun <- approxfun(interp_coords, method = "linear")
    object@func <- function(term) interp_fun(term)
    object
  }
)

#' @export
setMethod(
  "prepare_interpolation",
  signature(object = "LogLinear", x = "SpotRateCurve"),
  function(object, x, ...) {
    interp_coords <- xy.coords(as.numeric(x@terms), log(as.numeric(x)))
    interp_fun <- approxfun(interp_coords, method = "linear")
    object@func <- function(term) exp(interp_fun(term))
    object
  }
)

#' @export
setMethod(
  "prepare_interpolation",
  signature(object = "NaturalSpline", x = "SpotRateCurve"),
  function(object, x, ...) {
    interp_coords <- xy.coords(as.numeric(x@terms), as.numeric(x))
    interp_fun <- splinefun(interp_coords, method = "natural")
    object@func <- function(term) interp_fun(term)
    object
  }
)

#' @export
setMethod(
  "prepare_interpolation",
  signature(object = "HermiteSpline", x = "SpotRateCurve"),
  function(object, x, ...) {
    interp_coords <- xy.coords(as.numeric(x@terms), as.numeric(x))
    interp_fun <- splinefun(interp_coords, method = "monoH.FC")
    object@func <- function(term) interp_fun(term)
    object
  }
)

#' @export
setMethod(
  "prepare_interpolation",
  signature(object = "MonotoneSpline", x = "SpotRateCurve"),
  function(object, x, ...) {
    interp_coords <- xy.coords(as.numeric(x@terms), as.numeric(x))
    interp_fun <- splinefun(interp_coords, method = "hyman")
    object@func <- function(term) interp_fun(term)
    object
  }
)

ns <- function(t, b1, b2, b3, l1) {
  b1 +
    b2 * (1 - exp(-l1 * t)) / (l1 * t) +
    b3 * ((1 - exp(-l1 * t)) / (l1 * t) - exp(-l1 * t))
}

#' @export
setMethod(
  "prepare_interpolation",
  signature(object = "NelsonSiegel", x = "SpotRateCurve"),
  function(object, x, ...) {
    object@func <- function(term) {
      ns(term, object@beta1, object@beta2, object@beta3, object@lambda1)
    }
    object
  }
)

nss <- function(t, b1, b2, b3, b4, l1, l2) {
  ns(t, b1, b2, b3, l1) + b4 * ((1 - exp(-l2 * t)) / (l2 * t) - exp(-l2 * t))
}

#' @export
setMethod(
  "prepare_interpolation",
  signature(object = "NelsonSiegelSvensson", x = "SpotRateCurve"),
  function(object, x, ...) {
    object@func <- function(term) {
      nss(
        term, object@beta1, object@beta2, object@beta3, object@beta4,
        object@lambda1, object@lambda2
      )
    }
    object
  }
)

setMethod(
  "fit_interpolation",
  signature(object = "NelsonSiegel", x = "SpotRateCurve"),
  function(object, x, ...) {
    par <- parameters(object)
    res <- optim(par, function(par, x) {
      interpolation(x) <- do.call(interp_nelsonsiegel, as.list(par))
      interpolation_error(x)
    }, method = "BFGS", x = x)
    do.call(interp_nelsonsiegel, as.list(res$par))
  }
)

setMethod(
  "fit_interpolation",
  signature(object = "NelsonSiegelSvensson", x = "SpotRateCurve"),
  function(object, x, ...) {
    par <- parameters(object)
    res <- optim(par, function(par, x) {
      interpolation(x) <- do.call(interp_nelsonsiegelsvensson, as.list(par))
      interpolation_error(x)
    }, method = "BFGS", x = x)
    do.call(interp_nelsonsiegelsvensson, as.list(res$par))
  }
)
