#' @title Univariate kernel density estimate plot of VBGF parameter from
#' bootstrapping results
#'
#' @description This function plots a set of vertical plots with kernel density
#' distributions for univariate posterior distributions of the VBGF growth
#' parameters \eqn{L_{inf}}, \eqn{K}, and \eqn{Phi’}. The 95% confidence
#' interval and the most likely optimum fit estimate are shown for each parameter.
#'
#' @param res Object of class \code{lfqBoot}.
#' @param CI \code{numeric}. Confidence interval in \% (default: 95).
#' @param use_hist \code{logical} Plot histogram in addition to smoothed kernel
#' density.
#' @param nbreaks \code{numeric} vector specifying the number of breaks in the
#' histogram.
#' @param mar,oma,mgp,tcl,cex,... Additional arguments passed to
#' \link[graphics]{par}.
#'
#' @details
#' This function used the function \link[ks]{kde} to obtain kernel density
#' estimates for the VBGF growth parameters \eqn{L_{inf}}, \eqn{K}, and
#' \eqn{Phi’}. The 95 % confidence interval (i.e., the 95% quantiles for each
#' posterior distribution) and the most likely optimum fit estimate (i.e., the
#' mode of each posterior distribution) are then plotted inside each vertical
#' plot. The input used for plotting is usually the result of a bootstrapped
#' growth analysis (i.e. a \code{lfqBoot} object generated by \strong{fishboot}
#' functions such as \link{ELEFAN_SA_boot}, \link{ELEFAN_GA_boot},
#' \link{grotag_boot}, or \link{grolenage_boot}).
#'
#' @return This function returns just the described plot.
#'
#' @export
#'
#' @examples
#' data(alba_boot)
#' univariate_density(alba_boot)
univariate_density <- function(res, CI = 95, use_hist = FALSE, nbreaks = 10,
                               mar = c(1.5, 2, 2, 0), oma = c(1.5, 0, 0, 0.5),
                               mgp = c(2, 0.5, 0), tcl = -0.25, cex = 1,
                               ...){

  # Extract bootstrap results
  res <- res$bootRaw

  # Catch original par settings and restore them at the end
  op <- par(no.readonly = TRUE)
  on.exit(par(op))

  par(mfcol = c(1, ncol(res)), mar = mar, oma = oma, mgp = mgp,
      tcl = tcl, cex = cex, ...)

  VARS <- list(Linf     = expression(italic(L)[infinity]),
               K        = expression(italic(K)),
               t_anchor = expression(italic(t)[anchor]),
               C        = expression(italic(C)),
               ts       = expression(italic(t)[s]),
               phiL     = expression(paste(phi, "'")))

  # univariate plots
  for(i in seq(ncol(res))){
    x <- kde(x = res[,i])

    h <- hist(x = res[,i], plot = FALSE, breaks = nbreaks)

    xlim <- c(0, max(x$estimate))

    if(use_hist) xlim <- range(c(xlim, max(h$density)))

    xlim <- xlim * c(0,1.1)

    plot(x = x$estimate, y = x$eval.points, type = "n",
         xlim = xlim, xaxs = "i", ylab = NA, xlab = NA, col = 1, lty = 1)

    usr <- par()$usr

    CItxt <- sprintf("%1.0f%%", 100 - CI)
    inCI <- rle(x = x$estimate > x$cont[CItxt])

    start.idx <- c(1, cumsum(inCI$lengths[-length(inCI$lengths)])+1)
    end.idx <- cumsum(inCI$lengths)

    index <- seq(from = start.idx[which.min(inCI$values)],
                 to = end.idx[which.max(inCI$values)])
    limCI <- range(x$eval.points[index])

    in1 <- which(x$estimate > x$cont["99%"])
    mean1 <- mean(x$eval.points[in1])

    if(isTRUE(use_hist)){
      rect(xleft = 0, ybottom = h$breaks[-length(h$breaks)],
           xright = h$density, ytop = h$breaks[-1],
           col = "grey90", border = "grey50")
    }else{
      for(j in seq(inCI$lengths)){
        if(inCI$values[j]){
          polygon(y = c(x$eval.points[start.idx[j]:end.idx[j]],
                        rev(x$eval.points[start.idx[j]:end.idx[j]])),
                  x = c(x$estimate[start.idx[j]:end.idx[j]],
                        rep(0, length(x$estimate[start.idx[j]:end.idx[j]]))),
                  col = "grey90", border = NA, lty = 3, lwd = 1)
        }
      }
    }

    lines(x = x$estimate, y = x$eval.points, lwd = 1, col = "grey50")

    # rug
    segments(x0 = 0, x1 = par()$cxy[1]*0.3, y0 = x$x, y1 = x$x,
             col = rgb(0, 0, 0, 0.5), lwd = 0.3)

    # range of CI
    abline(h = limCI, lty = 1, lwd=1, col = 1)
    text(y = c(limCI), x = mean(usr[1:2]), labels = sprintf("%.2f", limCI),
         pos = c(1,3), offset = 0.25, col = 1)

    abline(h = mean1, lty = 1, lwd = 1, col = 1)

    text(x = mean(usr[1:2]), y = mean1, labels = sprintf("%.2f", mean1),
         pos = 3, offset = 0.25, col = 1)

    varlab <- VARS[[match(names(res)[i], names(VARS))]]

    mtext(text = varlab, side = 3, line = 0.25)

    box()
  }

  mtext("Density", side = 1, line = 0, outer = TRUE)

  invisible()
}
