% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fetwfe.R
\name{simulateDataCore}
\alias{simulateDataCore}
\title{Generate Random Panel Data for FETWFE Simulations}
\usage{
simulateDataCore(
  N,
  T,
  R,
  d,
  sig_eps_sq,
  sig_eps_c_sq,
  beta,
  seed = NULL,
  gen_ints = FALSE,
  distribution = "gaussian"
)
}
\arguments{
\item{N}{Integer. Number of units in the panel.}

\item{T}{Integer. Number of time periods.}

\item{R}{Integer. Number of treated cohorts (with treatment starting in periods 2 to T).}

\item{d}{Integer. Number of time-invariant covariates.}

\item{sig_eps_sq}{Numeric. Variance of the idiosyncratic (observation-level) noise.}

\item{sig_eps_c_sq}{Numeric. Variance of the unit-level random effects.}

\item{beta}{Numeric vector. Coefficient vector for data generation. Its required length depends
on the value of \code{gen_ints}:
\itemize{
\item If \code{gen_ints = TRUE} and \code{d > 0}, the expected length is
\eqn{p = R + (T-1) + d + dR + d(T-1) + num\_treats + num\_treats \times d}, where
\eqn{num\_treats = T \times R - \frac{R(R+1)}{2}}.
\item If \code{gen_ints = TRUE} and \code{d = 0}, the expected length is
\eqn{p = R + (T-1) + num\_treats}.
\item If \code{gen_ints = FALSE}, the expected length is
\eqn{p = R + (T-1) + d + num\_treats}.
}}

\item{seed}{(Optional) Integer. Seed for reproducibility.}

\item{gen_ints}{Logical. If \code{TRUE}, generate the full design matrix with interactions;
if \code{FALSE} (the default), generate a design matrix without any interaction terms.}

\item{distribution}{Character. Distribution to generate covariates.
Defaults to \code{"gaussian"}. If set to \code{"uniform"}, covariates are drawn uniformly
from \eqn{[-\sqrt{3}, \sqrt{3}]}.}
}
\value{
An object of class \code{"FETWFE_simulated"}, which is a list containing:
\describe{
\item{pdata}{A dataframe containing generated data that can be passed to \code{fetwfe()}.}
\item{X}{The design matrix. When \code{gen_ints = TRUE}, \eqn{X} has \eqn{p} columns with
interactions; when \code{gen_ints = FALSE}, \eqn{X} has no interactions.}
\item{y}{A numeric vector of length \eqn{N \times T} containing the generated responses.}
\item{covs}{A character vector containing the names of the generated features (if \eqn{d > 0}),
or simply an empty vector (if \eqn{d = 0})}
\item{time_var}{The name of the time variable in pdata}
\item{unit_var}{The name of the unit variable in pdata}
\item{treatment}{The name of the treatment variable in pdata}
\item{response}{The name of the response variable in pdata}
\item{coefs}{The coefficient vector \eqn{\beta} used for data generation.}
\item{first_inds}{A vector of indices indicating the first treatment effect for each treated
cohort.}
\item{N_UNTREATED}{The number of never-treated units.}
\item{assignments}{A vector of counts (of length \eqn{R+1}) indicating how many units fall into
the never-treated group and each of the \eqn{R} treated cohorts.}
\item{indep_counts}{Independent cohort assignments (for auxiliary purposes).}
\item{p}{The number of columns in the design matrix \eqn{X}.}
\item{N}{Number of units.}
\item{T}{Number of time periods.}
\item{R}{Number of treated cohorts.}
\item{d}{Number of covariates.}
\item{sig_eps_sq}{The idiosyncratic noise variance.}
\item{sig_eps_c_sq}{The unit-level noise variance.}
}
}
\description{
Generates a random panel data set for simulation studies of the fused extended two-way fixed
effects (FETWFE) estimator. The function creates a balanced panel with \eqn{N} units over \eqn{T}
time periods, assigns treatment status across \eqn{R} treated cohorts (with equal marginal
probabilities for treatment and non-treatment), and constructs a design matrix along with the
corresponding outcome. When \code{gen_ints = TRUE} the full design matrix is returned (including
interactions between covariates and fixed effects and treatment indicators). When
\code{gen_ints = FALSE} the design matrix is generated in a simpler format (with no interactions)
as expected by \code{fetwfe()}. Moreover, the covariates are generated according to the
specified \code{distribution}: by default, covariates are drawn from a normal distribution;
if \code{distribution = "uniform"}, they are drawn uniformly from \eqn{[-\sqrt{3}, \sqrt{3}]}.

When \eqn{d = 0} (i.e. no covariates), no covariate-related columns or interactions are
generated.

See the simulation studies section of Faletto (2025) for details.
}
\details{
When \code{gen_ints = TRUE}, the function constructs the design matrix by first generating
base fixed effects and a long-format covariate matrix (via \code{generateBaseEffects()}), then
appending interactions between the covariates and cohort/time fixed effects (via
\code{generateFEInts()}) and finally treatment indicator columns and treatment-covariate
interactions (via \code{genTreatVarsSim()} and \code{genTreatInts()}). When
\code{gen_ints = FALSE}, the design matrix consists only of the base fixed effects, covariates,
and treatment indicators.

The argument \code{distribution} controls the generation of covariates. For
\code{"gaussian"}, covariates are drawn from \code{rnorm}; for \code{"uniform"},
they are drawn from \code{runif} on the interval \eqn{[-\sqrt{3}, \sqrt{3}]}.

When \eqn{d = 0} (i.e. no covariates), the function omits any covariate-related columns
and their interactions.
}
\examples{
\dontrun{
  # Set simulation parameters
  N <- 100           # Number of units in the panel
  T <- 5             # Number of time periods
  R <- 3             # Number of treated cohorts
  d <- 2             # Number of time-invariant covariates
  sig_eps_sq <- 1    # Variance of observation-level noise
  sig_eps_c_sq <- 0.5  # Variance of unit-level random effects

  # Generate coefficient vector using genCoefsCore()
  # (Here, density controls sparsity and eff_size scales nonzero entries)
  coefs_core <- genCoefsCore(R = R, T = T, d = d, density = 0.2, eff_size = 2, seed = 123)

  # Now simulate the data. Setting gen_ints = TRUE generates the full design
  matrix with interactions.
  sim_data <- simulateDataCore(
    N = N,
    T = T,
    R = R,
    d = d,
    sig_eps_sq = sig_eps_sq,
    sig_eps_c_sq = sig_eps_c_sq,
    beta = coefs_core$beta,
    seed = 456,
    gen_ints = TRUE,
    distribution = "gaussian"
  )

  # Examine the returned list:
  str(sim_data)
}

}
\references{
Faletto, G (2025). Fused Extended Two-Way Fixed Effects for
Difference-in-Differences with Staggered Adoptions.
\emph{arXiv preprint arXiv:2312.05985}.
\url{https://arxiv.org/abs/2312.05985}.
}
