\name{landmarkreg}
\alias{landmarkreg}
\title{
  Landmark Registration of Functional Observations
}
\description{
It is common to see that among a set of functions certain prominent
features such peaks and valleys, called $landmarks$, do not occur
at the same times, or other argument values.  
This is called $phase variation$, and it can be essential to align 
these features before proceeding with further functional data analyses.  

This function uses the timings of these
features to align or register the curves.  The registration involves
estimating a nonlinear transformation of the argument continuum for each
functional observation.  This transformation is called a warping
function. It must be strictly increasing and smooth.

Warning:  As of March 2022, landmark registration cannot be done using
function \code{smooth.basis} instead of function \code{smooth.morph}.  The 
warping function must be strictly monotonic, and we have found that using 
\code{smooth.basis} too often violates this contraint.  Function 
\code{smooth.morph} ensures monotonicity.
}
\usage{
landmarkreg(unregfd, ximarks, x0marks=xmeanmarks,
            WfdPar=NULL, ylambda=1e-10)
}
\arguments{
  \item{unregfd}{
    a functional data object containing the curves to be registered.
  }
  \item{ximarks}{
    A matrix containing the timings or argument values associated with
    the landmarks for the observations in \code{fd} to be registered.
    The number of rows N equals the number of observations, and the
    number of columns NL equals the number of landmarks. These landmark
    times must be in the interior of the interval over which the
    functions are defined.
    Object ximarks may also be a vector or a data.frame.
  }
  \item{x0marks}{
    a vector of length NL of times of landmarks for target curve.  If
    not supplied, the mean of the landmark times in \code{ximarks} is
    used.
    Object x0marks may also be a vector.
  }
  \item{WfdPar}{
    a functional parameter object defining the warping functions that
    transform time in order to register the curves.
  }
  \item{ylambda}{
    Smoothing parameter controlling the smoothness of the registered
    functions.  It can happen with high dimensional bases that local
    wiggles can appear in the registered curves or their derivatives
    that are not seen in the unregistered versions.  In this case,
    this parameter should be increased to the point where they
    disappear.
  }
}
\value{
  a named list of length 4 with components:

  \item{fdreg}{
    a functional data object for the registered curves.
  }
  \item{warpfd}{
    a functional data object for the warping functions.
  }
  \item{warpinvfd}{
    a functional data object for the inverse warping functions.
  }
  \item{Wfd}{
    a functional data object for the core function defining
    the strictly monotone warping function.
  }
}
\details{
  A value of an arbitrary strictly monotone function at a point \code{x}
  can be defined as the indefinite integral from a fixed lower boundary 
  to \code{x} of the exponential of an unconstrained function value 
  \code{W(x)}.
  
  We use B-spline basis functions to define function \code{W}, and optimize
  the coefficients of its B-spline expansion with respect to the
  data we are fitting.  The optimized core function \code{W} is output 
  along with the registered functions, the warping function qnd the inverse
  warping function.
  
  It is essential that the location of every landmark be clearly defined
  in each of the curves as well as the template function.  If this is
  not the case, consider using the continuous registration function
  \code{register.fd}.  
  
  Not much curvature is usually required in the warping functions, so a
  rather lower power basis, usually B-splines, is suitable for defining
  the functional parameter argument \code{WfdPar}.  A registration with
  a few prominent landmarks is often a good preliminary to using the
  more sophisticated but more lengthy process in \code{register.fd}.
}
\seealso{
  \code{\link{register.fd}},
  \code{\link{landmarkreg2}},
  \code{\link{smooth.morph}}
}
\examples{
#See the analysis for the lip data in the examples.

## setting parameters
library(lattice)
data(landmark.reg.expData) ## containing an simple object called sampleData

# Preferred:
# eps <- .Machine$double.eps
# to reduce compute time:
eps <- 1000*.Machine$double.eps
from <- -1.0187
to   <-  9.4551
# Preferred:
# nb <- 201
# to reduce compute time:
nb <- 31
nbreaks <- 11
## assign landmarks
landmark <- matrix(c(0.4999, 0.657, 0.8141, 0.5523, 0.5523,
                     3.3279, 3.066, 3.0137, 3.2231, 3.2231),
                   ncol=2)
wbasis <- create.bspline.basis(rangeval=c(from, to),
	                       norder=4, breaks=seq(from, to, len=nbreaks))
Wfd0   <- fd(matrix(0,wbasis$nbasis,1),wbasis)
WfdPar <- fdPar(Wfd0, 1, 1e-4)
## get the density of the data
x <- split(sampleData, factor(sampleData$which))
# to save time, reduce the number of curves from 5 to 3
k <- 3
densY <- sapply(x[1:k], function(z){
            r <- range(z[, 1])
            z <- z[, 1]
            z <- z[z>r[1]+eps & z<r[2]-eps]
            density(z, from=from, to=to, n=nb, na.rm=TRUE)$y
        })

argvals <- seq(from, to, len=nb)
fdobj   <- smooth.basis(argvals, densY, wbasis,
                   fdnames = c("x", "samples", "density"))$fd
#  the target landmark values, x0marks, here default to
#  the row-wise mean of matrix landmark
regDens   <- landmarkreg(fdobj, landmark[1:k,], WfdPar=WfdPar)

warpfdobj <- regDens$warpfd
warpedX   <- as.matrix(eval.fd(warpfdobj, argvals))
matplot(argvals, warpedX, type="l")
funs      <- apply(warpedX, 2, approxfun, argvals)
}
\keyword{smooth}
