% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/explain.R
\name{explain}
\alias{explain}
\alias{explain.default}
\alias{explain.lm}
\alias{explain.xgb.Booster}
\alias{explain.lgb.Booster}
\title{Fast approximate Shapley values}
\usage{
explain(object, ...)

\method{explain}{default}(
  object,
  feature_names = NULL,
  X = NULL,
  nsim = 1,
  pred_wrapper = NULL,
  newdata = NULL,
  adjust = FALSE,
  baseline = NULL,
  shap_only = TRUE,
  parallel = FALSE,
  ...
)

\method{explain}{lm}(
  object,
  feature_names = NULL,
  X,
  nsim = 1,
  pred_wrapper,
  newdata = NULL,
  adjust = FALSE,
  exact = FALSE,
  baseline = NULL,
  shap_only = TRUE,
  parallel = FALSE,
  ...
)

\method{explain}{xgb.Booster}(
  object,
  feature_names = NULL,
  X = NULL,
  nsim = 1,
  pred_wrapper,
  newdata = NULL,
  adjust = FALSE,
  exact = FALSE,
  baseline = NULL,
  shap_only = TRUE,
  parallel = FALSE,
  ...
)

\method{explain}{lgb.Booster}(
  object,
  feature_names = NULL,
  X = NULL,
  nsim = 1,
  pred_wrapper,
  newdata = NULL,
  adjust = FALSE,
  exact = FALSE,
  baseline = NULL,
  shap_only = TRUE,
  parallel = FALSE,
  ...
)
}
\arguments{
\item{object}{A fitted model object (e.g., a \code{\link[ranger:ranger]{ranger::ranger()}},
\code{\link[xgboost:xgb.train]{xgboost::xgboost()}}, or \code{\link[earth:earth]{earth::earth()}} object, to name
a few).}

\item{...}{Additional optional arguments to be passed on to
\code{\link[foreach:foreach]{foreach::foreach()}} whenever \code{parallel = TRUE}. For example, you may need
to supply additional packages that the parallel task depends on via the
\code{.packages} argument to \code{\link[foreach:foreach]{foreach::foreach()}}. \strong{NOTE:}
\code{\link[foreach:foreach]{foreach::foreach()}}'s \code{.combine} argument is already set internally by
\code{explain()}, so passing it via the \code{...} argument would likely result in an
error.}

\item{feature_names}{Character string giving the names of the predictor
variables (i.e., features) of interest. If \code{NULL} (default) they will be
taken from the column names of \code{X}.}

\item{X}{A matrix-like R object (e.g., a data frame or matrix) containing
ONLY the feature columns from the training data (or suitable background data
set). \strong{NOTE:} This argument is required whenever \code{exact = FALSE}.}

\item{nsim}{The number of Monte Carlo repetitions to use for estimating each
Shapley value (only used when \code{exact = FALSE}). Default is 1.
\strong{NOTE:} To obtain the most accurate results, \code{nsim} should be set
as large as feasibly possible.}

\item{pred_wrapper}{Prediction function that requires two arguments,
\code{object} and \code{newdata}. \strong{NOTE:} This argument is required
whenever \code{exact = FALSE}. The output of this function should be
determined according to:

\describe{
\item{Regression}{A numeric vector of predicted outcomes.}
\item{Binary classification}{A vector of predicted class probabilities
for the reference class.}
\item{Multiclass classification}{A vector of predicted class probabilities
for the reference class.}
}}

\item{newdata}{A matrix-like R object (e.g., a data frame or matrix)
containing ONLY the feature columns for the observation(s) of interest; that
is, the observation(s) you want to compute explanations for. Default is
\code{NULL} which will produce approximate Shapley values for all the rows in
\code{X} (i.e., the training data).}

\item{adjust}{Logical indicating whether or not to adjust the sum of the
estimated Shapley values to satisfy the \emph{local accuracy} property; that is,
to equal the difference between the model's prediction for that sample and
the average prediction over all the training data (i.e., \code{X}). Default is
\code{FALSE} and setting to \code{TRUE} requires \code{nsim} > 1.}

\item{baseline}{Numeric baseline to use when adjusting the computed Shapley
values to achieve \emph{local accuracy}. Adjusted Shapley values for a single
prediction (\code{fx}) will sum to the difference \code{fx - baseline}. Defaults to
\code{NULL}, which corresponds to the average predictions computed from \code{X}, and
zero otherwise (i.e., no additional predictions will be computed and the
baseline attribute of the output will be set to zero).}

\item{shap_only}{Logical indicating whether or not to include additional
output useful for plotting (i.e., \code{newdata} and the \code{baseline} value.). This
is convenient, for example, when using \code{\link[shapviz:shapviz]{shapviz::shapviz()}} for plotting.
Default is \code{TRUE}.}

\item{parallel}{Logical indicating whether or not to compute the approximate
Shapley values in parallel across features; default is \code{FALSE}. \strong{NOTE:}
setting \code{parallel = TRUE} requires setting up an appropriate (i.e.,
system-specific) \emph{parallel backend} as described in the
\href{https://cran.r-project.org/package=foreach}{foreach}; for details, see
\code{vignette("foreach", package = "foreach")} in R.}

\item{exact}{Logical indicating whether to compute exact Shapley values.
Currently only available for \code{\link[stats:lm]{stats::lm()}},
\code{\link[xgboost:xgb.train]{xgboost::xgboost()}}, and \code{\link[lightgbm:lightgbm]{lightgbm::lightgbm()}} objects.
Default is \code{FALSE}. Note that setting \code{exact = TRUE} will return
explanations for each of the \code{\link[stats:terms]{stats::terms()}} in an
\code{\link[stats:lm]{stats::lm()}} object. Default is \code{FALSE}.}
}
\value{
If \code{shap_only = TRUE} (the default), a matrix is returned with one
column for each feature specified in \code{feature_names} (if
\code{feature_names = NULL}, the default, there will
be one column for each feature in \code{X}) and one row for each observation
in \code{newdata} (if \code{newdata = NULL}, the default, there will be one
row for each observation in \code{X}). Additionally, the returned matrix will
have an attribute called \code{"baseline"} containing the baseline value. If
\code{shap_only = FALSE}, then a list is returned with three components:
\itemize{
\item \code{shapley_values} - a matrix of Shapley values (as described above);
\item \code{feature_values} - the corresponding feature values (for plotting with
\code{\link[shapviz:shapviz]{shapviz::shapviz()}});
\item \code{baseline} - the corresponding baseline value (for plotting with
\code{\link[shapviz:shapviz]{shapviz::shapviz()}}).
}
}
\description{
Compute fast (approximate) Shapley values for a set of features using the
Monte Carlo algorithm described in Strumbelj and Igor (2014). An efficient
algorithm for tree-based models, commonly referred to as Tree SHAP, is also
supported for \href{https://cran.r-project.org/package=lightgbm}{lightgbm} and
\href{https://cran.r-project.org/package=xgboost}{xgboost} models; see Lundberg
et. al. (2020) for details.
}
\note{
Setting \code{exact = TRUE} with a linear model (i.e., an
\code{\link[stats:lm]{stats::lm()}} or \code{\link[stats:glm]{stats::glm()}} object) assumes that the
input features are independent. Also, setting \code{adjust = TRUE} is
experimental and we follow the same approach as in
\href{https://github.com/slundberg/shap}{shap}.
}
\examples{
#
# A projection pursuit regression (PPR) example
#

# Load the sample data; see ?datasets::mtcars for details
data(mtcars)

# Fit a projection pursuit regression model
fit <- ppr(mpg ~ ., data = mtcars, nterms = 5)

# Prediction wrapper
pfun <- function(object, newdata) {  # needs to return a numeric vector
  predict(object, newdata = newdata)  
}

# Compute approximate Shapley values using 10 Monte Carlo simulations
set.seed(101)  # for reproducibility
shap <- explain(fit, X = subset(mtcars, select = -mpg), nsim = 10, 
                pred_wrapper = pfun)
head(shap)
}
\references{
Strumbelj, E., and Igor K. (2014). Explaining prediction models and
individual predictions with feature contributions. Knowledge and information
systems, 41(3), 647-665.

Lundberg, S. M., Erion, G., Chen, H., DeGrave, A., Prutkin, J. M., Nair, B.,
Katz, R., Himmelfarb, J., Bansal, N., and Lee, Su-In (2020). From local
explanations to global understanding with explainable AI for trees.
Nature Machine Intelligence, 2(1), 2522–5839.
}
\seealso{
You can find more examples (with larger and more realistic data
sets) on the \strong{fastshap} GitHub repository:
\url{https://github.com/bgreenwell/fastshap}.
}
