% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shiny.R
\name{ui_epoxy_markdown}
\alias{ui_epoxy_markdown}
\title{Epoxy Markdown Template for Shiny}
\usage{
ui_epoxy_markdown(
  .id,
  ...,
  .markdown_fn = NULL,
  .markdown_args = list(),
  .class = NULL,
  .class_item = NULL,
  .container = "div",
  .container_item = "span",
  .placeholder = "",
  .sep = "",
  .open = "{{",
  .close = "}}",
  .na = "",
  .null = "",
  .literal = FALSE,
  .trim = FALSE,
  .aria_live = c("polite", "off", "assertive"),
  .aria_atomic = TRUE
)
}
\arguments{
\item{.id}{The output id}

\item{...}{Unnamed arguments are treated as lines of markdown text, and named
arguments are treated as initial values for templated variables.}

\item{.markdown_fn}{The function used to convert the markdown to HTML. This
function is passed the markdown text as a character vector for the first
argument and any additional arguments from the list \code{.markdown_args}. By
default, we use \code{\link[pandoc:pandoc_convert]{pandoc::pandoc_convert()}} if \pkg{pandoc} is available,
otherwise we use \code{\link[commonmark:commonmark]{commonmark::markdown_html()}}.}

\item{.markdown_args}{A list of arguments to pass to
\code{\link[commonmark:commonmark]{commonmark::markdown_html()}}.}

\item{.class}{Classes added to the output div, in addition to \code{.epoxy-html}}

\item{.class_item}{Classes added to the \code{.container} wrapping each template
variable.}

\item{.container}{The name of the HTML element to be used for the output
element, by default \code{"div"}.}

\item{.container_item}{The name of the HTML element to be used for each
template item, by default \code{"span"}.}

\item{.placeholder}{Default placeholder if a template variable placeholder
isn't provided.}

\item{.sep}{[\code{character(1)}: \sQuote{""}]\cr Separator used to separate elements.}

\item{.open}{[\code{character(1)}: \sQuote{\\\{}]\cr The opening delimiter around
the template variable or expression. Doubling the full delimiter escapes
it.}

\item{.close}{[\code{character(1)}: \sQuote{\\\}}]\cr The closing delimiter
around the template variable or expression. Doubling the full delimiter
escapes it.}

\item{.na}{[\code{character(1)}: \sQuote{NA}]\cr Value to replace \code{NA} values
with. If \code{NULL} missing values are propagated, that is an \code{NA} result will
cause \code{NA} output. Otherwise the value is replaced by the value of \code{.na}.}

\item{.null}{[\code{character(1)}: \sQuote{character()}]\cr Value to replace
NULL values with. If \code{character()} whole output is \code{character()}. If
\code{NULL} all NULL values are dropped (as in \code{paste0()}). Otherwise the
value is replaced by the value of \code{.null}.}

\item{.literal}{[\code{boolean(1)}: \sQuote{FALSE}]\cr Whether to treat single or
double quotes, backticks, and comments as regular characters (vs. as
syntactic elements), when parsing the expression string. Setting \code{.literal = TRUE} probably only makes sense in combination with a custom
\code{.transformer}, as is the case with \code{glue_col()}. Regard this argument
(especially, its name) as experimental.}

\item{.trim}{[\code{logical(1)}: \sQuote{TRUE}]\cr Whether to trim the input
template with \code{\link[glue:trim]{trim()}} or not.}

\item{.aria_live, .aria_atomic}{The
\href{https://developer.mozilla.org/en-US/docs/Web/Accessibility/ARIA/Attributes/aria-live}{aria-live}
and \href{https://developer.mozilla.org/en-US/docs/Web/Accessibility/ARIA/Attributes/aria-atomic}{aria-atomic}
attribute values for the entire template region. By default, with
\code{"polite"}, any updates within the region will be announced via screen
readers.

If your template includes changes in lots of disparate areas, it would be
better to set \code{"aria-live" = "polite"} and \code{"aria-atomic" = "true"} on
specific regions that should be announced together. Otherwise, the default
is to announce the entire region within the \code{ui_epoxy_html()} whenever any
of the values within change. In other words, set \code{.aria_live = "off"} and
\code{.aria_atomic = NULL} on the \code{ui_epoxy_html()} parent item and then set
\code{"aria-live" = "polite"} and \code{"aria-atomic" = "true"} on the parent
containers of each region in the app that receives updates.
\code{ui_epoxy_html()} does targeted updates, changing only the parts of the
UI that have changed.}
}
\value{
An HTML object.
}
\description{
Create reactive HTML from a Markdown template. \code{ui_epoxy_markdown()} uses the
same template syntax as \code{\link[=ui_epoxy_html]{ui_epoxy_html()}}, but rather than requiring HTML
inputs, you can write in markdown. The template is first rendered from
markdown to HTML using \code{\link[pandoc:pandoc_convert]{pandoc::pandoc_convert()}} (if \pkg{pandoc} is
available) or \code{\link[commonmark:commonmark]{commonmark::markdown_html()}} otherwise.
}
\examples{
\dontshow{if (rlang::is_installed("shiny")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(shiny)

# Shiny epoxy template functions don't support inline transformations,
# so we still have to do some prep work ourselves.
bechdel <- epoxy::bechdel

as_dollars <- scales::label_dollar(
  scale_cut = scales::cut_short_scale()
)
bechdel$budget <- as_dollars(bechdel$budget)
bechdel$domgross <- as_dollars(bechdel$domgross)

vowels <- c("a", "e", "i", "o", "u")
bechdel$genre  <- paste(
  ifelse(substr(tolower(bechdel$genre), 1, 1) \%in\% vowels, "an", "a"),
  tolower(bechdel$genre)
)

movie_ids <- rlang::set_names(
  bechdel$imdb_id,
  bechdel$title
)

ui <- fixedPage(
  fluidRow(
    column(
      width = 3,
      selectInput("movie", "Movie", movie_ids),
      uiOutput("poster")
    ),
    column(
      width = 9,
      ui_epoxy_markdown(
        .id = "about_movie",
        "
## {{title}}

**Released:** {{ year }} \\\\
**Rated:** {{ rated }} \\\\
**IMDB Rating:** {{ imdb_rating }}

_{{ title }}_ is {{ genre }} film released in {{ year }}.
It was filmed in {{ country }} with a budget of {{ budget }}
and made {{ domgross }} at the box office.
_{{ title }}_ recieved a Bechdel rating of **{{ bechdel_rating }}**
for the following plot:

> {{ plot }}
"
      )
    )
  )
)

server <- function(input, output, session) {
  movie <- reactive({
    bechdel[bechdel$imdb_id == input$movie, ]
  })

  output$about_movie <- render_epoxy(.list = movie())
  output$poster <- renderUI(
    img(
      src = movie()$poster,
      alt = paste0("Poster for ", movie()$title),
      style = "max-height: 400px; max-width: 100\%; margin: 0 auto; display: block;"
    )
  )
}

if (interactive()) {
  shinyApp(ui, server)
}
\dontshow{\}) # examplesIf}
\dontshow{if (rlang::is_interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
run_epoxy_example_app("ui_epoxy_markdown")
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=ui_epoxy_html]{ui_epoxy_html()}}, \code{\link[=ui_epoxy_mustache]{ui_epoxy_mustache()}}, \code{\link[=render_epoxy]{render_epoxy()}}
}
